/************************************************************************************************************
* Slice32 1.0 Copyright (c) 1999
* Ziff-Davis, Inc.
* All rights reserved
* First Published in PC Magazine, US Edition, 12/01/1999
* Programmer: Paul Trout
*
* 01 Mar. 1998 - Initial work.  Structure definition for the slice file header.
*
* 28 Mar. 1998 - Decoded the DOS version file header.
*
* 31 Dec. 1998 - Added a variable to main, csize, to track the number of bytes read from the source during a 
*                  slice operation and the number of bytes written to the destination during a splice.
*              - Added an opcode variable to main.  If this is 0, we are currently slicing a file.  If it is 
*                  1, then we are splicing a file.
*              - Added code to set opcode depending on the value of argv[1].
*              - Began to define the 32 bit equivalent to dos_rst_header in slice32_header.  The major 
*                  differences are: 
*                  - NTFS supports 3 file date/time(s) Creation, Last Written and Last Accessed.  These have 
*                    to be saved, and they are in a special format, requiring the use of FILETIME structures
*                    from the Win32 API.
*                  - Attributes are a DWORD under Win32.
*                  - Filesizes are 64 bits under Win32 so instead of LO and HI order words, we have LO and HI
*                    order dwords.
*                  - Of course, Win32 allows 250 character filenames.
*                  - Win32 system date and time can be stored in a SYSTEMTIME structure.
*
* 01 Jan. 1999 - Began to add the code to perform a slice.
*              - Got the code for the slice32_header structure initialization.
*              - Added the structure for error handling.
*              - Added code to ensure the destination drive supports removable media.
*              - Added code to separate the actual source filename from any drive and directory information 
*                  that is part of the source filespec.
*              - Removed the opcode variable from main.
*              - Whereas the original slice allowed 99 disks (01-99) NTSLICE will allow 100 disks (00-99).
*                  Two char variables in main, lsd_dsk_id (least significant digit) and msd_dsk_id (most 
*                  significant digit) will track this.
*              - Created a function, get_drive_free_bytes to return the number of bytes available on the 
*                  destination media.  This is essentially a wrapper function for the Win32 function, 
*                  GetDiskFreeSpace.
*              - Renamed dos_rst_header to slice16_header.
*              - Completed the SLICE portion of the code.  It still has to be tested, but the actual grunt 
*                  work is done.
*
* 02 Jan. 1999 - Worked on debugging the slice code.  Had to change all file I/O to use WriteFile and 
*                  ReadFile instead of WriteFileEx and ReadFileEx.  The Ex versions are designed for 
*                  asynchronous I/O ONLY.
*              - It wrote about 41K more than necessary during the test.  I also need to go ahead and flush
*                  the disk buffers for the destination drive before prompting the user to insert the next 
*                  disk.
*              - Still writing more than is necessary at the end.
*              - It's not writing more at the end of the last disk.  There is a problem with filling the last
*                  less than BUFF_SIZE bytes on the previous disks.
*              - FIXED! When dsk_free-cur_xfer was less than BUFF_SIZE, the result of dsk_free-cur_xfer had to
*                  be preserved in order to correctly increment csize and cur_xfer.
*
* 03 Jan. 1999 - Changed name from ntslice to slice32.
*              - Added code in the actual disk transfer loop to account for dsk_free-cur_xfer being less than
*                  BUFF_SIZE and size-csize < dsk_free-cur_xfer.
*              - Added code to set the debug flag to 1 if the last command line argument is debug.  This will 
*                  activate some extremely verbose status messages.
*              - Added code to flush the destination drive buffers before asking the user for the next disk.
*
* 04 Jan. 1999 - Began to add the SPLICE operation code.
*
* 05 Jan. 1999 - Continued with the SPLICE code.
*              - Changed the error handling to jump to SLICE_EXIT if SLICE is the current operation, and 
*                  SPLICE_EXIT if SPLICE is the current operation.
*
* 06 Jan. 1999 - Cleaned up some comments knocked out of whack by renaming EXIT to SLICE_EXIT.
*
* 23 Jan. 1999 - Wrote the file transfer code for the SPLICE operation.  Still needs to be tested and
*                  debugged.
*              - On the first splice, did not properly skip the header of the source.  Also did not end 
*                  properly.
*              - Fixed the not ending problem and the apparent lack of skip of the header problem.
*              - SPLICE worked!  Need to add error trapping, and code to set the file date/time to same as 
*                  original.  However, core works.
*              - Added code to reset the file date and time fields to the original values.  Error trapping is 
*                  all that remains to have a fully functional 32 bit version of SLICE.  Will still have to 
*                  add the code for backwards compatibility with the 16 bit slice.
*
* 24 Jan. 1999 - Added code to build the destination path and file name for the SPLICE operation.  Due to this
*                  oversight, you could only splice to the default drive and directory.
*
* 25 Jan. 1999 - Added the SPLICE error handling code.  Did this with a separate switch/case statement and 
*                  label, SPLICE_EXIT, instead of SLICE_EXIT.  Added the label EXIT so the program has only 
*                  one exit point.
*              - In the copyright notice, changeed the version from V0R0 to V0R1.  It's time for beta testing.
*
* 26 Jan. 1999 - Began to investigate how to build some kind of equivalent to SPLICE.COM on the first 
*                  destination disk.  The first problem is how to get a copy of slice32.exe onto the target.
*                  Under Windows NT, argv[0] is exactly what was typed.  Windows 95, however, and probably 
*                  Windows 98, have an argv[0] that is a fully qualified path and filename for executable.  
*                  I think the Win32 function, SearchPath, will solve this problem.
*
* 27 Jan. 1999 - Added the code to find SLICE32.EXE, as noted in the 26 Jan. 1999 comment.  Used SearchPath.
*              - Added code to copy slice32.exe to the destination media.  Used the Win32 function, CopyFile.
*              - Added code to create splice32.bat on the destination media.  Right now it has no error 
*                  trapping, and is straight line and ugly, but it works.
*              - MAJOR ODD THING, I first named the handle for the SPLICE32.BAT file spl_bat.  After I 
*                  compiled the program (with no errors), the SearchPath call would generate an illegal 
*                  access message.  At first I thought it was stack related, but after checking, I determined
*                  the default stack to be 1 MB.  I then renamed it to spl_cmd.  Same thing happened.  
*                  Spl_batch didn't work either.  Changing the name to spl32_bat made everything work just 
*                  fine.
*
* 28 Jan. 1999 - Added code to the SPLICE operation to restore the original file attributes.
* 
* 29 Jan. 1999 - One of the changes has caused the SLICE operation to fail.  An ACCESS_DENIED error is 
*                  generated when the attempt to lock the destination with DeviceIOControl.  Fixed a bug 
*                  introduced by the splice32.bat generation, csize is used during the slice32.exe copy, 
*                  and was not reset to zero prior to the actual slicing taking place.  Fixing this, did not 
*                  fix the ACCESS_DENIED error.
*              - Commenting out the slice32.exe copy as well as the splice32.bat generation did not fix the 
*                  ACCESS_DENIED error.  However, since Win95 does not support the same IOCTL commands as NT
*                  (missed that in the first pass of the documentation), I think I'll change the slice code 
*                  to not require the buffers be flushed.  This can be done by opening the destination files 
*                  with the FILE_FLAG_NO_BUFFERING flag.
*              - Changed the slice code to open the destination files with the FILE_FLAG_NO_BUFFERING flag, 
*                  and removed the device flush code.  Didn't work worth a damn.  Using this flag requires 
*                  ALL disk i/o to be in multiples of the device's sector size, as well as having the buffers 
*                  aligned on sector size boundaries.
*              - Changing the code to use the FILE_FLAG_WRITE_THROUGH flag.  This disables lazy writing.  It 
*                  looks as if this works.  Copying slice32.c to slice32.c9.
*              - Removing the device flushing code and its associated variables.  Some time, it may be 
*                  profitable to see if the problem with locking the device was related to the problem 
*                  encountered on 27 Jan. 1999 while adding the splice32.bat code.
*
* 30 Jan. 1999 - Fixed a bug in the splice32.bat generation.  It was writing the entire destination path and 
*                  filename for the name of the first sliced file rather than just the filename.
*              - Saved the 16 bit header decode data and code out to slice32.16bit.header.  At this point, I'm
*                  not certain that slice32 needs to be backwardly compatible with original slice files.
*              - Broke the code that copies slice32.exe and creates splice32.bat on the first destination 
*                  media out to a separate function, make_splice32.  The only problem with using a batch file  
*                  to reconstruct the file is after the last disk has been read, the user is prompted for the 
*                  disk with the batch file.
*              - Added code to the SPLICE operation to check for and append, if necessary, a backslash to the 
*                  destination path.  This will permit the restoration of files to a directory other than the 
*                  default on the specified drive.
*
* 02 Feb. 1999 - Created a function, win32_ram, to return the amount of RAM in the machine.  This will be used
*                  to calc the transfer buffer size based on physical RAM, rather than a compile time constant.
*              - Changed BUFF_SIZE from a macro to the variable buff_size in main().
*              - Changed version number to V0R2.
*
* 03 Feb. 1999 - Created function prototypes for win32_ram, get_drive_free_bytes, and make_splice32.
*              - Created function size_xfer_buffer to calculate the appropriate size of the transfer buffer 
*                  based on available physical RAM and the size of the destination media.  Did not test the 
*                  function, just wrote it.
*
* 06 Feb. 1999 - Moved the buff_size initialization code into the individual operation blocks.  Also, instead 
*                  of explicitly allocating 128K to the buffer, it gets sized by size_xfer_buffer function.
*              - Had to fix the comparison amounts for the size_xfer_buffer function.  I neglected to account
*                  for the BPB, MBR, and 2 copies of the FAT.
*              - Changed version number to V0R3.
*
* 08 Feb. 1999 - During the SPLICE operation, I changed the mode of the destination file to 
*                  FILE_FLAG_WRITE_THROUGH.  This will ensure data gets flushed from the buffers before the 
*                  source media gets ejected.  In fact, I changed the mode on all CreateFile calls to 
*                  FILE_FLAG_WRITE_THROUGH.
*              - Boy do I feel stupid.  The FlushFileBuffers function clears the file buffers and writes all 
*                  information out to disk.  Replaced all occurences of FILE_FLAG_WRITE_THROUGH with 
*                  FILE_ATTRIBUTE_NORMAL, and added the appropriate calls to FlushFileBuffers.
*
* 09 Mar. 1999 - Changed the splice code to automatically find the first file of the sliced set.  This will 
*                  remove the need to pass the first file name on the command line.  Works!
*              - Removed code in make_splice32 that created the splice32.bat file on the destination drive.
*              - Added code to the splice operation to prompt the user for the first disk.
*              - Changed version number to V0R4.
*              - Need to change error handling because of the inserted operations in the splice code.
*
* 13 Mar. 1999 - Worked on the comments about the recent FindFirst File calls.
*
* 09 May. 1999 - Began work on the modifications for turning this into a PC Magazine Utility column utility 
*                  and accompanying article.
*              - The first thing to change is the copyright and title notice.  In order to properly incorporate
*                  the changes, I need to separate the usage info from the copyright notice.
*              - Next is to work on integrating split functionality while keeping the integrity of the system.
*              - Split the function protoypes, macro definitions, and structure definitions into slice32.h
*              - Added a new variable to main called opcode.  It is an unsigned long integer.  If it's value 
*                  is 0 then a slice will be performed.  If it's value is 1 then a splice will be performed.  
*                  If it's value is 2 then a split will be performed.
*              - Modified the make_splice32 function so splice32.exe is the destination filename.  This will 
*                  make the argv[1] command redundant in some special cases.
*              - Split the size_xfer_buffer, win32_ram, get_drive_free_bytes, and make_splice32 functions 
*                  into slc32lib.c.  Eventually, these will be stuck into a function library.
*              - Pulled SLICE operation out into a separate function slice.
*              - In order for the slice function to work, I had to make debug a global variable in slice32.h.
*              - Pulled SPLICE operation out into a separate function splice.
*
* 12 May. 1999 - Changes made on 09 May. 1999 appear to be working just fine.
*              - Created a split function to handle splitting a single large file into smaller files in the 
*                  same location as the original.  They get reassembled with a binary copy operation rather 
*                  than a splice.  To keep it simple, main passes split() argv[2-4] as character strings so
*                  that split can do its processing with as little modification as possible.  I will need to
*                  do a massive renaming to bring the variable names into line with the other functions, as 
*                  well as adding similar error processing code.
*              - Added the error processing code to split.
*              - Changed split to use win32_ram instead of directly calling GlobalMemoryStatus.
*              - Split code appears to be working just fine.
*
* 13 May. 1999 - Began to add code to cause the opcode to be set based on the value of argv[0].  This will 
*                  require some finagling because under DOS/Win95 and presumably Win98, argv[0] is a fully
*                  qualified path and filename of the executable, while under NT, argv[0] is simply the 
*                  command.
*              - Killed opcode in main (yet again).  I should probably finalize the decision on this one.
*              - Using renamed slice32.exe to splice32.exe and split32.exe works appears to be working just 
*                  fine on both NT and 95.
*              - Moved the global declaration of debug out of slice32.h, and into slice32.c.  I needed to do
*                  this so I could turn slc32lib.c into slc32lib.lib.
*              - Turned slc32lib.c into a linkable library, and modified the LINK statement to reflect this.
*                  Creating the library allowed me to eliminate the #include "slc32lib.c" statement.
*              - Since this is now a multi-file project, created a MAKEFILE to control builds.
*
* 02 Jul. 1999 - Replaced all calls to _getche with _getch.  This fixes a boneheaded mistake on my part that 
*                  confused _getch (no echo of character) with _getche (echo of character).  Of course I run
*                  into the famous VC++ bug that generates bad code for the IF statement after the prompt in
*                  the slice function.  The splice function works just beautifully.
*              _ _getch is causing the SearchPath call in make_splice32 to bomb.  With _getche, all is OK.
*                  For the time being, slice() uses _getche.
*
* 17 Jul. 1999 - Now using VC++ 6.0 Professional.  Trying the _getch experiment one more time.  Still bombed 
*                  out.
*              - getchar doesn't work any better.  In fact, because it is a stream I/O function, it's worse.
*              - It appears to be some problem with an overflow condition.  keystroke is defined before 
*                  dsk_num, and therein lies the problem.
*              - Changed from straight _getch to a loop based on the return value of _kbhit.
*              - The flag printf statements arounf the make_splice32 call are still acting wierd.  Especially
*                  the one after it returns.  Splice has also stopped working.  This is not a good thing.
*              - Restored the calls to _getche, and the problems are still present.
*
* 18 Jul. 1999 - Must remember splice has two sections where the user is prompted to strike any key.
*              - Fixed the problem with the SearchPath call in make_splice32.  The exec_name variable needs 
*                  to be passed by reference, not defined as a pointer to a pointer.  The slice function is 
*                  now working properly with _kbhit, _getch, etc.
*              - Replaced the _getch call in splice with the _kbhit, _getch combination from slice.  No dice.
*                  Splice still bombs after the first disk.  It never asks for the next disk.
*              - The problem with splice is that slice is setting rstlast to one for all of the disks.  Fixed 
*                  this by initializing fhead32.rstlast to 0 at the start of the slice operation.
*              - Everything WORKS!! and with no keystroke echoing.
*
* 13 Aug. 1999 - Removed a debugging printf.  Still works.
*              - Rather than spin the disk I/O code off into thread ready functions, check into changing the 
*                  Process priority instead.  Kick it down for the reads and writes, and back up for everything
*                  else.
*
* 03 Sep. 1999 - Removed the call to size_xfer_buffer.  Fixed transfer buffer at 64K.  For explanation, see
*                  slice32.c and slc32lib.c.
*
* 05 Sep. 1999 - Removed the comments about SLICE and SPLIT errorlevels.
*              - Added code to main to use %TEMP% as the destination if one is not specified on the command 
*                  line.
*              - Added code to splice to print a file restored message when the splice has successfully 
*                  completed.
*              - Added a default case to the SPLICE_EXIT switch statement.  This will eliminate the compile
*                  error indicating that not all of the splice function control paths return a value.
*
* 07 Sep. 1999 - Added code to allow splice to operate on files in the same directory as it is.  This is 
*                  necessary if splice is to work for files "sliced into a specified number of files" or 
*                  files "sliced into portions of a specified size".
*              - It works as long as I pass . as argv[1].  Otherwise the usage information kicks in.  Fixed 
*                  this by moving the usage information into the SPLICE_EXIT error trap for no source file 
*                  found.
*              - Removed EXIT label from main.
*
* 12 Sep. 1999 - Changed the code that determines if the source is removable.  GetDriveType works on relative 
*                  paths.
*              - Got rid of debug variable.
*              - Got rid of code in main that set debug.
*              - Got rid of all debug based printf statements.
*              - Changed code in main to pass the current working directory as the source if none is spec'd on
*                  the command line, instead of .\.  This makes the prompts for removable media more readable.
*
* 18 Sep. 1999 - In keeping with the other source files, widened this to 110 characters, and reformatted to
*                  that width.
*
* 20 Sep. 1999 - If there is an error opening the current source file, and the source drive is removable, the
*                  use is reprompted for the disk.  If the source drive is not removable, then an error 
*                  message is displayed, and the program terminates.  Still works.
*
* 21 Sep. 1999 - When splicing, if the destination file already exists, added code to prompt to the user
*                  to overwrite it instead of simply terminating.  If the user hits 'Y' or 'y', then the
*                  destination is overwritten.  Otherwise it's left alone, and the application terminates.
*
* 25 Sep. 1999 - Added a variable that will be 1 larger than dsk_num.  It is this new number that will be
*                  used in all user messages.
*              - When I changed the user seen disk numbering to be based on one, I didn't change all places
*                  where the old 0 based disk number was being displayed.  This has now been corrected.
*
* 28 Sep. 1999 - Modified the code, in main, that selects the destination directory if none is passed on the
*                  command line.  If the current directory is on removable media, then the file gets restored
*                  to the system temp directory.  Otherwise, it gets restored to the current directory.
*              - Removed the code in main left over from the days when splice32.exe was a renamed slice32.exe
*                  or split32.exe.
*
* 05 Oct. 1999 - Source editing.  Comment reformatting, etc.
*
* 28 Oct. 1999 - Changed the usage information to explain that the command line parameters, if passed, 
*                  specify folders, not files.
*
* 31 Oct. 1999 - Changed the message for error 253 - can't find first file of a slice set.  It now says
*                  "Could not find the first file of a slice set in [src]" and prints out the specified
*                  source rather than [src].
*              - Changed the message printed when the current source file cannot be opened.  Instead of
*                  printing "Error opening file on disk #X = XX", it now says "The file, [src], could not be
*                  opened".  If splicing from removable media, then the user is prompted for the correct disk.
*                  Otherwise, the program terminates.
*              - Added code to splice function to determine if the specified src is a file or a directory,
*                  and, if it's a file, to determine if it's the first file of a slice set, or the ultimate
*                  destination filename.
*
* 01 Nov. 1999 - Added message to the usage message decribing the new option for passing either the first 
*                  file of a slice set, or the ultimate destination name of a slice set as argv[1].
*              - Passing a ? as argv[1] causes an invalid page fault.  To correct this, I pulled the usage
*                  statements into a slice32_usage function, and added code in main to trap ? or * in either
*                  argv[1] or argv[2].  ERRORLEVEL is 1 if this occurs.
*              - Added code to tell the user which disk they actually inserted when s/he insert the wrong
*                  disk in the sequence.
*
* ERRORLEVELS: SPLICE OPERATION
*               254 = Error allocating buffer for the source file path and name.
*               253 = Error finding first file of sliced source.
*               252 = Error reading header from first file of the sliced source.
*               251 = Error closing first file of the sliced source after reading the header.
*               250 = Error allocating buffer for the destination file path and name.
*               249 = Error opening the destination file.
*               248 = Error allocating transfer buffer.
*               247 = Error opening current source file.
*               246 = Error reading header from one of the sliced source files.
*               245 = Error setting the date/time(s) on the destination file.
*               244 = Error closing the destination file.
*               243 = Error restoring file attributes.
*                 1 = Wildcard character (? or *) passed as an argument
*
* Compile: cl /nologo /W3 /D "WIN32" /D "CONSOLE"  /Zp1 /c splice32.c
* Link   : link /NOLOGO /MACHINE:IX86 /SUBSYSTEM:CONSOLE /OUT:splice32.exe splice32.obj kernel32.lib 
*            slc32lib.lib
* OR
* nmake - to build whole shebang!
************************************************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>                                              /* _getch() */

#include <windows.h>
#include <winnt.h>
#include <winbase.h>                                            /* Includes SearchPath function */

#include "slice32.h"                                            /* Slice32 specific */

void splice32_usage(void) {                                     /* Display slice usage */
  printf("\nUSAGE: SPLICE32 [source folder] [destination folder]\n");
  printf("SPLICE32.EXE will be on the first disk of a slice set\n");
  printf("If a destination folder is not specified, TEMP will be used.\n");
  printf("If a source folder is not specified, will try to find first file in current directory.\n");
  printf("\nOR\n");
  printf("\nUSAGE: SPLICE32 name_of_first_file_of_slice_set [destination folder]\n");
  printf("The last two characters of the first file of a slice set will be 00, i.e. BIG.D00.\n");
  printf("If a destination folder is not specified, TEMP will be used.\n");
  printf("\nOR\n");
  printf("\nUSAGE: SPLICE32 destination_filename [destination folder]\n");
  printf("Destination name is original name of the sliced file, i.e. BIG.DAT.\n");
  printf("If a destination folder is not specified, TEMP will be used.\n");
  printf("\nNeither the source nor the destination may contain wildcards\n");
}

unsigned long int splice(char *src, char *dst) {
  unsigned char        *src_file;                               /* Bare source filename */
  unsigned char        *dst_file;                               /* Destination filename */
  unsigned char        *xfer_buff;                              /* Buffer for file I/O */
  unsigned char        *act_file;                               /* Actual file present on disk */
  unsigned char         lsd_dsk_id;                             /* Disk number lo digit */
  unsigned char         msd_dsk_id;                             /* Disk number hi digit */
  int                   keystroke;                              /* Used in loops */
  int                   pri_org;                                /* Original thread priority */
  unsigned long int     dsk_num=0;                              /* Current disk number 0-99 */
  unsigned long int     usr_num=1;                              /* User disk number 1-100 */
  unsigned long int     act_num=0;                              /* Actual file number on disk */
  unsigned long int     size;                                   /* Source file original size */
  unsigned long int     csize=0;                                /* Current output file size */
  unsigned long int     bytes_rw;                               /* Bytes read or written */
  unsigned long int     errorlevel=0;                           /* ERRORLEVEL return code */
  unsigned long int     win32_err=0;                            /* GetLastError result */
  unsigned long int     buff_size;                              /* Transfer buffer size */
  unsigned long int     src_attrs;                              /* Src attributes */
  struct slice32_header fhead32;                                /* 32 bit file header */
  BOOL                  src_rmv;                                /* TRUE=src removable, FALSE=isn't */
  BOOL                  src_is_file;                            /* TRUE=src is file, FALSE=src is folder */
  BOOL                  src_file_is_dest;                       /* TRUE=src file is the dest filename */
  BOOL                  tfrtc;                                  /* Win32 return code */
  HANDLE                sl_src;                                 /* Source file */
  HANDLE                sl_dst;                                 /* Destination file */
  HANDLE                tgt_hnd;                                /* Search handle used in splice */
  HANDLE                cur_thrd;                               /* Handle to current thread */
  WIN32_FIND_DATA       tgt_file;                               /* Splice target file */

  /**********************************************************************************************************
  * Get handle to current thread.
  **********************************************************************************************************/
  cur_thrd=GetCurrentThread();

  /**********************************************************************************************************
  * Call GetDriveType to determine if source drive is removable.
  **********************************************************************************************************/
  src_rmv=TRUE;                                                 /* Begin assuming it's removable */
  if(GetDriveType(src)!=DRIVE_REMOVABLE)
    src_rmv=FALSE;

  /**********************************************************************************************************
  * Decide if the specified src is a file or a folder.
  **********************************************************************************************************/
  src_is_file=FALSE;                                            /* Begin assuming it's a folder */
  src_attrs=GetFileAttributes(src);                             /* Get the attributes */
  if(src_attrs!=0xFFFFFFFF) {                                   /* If the function succeeds */
    if(!(src_attrs&FILE_ATTRIBUTE_DIRECTORY)) {                 /* If it's not a directory */
      src_is_file=TRUE;                                         /* Then it's a file */
    }
  }
  else {                                                        /* If the function fails */
    src_is_file=TRUE;                                           /* For right now, assume src is a file */
  }

  /**********************************************************************************************************
  * If the src is a file, decide if it's the ultimate destination filename, or the first file of a slice set.
  **********************************************************************************************************/
  src_file_is_dest=FALSE;                                       /* Assume it's 1st file of slice set */
  if(src[strlen(src)-1]!='0' && src[strlen(src)-2]!='0') {      /* If last two characters are not 00 */
    src_file_is_dest=TRUE;                                      /* Then it's the ultimate destination name */
  }

  /**********************************************************************************************************
  * Prompt the user for the first disk of the slice set, as long as the source is removable.
  **********************************************************************************************************/
  if(src_rmv==TRUE) {
    printf("Please insert disk #1 in drive %.2s. ",src);
    printf("Press any key when ready.\n");
    while(!_kbhit()) ;
    keystroke=_getch();                                         /* Eat the character */
    if(keystroke==0x00||keystroke==0xE0)
      keystroke=_getch();
  }

  if(src_is_file==FALSE) {                                      /* If the source is a folder */
    /********************************************************************************************************
    * Build the source file search path.  If the drive is removable, this is composed of the source drive, 
    *   src, and a filespec of *00.  If the drive isn't removable, this is composed of the source drive, a
    *   \ (if necessary), and a filespec of *00.
    ********************************************************************************************************/
    src_file=(unsigned char *)calloc(strlen(src)+5,sizeof(unsigned char));
    if(src_file==NULL) {                                        /* If there was an error */
      win32_err=1;                                              /* Trigger error message */
      errorlevel=254;                                           /* Error allocating src path */
      goto SPLICE_EXIT;                                         /* Terminate w/errorlevel */
    }
    strcpy(src_file,src);                                       /* Copy the spec'd source */
    if(src_rmv==FALSE) {                                        /* If source isn't removable */
      if(src[strlen(src)-1]!='\\' && src[strlen(src)-1]!=':')   /* Append \ if req'd */
        strcat(src_file,"\\");
    }
    strcat(src_file,"*00");                                     /* Copy search file spec */

    /********************************************************************************************************
    * Find the first file of the slice set.
    ********************************************************************************************************/
    tgt_hnd=FindFirstFile(src_file,&tgt_file);
    if(tgt_hnd==INVALID_HANDLE_VALUE) {
      win32_err=GetLastError();
      errorlevel=253;
      goto SPLICE_EXIT;
    }
    
    /********************************************************************************************************
    * Concatenate the filename onto the existing source path.  For this to work, free the src_file buffer, 
    *   and reallocate it so there is no buffer overflow because the filename is longer than the wildcard 
    *   spec.  Same rules for src being removable or not as above.
    ********************************************************************************************************/
    free(src_file);
    src_file=(unsigned char *)calloc(strlen(src)+strlen(tgt_file.cFileName)+2,sizeof(unsigned char));
    strcpy(src_file,src);                                       /* Copy the spec'd source */
    if(src_rmv==FALSE) {                                        /* If source isn't removable */
      if(src[strlen(src)-1]!='\\' && src[strlen(src)-1]!=':')   /* Append \ if req'd */
        strcat(src_file,"\\");
    }
    strcat(src_file,tgt_file.cFileName);                        /* Copy filename */

    /********************************************************************************************************
    * Close the search handle.
    ********************************************************************************************************/
    tfrtc=FindClose(tgt_hnd);
  }
  else {                                                        /* If the source is a filename */
    /********************************************************************************************************
    * Build the source file name.  This is initially composed of the specified source.
    ********************************************************************************************************/
    src_file=(unsigned char *)calloc(strlen(src)+1,sizeof(unsigned char));
    if(src_file==NULL) {                                        /* If there was an error */
      win32_err=1;                                              /* Trigger error message */
      errorlevel=254;                                           /* Error allocating src path */
      goto SPLICE_EXIT;                                         /* Terminate w/errorlevel */
    }
    strcpy(src_file,src);                                       /* Copy the source */
    if(src_file_is_dest) {                                      /* If source is an ultimate destination */
      src_file[strlen(src_file)-1]='0';                         /* Set last character to 0 */
      src_file[strlen(src_file)-2]='0';                         /* Set next to last character to 0 */
    }
    src_attrs=GetFileAttributes(src_file);                      /* Retrieve source file attributes */
    if(src_attrs==0xFFFFFFFF) {                                 /* If that fails */
      if(GetLastError()==ERROR_FILE_NOT_FOUND) {                /* Because the file was not found */
        if(src_file_is_dest) {                                  /* If spec'd src was a destination */
          printf("There is no slice set corresponding to %s in %s.\n",src,dst);
          win32_err=0;                                          /* No win32_err information prints */
          errorlevel=253;                                       /* Error opening file */
          goto SPLICE_EXIT;                                     /* Terminate w/errorlevel */
        }
        else {                                                  /* If spec'd source was slice set #1 */
          printf("In %s, %s is not a valid starting file for Splice32.\n",dst,src);
          win32_err=0;                                          /* No win32_err information prints */
          errorlevel=253;                                       /* Error opening file */
          goto SPLICE_EXIT;                                     /* Terminate w/errorlevel */
        }
      }                                                         /* End file not found check */
    }                                                           /* End source file attributes check */
  }                                                             /* End source is filename processing */

  /**********************************************************************************************************
  * Open the 1st file of the source, read the header, and close the file.
  **********************************************************************************************************/
  sl_src=CreateFile(src_file,GENERIC_READ,0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
  if(sl_src==INVALID_HANDLE_VALUE) {                            /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error */
    errorlevel=253;                                             /* Error opening file */
    goto SPLICE_EXIT;                                           /* Terminate w/errorlevel */
  }

  tfrtc=ReadFile(sl_src,&fhead32,sizeof(fhead32),&bytes_rw,NULL);
  if(tfrtc==FALSE) {                                            /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error */
    errorlevel=252;                                             /* Error reading header */
    goto SPLICE_EXIT;                                           /* Terminate w/errorlevel */
  }

  tfrtc=CloseHandle(sl_src);
  if(tfrtc==FALSE) {                                            /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error */
    errorlevel=251;                                             /* Error closing source */
    goto SPLICE_EXIT;                                           /* Terminate w/errorlevel */
  }

  /**********************************************************************************************************
  * Build the destination file name.  this is composed of the destination path, argv[3], possibly followed by
  *   a \ and the filename from the fhead32 structure at the beginning of each slice file.
  **********************************************************************************************************/
  dst_file=(unsigned char *)calloc(strlen(dst)+strlen(fhead32.rstname)+2,sizeof(unsigned char));
  if(dst_file==NULL) {                                          /* If error */
    win32_err=1;                                                /* Trigger error message */
    errorlevel=250;                                             /* Error allocating dst path */
    goto SPLICE_EXIT;                                           /* Terminate w/errorlevel */
  }

  strcpy(dst_file,dst);
  if(dst[strlen(dst)-1]!=':' && dst[strlen(dst)-1]!='\\') {
    strcat(dst_file,"\\");
  }
  strcat(dst_file,fhead32.rstname);

  /**********************************************************************************************************
  * Open the destination file.  If it exists, prompt the user to overwrite it.  If the user responds with
  *   'Y' or 'y' then overwrite, otherwise terminate the application.
  **********************************************************************************************************/
  sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
  if(sl_dst==INVALID_HANDLE_VALUE) {                            /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error */
    if(win32_err==ERROR_FILE_EXISTS) {                          /* If the destination exists */
      printf("%s already exists.  Overwrite? (Y/N): ",dst_file);
      while(!_kbhit()) ;
      keystroke=_getch();                                       /* Eat the character */
      if(keystroke==0x00||keystroke==0xE0) {
        keystroke=_getch();
        printf("\n");
      }
      else if(keystroke=='Y'||keystroke=='y') {
        tfrtc=DeleteFile(dst_file);
        sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
        win32_err=0;
        printf("\n");
      }
      else {
        printf("\n");
        errorlevel=249;                                         /* Error opening dest file */
        goto SPLICE_EXIT;                                       /* Terminate w/errorlevel */
      }
    }
    else {                                                      /* If error, not because destination existed */
      errorlevel=249;                                           /* Error opening dest file */
      goto SPLICE_EXIT;                                         /* Terminate w/errorlevel */
    }
  }                                                             /* End INVALID_HANDLE_VALUE processing */

  size=0;                                                       /* Nothing has been restored */
  dsk_num=0;                                                    /* Start with disk number 0 */

  /**********************************************************************************************************
  * Allocate the transfer buffer.
  **********************************************************************************************************/
  buff_size=65536;
  xfer_buff=(unsigned char *)calloc(buff_size,sizeof(unsigned char));
  if(xfer_buff==NULL) {                                         /* If error */
    win32_err=1;                                                /* Trigger error message */
    errorlevel=248;                                             /* Error allocating xfer buffer */
    goto SPLICE_EXIT;                                           /* Terminate w/errorlevel */
  }

  while(size<fhead32.rstlsiz) {                                 /* While the file is not complete */
    sl_src=CreateFile(src_file,GENERIC_READ,0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
    /********************************************************************************************************
    * If the current source file cannot be opened, and the source drive is removable, figure out which disk
    *   was inserted by replacing the last two characters of the source filename with ?? and issuing a find
    *   first file call.  If no file is found, display a message telling the user there is no slice file on
    *   the disk.  If a file is found, convert the last two characters to a number, and tell the user "You
    *   need to insert disk #XX.  You inserted disk #YY",and then reprompt the user for the disk.  If the 
    *   current source file cannot be opened, and the source drive is not removable, the display an error 
    *   message, and exit.
    ********************************************************************************************************/
    while(sl_src==INVALID_HANDLE_VALUE) {                       /* If error */
      win32_err=GetLastError();                                 /* Store Win32 error */
      if(src_rmv==TRUE) {                                       /* If drive is removable */
        act_file=(unsigned char *)calloc(strlen(src_file)+1,sizeof(unsigned char));
        strcpy(act_file,src_file);                              /* Copy the source file */
        act_file[strlen(act_file)-1]='?';                       /* Replace the last two characters with */
        act_file[strlen(act_file)-2]='?';                       /* ?? for file search */
        tgt_hnd=FindFirstFile(act_file,&tgt_file);              /* Issue the FindFirstFile call */
        if(tgt_hnd==INVALID_HANDLE_VALUE) {                     /* If no file found, tell the user */
          printf("\nThe current disk in %s does not have a slice file on it.\n",src);
        }
        else {                                                  /* If a file was found */
          act_num=(tgt_file.cFileName[strlen(tgt_file.cFileName)-2]-48)*10;
          act_num+=(tgt_file.cFileName[strlen(tgt_file.cFileName)-1]-48);
          act_num++;                                            /* Convert last two characters to number */
          printf("\nYou need to insert Disk #%d.  You inserted Disk #%d.\n",\
                 usr_num,act_num);
          tfrtc=CloseHandle(tgt_hnd);                           /* Close the find handle */
        }
        free(act_file);                                         /* Free the source file copy */
        printf("Please insert disk #%d in drive %.2s. ",usr_num,src);
        printf("Press any key when ready.\n");
        while(!_kbhit()) ;
        keystroke=_getch();                                     /* Eat the character */
        if(keystroke==0x00||keystroke==0xE0)
          keystroke=_getch();
        sl_src=CreateFile(src_file,GENERIC_READ,0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
      }
      else {
        errorlevel=247;
        goto SPLICE_EXIT;
      }
    }
    win32_err=0;                                                /* If we got this far, it can be reset to 0 */

    tfrtc=ReadFile(sl_src,&fhead32,sizeof(fhead32),&bytes_rw,NULL);
    if(tfrtc==FALSE) {                                          /* If error */
      win32_err=GetLastError();                                 /* Store Win32 error */
      errorlevel=246;                                           /* Error reading header */
      goto SPLICE_EXIT;                                         /* Terminate w/errorlevel */
    }

    csize=(GetFileSize(sl_src,NULL))-sizeof(fhead32);

    /********************************************************************************************************
    * Kick the thread priority down prior to disk I/O.
    ********************************************************************************************************/
    pri_org=GetThreadPriority(cur_thrd);
    tfrtc=SetThreadPriority(cur_thrd,THREAD_PRIORITY_LOWEST);

    /********************************************************************************************************
    * Copy the current file buff_size bytes at a time.
    ********************************************************************************************************/
    while(csize>0) {                                            /* As long as there is */
      if(csize>=buff_size) {                                    /* data, transfer it */
        tfrtc=ReadFile(sl_src,xfer_buff,buff_size,&bytes_rw,NULL);
        tfrtc=WriteFile(sl_dst,xfer_buff,buff_size,&bytes_rw,NULL);
        size+=buff_size;
        csize-=buff_size;
      }
      if(csize<buff_size) {
        tfrtc=ReadFile(sl_src,xfer_buff,csize,&bytes_rw,NULL);
        tfrtc=WriteFile(sl_dst,xfer_buff,csize,&bytes_rw,NULL);
        size+=csize;
        csize-=csize;
      }
      memset(xfer_buff,'\0',buff_size);
    }
    /********************************************************************************************************
    * Flush the buffers on the destination file before closing the source file to prevent data loss due to 
    *   incomplete reads from the source when it's ejected.
    ********************************************************************************************************/
    tfrtc=FlushFileBuffers(sl_dst);
    tfrtc=CloseHandle(sl_src);
    dsk_num++;
    usr_num++;

    /********************************************************************************************************
    * Reset the thread priority
    ********************************************************************************************************/
    tfrtc=SetThreadPriority(cur_thrd,pri_org);

    /********************************************************************************************************
    * Initialize lsd_dsk_id and msd_dsk_id and write them to the last two characters of the source filespec.
    ********************************************************************************************************/
    if(dsk_num<10) {                                            /* If disk number is 0-9 */
      msd_dsk_id='0';                                           /* MSD is 0 */
      lsd_dsk_id=(char)(dsk_num+48);                            /* Convert LSD */
    }
    else {                                                      /* Otherwise */
      msd_dsk_id=(char)((dsk_num/10)+48);                       /* Convert MSD */
      lsd_dsk_id=(char)((dsk_num%10)+48);                       /* Convert LSD */
    }
    src_file[strlen(src_file)-1]=lsd_dsk_id;
    src_file[strlen(src_file)-2]=msd_dsk_id;

    /********************************************************************************************************
    * Prompt user for next disk, unless it's the last one, and as long as the source is removable.
    ********************************************************************************************************/
    if(fhead32.rstlast!=1) {
      if(src_rmv==TRUE) {
        printf("Please insert disk #%d in drive %.2s. ",usr_num,src);
        printf("Press any key when ready.\n");
        while(!_kbhit()) ;
        keystroke=_getch();                                     /* Eat the character */
        if(keystroke==0x00||keystroke==0xE0)
          keystroke=_getch();
      }
    }
    /********************************************************************************************************
    * If it was the last one, then set the file date/time fields to the original values.  Close the 
    *   destination file, and finally, restore the file attributes.
    ********************************************************************************************************/
    else {
      tfrtc=SetFileTime(sl_dst,&(fhead32.creat_time),&(fhead32.acces_time),&(fhead32.write_time));
      if(tfrtc==FALSE) {                                        /* If error */
        win32_err=GetLastError();                               /* Store Win32 error */
        errorlevel=245;                                         /* Error setting dates */
        goto SPLICE_EXIT;                                       /* Terminate w/errorlevel */
      }
      tfrtc=CloseHandle(sl_dst);
      if(tfrtc==FALSE) {                                        /* If error */
        win32_err=GetLastError();                               /* Store Win32 error */
        errorlevel=244;                                         /* Error closing dest */
        goto SPLICE_EXIT;                                       /* Terminate w/errorlevel */
      }
      tfrtc=SetFileAttributes(dst_file,fhead32.rstattr);
      if(tfrtc==FALSE) {                                        /* If error */
        win32_err=GetLastError();                               /* Store Win32 error */
        errorlevel=243;                                         /* Error setting attributes */
        goto SPLICE_EXIT;                                       /* Terminate w/errorlevel */
      }
    }
  }                                                             /* Finish with this disk */

  SPLICE_EXIT:                                                  /* Begin SPLICE cleanup code */
  switch(errorlevel) {
    case   0: printf("%s Successfully restored to %s\n",fhead32.rstname,dst);
    case 243: if(win32_err!=0) {
                printf("Error setting attributes on %s = %ld\n",dst_file,win32_err);
                win32_err=0;
              }
    case 244: if(win32_err!=0) {
                printf("Error closing %s = %ld\n",dst_file,win32_err);
                win32_err=0;
              }
    case 245: if(win32_err!=0) {
                printf("Error setting date/time on %s = %ld\n",dst_file,win32_err);
                tfrtc=CloseHandle(sl_dst);
                win32_err=0;
              }
    case 246: if(win32_err!=0) {
                printf("Error reading header from %s = %ld\n",src_file,win32_err);
                tfrtc=CloseHandle(sl_src);
                win32_err=0;
              }
    case 247: if(win32_err!=0) {
                printf("The file, %s, could not be opened.\n",src_file);
                win32_err=0;
              }
              free(xfer_buff);                                  /* Free transfer buffer */
    case 248: if(win32_err!=0) {
                printf("Error allocating transfer buffer\n");
                tfrtc=CloseHandle(sl_dst);
                win32_err=0;
              }
    case 249: if(win32_err!=0) {
                printf("Error opening %s = %ld\n",dst_file,win32_err);
                win32_err=0;
              }
              free(dst_file);                                   /* Free destination filename */
    case 250: if(win32_err!=0) {
                printf("Error allocating dest file path buffer\n");
                win32_err=0;
              }
    case 251: if(win32_err!=0) {
                printf("Error closing %s = %ld\n",src_file,win32_err);
                win32_err=0;
              }
    case 252: if(win32_err!=0) {
                printf("Error reading header from %s = %ld\n",src_file,win32_err);
                tfrtc=CloseHandle(sl_src);
                win32_err=0;
              }
    case 253: if(win32_err!=0) {
                printf("Could not find the first file of a slice set in %s.\n",src);
                splice32_usage();                               /* Display usage */
                win32_err=0;
              }
              free(src_file);                                   /* Free source filename */
    case 254: if(win32_err!=0) {
                printf("Error allocating source file path buffer\n");
                win32_err=0;
              }
    default : return(errorlevel);                               /* Display program info and exit */
  }
}

void main(int argc, char *argv[]) {
  char              *dest;                                      /* Destination */
  char              *src;                                       /* Source */
  char              *cur_dir;                                   /* Current directory */
  char              *wcard_finder;                              /* Locate wildcards in argv[1] or argv[2] */
  unsigned long int  errorlevel=0;                              /* ERRORLEVEL return code */
  unsigned long int  rtc;
  BOOL               dst_rmv=FALSE;                             /* FALSE=current directory not removable */

  /**********************************************************************************************************
  * Print Ziff-Davis and PC Magazine copyright notice.  To make sure the copyright displays OK regardless of
  *   the value of true_cmd, true_cmd is passed to the printf statement.
  **********************************************************************************************************/
  printf("\nSPLICE32 1.0 Copyright (c) 1999\n");
  printf("Ziff-Davis, Inc.\n");
  printf("All rights reserved.\n");
  printf("First Published in PC Magazine, US Edition, 12/01/1999.\n");
  printf("Programmer: Paul Trout\n");

  if(argc>=2) {                                                 /* Check for wildcards in argv[1] */
    wcard_finder=strchr(argv[1],'?');                           /* Check for ? first */
    if(wcard_finder==NULL) {                                    /* If no ? is found */
      wcard_finder=strchr(argv[1],'*');                         /* Check for * next */
      if(wcard_finder==NULL)                                    /* If no * is found */
        ;                                                       /* Do nothing */
      else {                                                    /* If * was found */
       splice32_usage();                                        /* Display usage */
       exit(1);                                                 /* And exit */
      }
    }
    else {                                                      /* If ? was found */
      splice32_usage();                                         /* Display usage */
      exit(1);                                                  /* And exit */
    }
  }

  if(argc>=3) {                                                 /* Check for wildcards in argv[2] */
    wcard_finder=strchr(argv[2],'?');                           /* Check for ? first */
    if(wcard_finder==NULL) {                                    /* If no ? is found */
      wcard_finder=strchr(argv[2],'*');                         /* Check for * next */
      if(wcard_finder==NULL)                                    /* If no * is found */
        ;                                                       /* Do nothing */
      else {                                                    /* If * was found */
       splice32_usage();                                        /* Display usage */
       exit(1);                                                 /* And exit */
      }
    }
    else {                                                      /* If ? was found */
      splice32_usage();                                         /* Display usage */
      exit(1);                                                  /* And exit */
    }
  }

  /**********************************************************************************************************
  * If no destination is specified, the retrieve the current directory.  If the current directory is
  *   removable, then file gets restored to %TEMP%.  Otherwise, it gets restored to the current directory.
  **********************************************************************************************************/
  if(argc<=2) {
    cur_dir=(char *)calloc(MAX_PATH+1,sizeof(char));
    rtc=GetCurrentDirectory(MAX_PATH,cur_dir);
    if(GetDriveType(cur_dir)==DRIVE_REMOVABLE) {
      dest=getenv("TEMP");
      dst_rmv=TRUE;
    }
    else {
      dest=cur_dir;
      dst_rmv=FALSE;
    }
  }
  else {
    dest=calloc(strlen(argv[2])+1,sizeof(char));                          
    strcpy(dest,argv[2]);
  }

  /********************************************************************************************************
  * If source is spec'd on command line, then the destination has either been spec'd, or it will default 
  *   as described above.
  ********************************************************************************************************/
  if(argc>=2) {                      
    errorlevel=splice(argv[1],dest);                            /* Splice the file */
    if(argc>2)                                                  /* If destination spec'd */
      free(dest);                                               /* Free the destination */
    else {                                                      /* Otherwise destination defaulted */
      free(cur_dir);                                            /* And cur_dir needs to be freed */
    }
  }
  /********************************************************************************************************
  * Otherwise, source is not spec'd and neither is destination(argc=1), then pass current working directory
  *   as source and destination will be %TEMP%.
  ********************************************************************************************************/
  else {
    src=(char *)calloc(MAX_PATH+1,sizeof(char));
    rtc=GetCurrentDirectory(MAX_PATH,src);
    errorlevel=splice(src,dest);                                /* Splice the file */
    free(src);
  }

  exit(errorlevel);
}

