/************************************************************************************************************
* Slice32 1.0 Copyright (c) 1999
* Ziff-Davis, Inc.
* All rights reserved
* First Published in PC Magazine, US Edition, 12/01/1999
* Programmer: Paul Trout
*
* 01 Mar. 1998 - Initial work.  Structure definition for the slice file header.
*
* 28 Mar. 1998 - Decoded the DOS version file header.
*
* 31 Dec. 1998 - Added a variable to main, csize, to track the number of bytes read from the source during a 
*                  slice operation and the number of bytes written to the destination during a splice.
*              - Added an opcode variable to main.  If this is 0, we are currently slicing a file.  If it is 
*                  1, then we are splicing a file.
*              - Added code to set opcode depending on the value of argv[1].
*              - Began to define the 32 bit equivalent to dos_rst_header in slice32_header.  The major 
*                  differences are: 
*                  - NTFS supports 3 file date/time(s) Creation, Last Written and Last Accessed.  These have 
*                    to be saved, and they are in a special format, requiring the use of FILETIME structures
*                    from the Win32 API.
*                  - Attributes are a DWORD under Win32.
*                  - Filesizes are 64 bits under Win32 so instead of LO and HI order words, we have LO and HI
*                    order dwords.
*                  - Of course, Win32 allows 250 character filenames.
*                  - Win32 system date and time can be stored in a SYSTEMTIME structure.
*
* 01 Jan. 1999 - Began to add the code to perform a slice.
*              - Got the code for the slice32_header structure initialization.
*              - Added the structure for error handling.
*              - Added code to ensure the destination drive supports removable media.
*              - Added code to separate the actual source filename from any drive and directory information 
*                  that is part of the source filespec.
*              - Removed the opcode variable from main.
*              - Whereas the original slice allowed 99 disks (01-99) NTSLICE will allow 100 disks (00-99).
*                  Two char variables in main, lsd_dsk_id (least significant digit) and msd_dsk_id (most 
*                  significant digit) will track this.
*              - Created a function, get_drive_free_bytes to return the number of bytes available on the 
*                  destination media.  This is essentially a wrapper function for the Win32 function, 
*                  GetDiskFreeSpace.
*              - Renamed dos_rst_header to slice16_header.
*              - Completed the SLICE portion of the code.  It still has to be tested, but the actual grunt 
*                  work is done.
*
* 02 Jan. 1999 - Worked on debugging the slice code.  Had to change all file I/O to use WriteFile and 
*                  ReadFile instead of WriteFileEx and ReadFileEx.  The Ex versions are designed for 
*                  asynchronous I/O ONLY.
*              - It wrote about 41K more than necessary during the test.  I also need to go ahead and flush
*                  the disk buffers for the destination drive before prompting the user to insert the next 
*                  disk.
*              - Still writing more than is necessary at the end.
*              - It's not writing more at the end of the last disk.  There is a problem with filling the last
*                  less than BUFF_SIZE bytes on the previous disks.
*              - FIXED! When dsk_free-cur_xfer was less than BUFF_SIZE, the result of dsk_free-cur_xfer had to
*                  be preserved in order to correctly increment csize and cur_xfer.
*
* 03 Jan. 1999 - Changed name from ntslice to slice32.
*              - Added code in the actual disk transfer loop to account for dsk_free-cur_xfer being less than
*                  BUFF_SIZE and size-csize < dsk_free-cur_xfer.
*              - Added code to set the debug flag to 1 if the last command line argument is debug.  This will 
*                  activate some extremely verbose status messages.
*              - Added code to flush the destination drive buffers before asking the user for the next disk.
*
* 04 Jan. 1999 - Began to add the SPLICE operation code.
*
* 05 Jan. 1999 - Continued with the SPLICE code.
*              - Changed the error handling to jump to SLICE_EXIT if SLICE is the current operation, and 
*                  SPLICE_EXIT if SPLICE is the current operation.
*
* 06 Jan. 1999 - Cleaned up some comments knocked out of whack by renaming EXIT to SLICE_EXIT.
*
* 23 Jan. 1999 - Wrote the file transfer code for the SPLICE operation.  Still needs to be tested and
*                  debugged.
*              - On the first splice, did not properly skip the header of the source.  Also did not end 
*                  properly.
*              - Fixed the not ending problem and the apparent lack of skip of the header problem.
*              - SPLICE worked!  Need to add error trapping, and code to set the file date/time to same as 
*                  original.  However, core works.
*              - Added code to reset the file date and time fields to the original values.  Error trapping is 
*                  all that remains to have a fully functional 32 bit version of SLICE.  Will still have to 
*                  add the code for backwards compatibility with the 16 bit slice.
*
* 24 Jan. 1999 - Added code to build the destination path and file name for the SPLICE operation.  Due to this
*                  oversight, you could only splice to the default drive and directory.
*
* 25 Jan. 1999 - Added the SPLICE error handling code.  Did this with a separate switch/case statement and 
*                  label, SPLICE_EXIT, instead of SLICE_EXIT.  Added the label EXIT so the program has only 
*                  one exit point.
*              - In the copyright notice, changeed the version from V0R0 to V0R1.  It's time for beta testing.
*
* 26 Jan. 1999 - Began to investigate how to build some kind of equivalent to SPLICE.COM on the first 
*                  destination disk.  The first problem is how to get a copy of slice32.exe onto the target.
*                  Under Windows NT, argv[0] is exactly what was typed.  Windows 95, however, and probably 
*                  Windows 98, have an argv[0] that is a fully qualified path and filename for executable.  
*                  I think the Win32 function, SearchPath, will solve this problem.
*
* 27 Jan. 1999 - Added the code to find SLICE32.EXE, as noted in the 26 Jan. 1999 comment.  Used SearchPath.
*              - Added code to copy slice32.exe to the destination media.  Used the Win32 function, CopyFile.
*              - Added code to create splice32.bat on the destination media.  Right now it has no error 
*                  trapping, and is straight line and ugly, but it works.
*              - MAJOR ODD THING, I first named the handle for the SPLICE32.BAT file spl_bat.  After I 
*                  compiled the program (with no errors), the SearchPath call would generate an illegal 
*                  access message.  At first I thought it was stack related, but after checking, I determined
*                  the default stack to be 1 MB.  I then renamed it to spl_cmd.  Same thing happened.  
*                  Spl_batch didn't work either.  Changing the name to spl32_bat made everything work just 
*                  fine.
*
* 28 Jan. 1999 - Added code to the SPLICE operation to restore the original file attributes.
* 
* 29 Jan. 1999 - One of the changes has caused the SLICE operation to fail.  An ACCESS_DENIED error is 
*                  generated when the attempt to lock the destination with DeviceIOControl.  Fixed a bug 
*                  introduced by the splice32.bat generation, csize is used during the slice32.exe copy, 
*                  and was not reset to zero prior to the actual slicing taking place.  Fixing this, did not 
*                  fix the ACCESS_DENIED error.
*              - Commenting out the slice32.exe copy as well as the splice32.bat generation did not fix the 
*                  ACCESS_DENIED error.  However, since Win95 does not support the same IOCTL commands as NT
*                  (missed that in the first pass of the documentation), I think I'll change the slice code 
*                  to not require the buffers be flushed.  This can be done by opening the destination files 
*                  with the FILE_FLAG_NO_BUFFERING flag.
*              - Changed the slice code to open the destination files with the FILE_FLAG_NO_BUFFERING flag, 
*                  and removed the device flush code.  Didn't work worth a damn.  Using this flag requires 
*                  ALL disk i/o to be in multiples of the device's sector size, as well as having the buffers 
*                  aligned on sector size boundaries.
*              - Changing the code to use the FILE_FLAG_WRITE_THROUGH flag.  This disables lazy writing.  It 
*                  looks as if this works.  Copying slice32.c to slice32.c9.
*              - Removing the device flushing code and its associated variables.  Some time, it may be 
*                  profitable to see if the problem with locking the device was related to the problem 
*                  encountered on 27 Jan. 1999 while adding the splice32.bat code.
*
* 30 Jan. 1999 - Fixed a bug in the splice32.bat generation.  It was writing the entire destination path and 
*                  filename for the name of the first sliced file rather than just the filename.
*              - Saved the 16 bit header decode data and code out to slice32.16bit.header.  At this point, I'm
*                  not certain that slice32 needs to be backwardly compatible with original slice files.
*              - Broke the code that copies slice32.exe and creates splice32.bat on the first destination 
*                  media out to a separate function, make_splice32.  The only problem with using a batch file  
*                  to reconstruct the file is after the last disk has been read, the user is prompted for the 
*                  disk with the batch file.
*              - Added code to the SPLICE operation to check for and append, if necessary, a backslash to the 
*                  destination path.  This will permit the restoration of files to a directory other than the 
*                  default on the specified drive.
*
* 02 Feb. 1999 - Created a function, win32_ram, to return the amount of RAM in the machine.  This will be used
*                  to calc the transfer buffer size based on physical RAM, rather than a compile time constant.
*              - Changed BUFF_SIZE from a macro to the variable buff_size in main().
*              - Changed version number to V0R2.
*
* 03 Feb. 1999 - Created function prototypes for win32_ram, get_drive_free_bytes, and make_splice32.
*              - Created function size_xfer_buffer to calculate the appropriate size of the transfer buffer 
*                  based on available physical RAM and the size of the destination media.  Did not test the 
*                  function, just wrote it.
*
* 06 Feb. 1999 - Moved the buff_size initialization code into the individual operation blocks.  Also, instead 
*                  of explicitly allocating 128K to the buffer, it gets sized by size_xfer_buffer function.
*              - Had to fix the comparison amounts for the size_xfer_buffer function.  I neglected to account
*                  for the BPB, MBR, and 2 copies of the FAT.
*              - Changed version number to V0R3.
*
* 08 Feb. 1999 - During the SPLICE operation, I changed the mode of the destination file to 
*                  FILE_FLAG_WRITE_THROUGH.  This will ensure data gets flushed from the buffers before the 
*                  source media gets ejected.  In fact, I changed the mode on all CreateFile calls to 
*                  FILE_FLAG_WRITE_THROUGH.
*              - Boy do I feel stupid.  The FlushFileBuffers function clears the file buffers and writes all 
*                  information out to disk.  Replaced all occurences of FILE_FLAG_WRITE_THROUGH with 
*                  FILE_ATTRIBUTE_NORMAL, and added the appropriate calls to FlushFileBuffers.
*
* 09 Mar. 1999 - Changed the splice code to automatically find the first file of the sliced set.  This will 
*                  remove the need to pass the first file name on the command line.  Works!
*              - Removed code in make_splice32 that created the splice32.bat file on the destination drive.
*              - Added code to the splice operation to prompt the user for the first disk.
*              - Changed version number to V0R4.
*              - Need to change error handling because of the inserted operations in the splice code.
*
* 13 Mar. 1999 - Worked on the comments about the recent FindFirst File calls.
*
* 09 May. 1999 - Began work on the modifications for turning this into a PC Magazine Utility column utility 
*                  and accompanying article.
*              - The first thing to change is the copyright and title notice.  In order to properly incorporate
*                  the changes, I need to separate the usage info from the copyright notice.
*              - Next is to work on integrating split functionality while keeping the integrity of the system.
*              - Split the function protoypes, macro definitions, and structure definitions into slice32.h
*              - Added a new variable to main called opcode.  It is an unsigned long integer.  If it's value 
*                  is 0 then a slice will be performed.  If it's value is 1 then a splice will be performed.  
*                  If it's value is 2 then a split will be performed.
*              - Modified the make_splice32 function so splice32.exe is the destination filename.  This will 
*                  make the argv[1] command redundant in some special cases.
*              - Split the size_xfer_buffer, win32_ram, get_drive_free_bytes, and make_splice32 functions 
*                  into slc32lib.c.  Eventually, these will be stuck into a function library.
*              - Pulled SLICE operation out into a separate function slice.
*              - In order for the slice function to work, I had to make debug a global variable in slice32.h.
*              - Pulled SPLICE operation out into a separate function splice.
*
* 12 May. 1999 - Changes made on 09 May. 1999 appear to be working just fine.
*              - Created a split function to handle splitting a single large file into smaller files in the 
*                  same location as the original.  They get reassembled with a binary copy operation rather 
*                  than a splice.  To keep it simple, main passes split() argv[2-4] as character strings so
*                  that split can do its processing with as little modification as possible.  I will need to
*                  do a massive renaming to bring the variable names into line with the other functions, as 
*                  well as adding similar error processing code.
*              - Added the error processing code to split.
*              - Changed split to use win32_ram instead of directly calling GlobalMemoryStatus.
*              - Split code appears to be working just fine.
*
* 13 May. 1999 - Began to add code to cause the opcode to be set based on the value of argv[0].  This will 
*                  require some finagling because under DOS/Win95 and presumably Win98, argv[0] is a fully
*                  qualified path and filename of the executable, while under NT, argv[0] is simply the 
*                  command.
*              - Killed opcode in main (yet again).  I should probably finalize the decision on this one.
*              - Using renamed slice32.exe to splice32.exe and split32.exe works appears to be working just 
*                  fine on both NT and 95.
*              - Moved the global declaration of debug out of slice32.h, and into slice32.c.  I needed to do
*                  this so I could turn slc32lib.c into slc32lib.lib.
*              - Turned slc32lib.c into a linkable library, and modified the LINK statement to reflect this.
*                  Creating the library allowed me to eliminate the #include "slc32lib.c" statement.
*              - Since this is now a multi-file project, created a MAKEFILE to control builds.
*
* 02 Jul. 1999 - Replaced all calls to _getche with _getch.  This fixes a boneheaded mistake on my part that 
*                  confused _getch (no echo of character) with _getche (echo of character).  Of course I run
*                  into the famous VC++ bug that generates bad code for the IF statement after the prompt in
*                  the slice function.  The splice function works just beautifully.
*              _ _getch is causing the SearchPath call in make_splice32 to bomb.  With _getche, all is OK.
*                  For the time being, slice() uses _getche.
*
* 17 Jul. 1999 - Now using VC++ 6.0 Professional.  Trying the _getch experiment one more time.  Still bombed 
*                  out.
*              - getchar doesn't work any better.  In fact, because it is a stream I/O function, it's worse.
*              - It appears to be some problem with an overflow condition.  keystroke is defined before 
*                  dsk_num, and therein lies the problem.
*              - Changed from straight _getch to a loop based on the return value of _kbhit.
*              - The flag printf statements arounf the make_splice32 call are still acting wierd.  Especially
*                  the one after it returns.  Splice has also stopped working.  This is not a good thing.
*              - Restored the calls to _getche, and the problems are still present.
*
* 18 Jul. 1999 - Must remember splice has two sections where the user is prompted to strike any key.
*              - Fixed the problem with the SearchPath call in make_splice32.  The exec_name variable needs 
*                  to be passed by reference, not defined as a pointer to a pointer.  The slice function is 
*                  now working properly with _kbhit, _getch, etc.
*              - Replaced the _getch call in splice with the _kbhit, _getch combination from slice.  No dice.
*                  Splice still bombs after the first disk.  It never asks for the next disk.
*              - The problem with splice is that slice is setting rstlast to one for all of the disks.  Fixed 
*                  this by initializing fhead32.rstlast to 0 at the start of the slice operation.
*              - Everything WORKS!! and with no keystroke echoing.
*
* 13 Aug. 1999 - Removed a debugging printf.  Still works.
*              - Rather than spin the disk I/O code off into thread ready functions, check into changing the 
*                  Process priority instead.  Kick it down for the reads and writes, and back up for everything
*                  else.
*
* 15 Aug. 1999 - Implemented CPU management during slice operation.  Using the idea from 13 Aug. 1999.  
*                  Appears to work.  Had to use the current thread rather than the current process, but the 
*                  idea worked just the same.
*              - Implemented CPU management during splice operation.
*              - Spun splice into separate executable.  Modified the makefile accordingly.
*              - Modified slc32lib.c to look for splice32.exe in the make_splice32 function.
*
* 22 Aug. 1999 - Modified make_splice32 in slc32lib.c to create splice32.exe from a resource that is part of
*                  slice32.exe.  This keep splice32 and slice32 in one file.  No matter where you go, you still
*                  only need slice32.exe.
*
* 23 Aug. 1999 - Removed main from this source file.
*              - Removed split from this source file.
*              - Removed splice from this source file.
*              - Removed the SLICE and SPLIT errorlevel comments, as well as the separate build instructions 
*                  from this source file.
*              - Removed the global debug flag from this source file.
*              - Removed all debug printf statements from this source file.
*              - Removed the "Slicing source to dest" printf from this source file.
*              - Changed all of the printf calls in the error handler to MessageBox calls.
*              - Converted the disk prompt from printf w/_kbhit to MessageBox.
*              - Removed #include <conio.h>.
*              - Removed the keystroke variable.  It's not needed now that the disk prompts don't use _kbhit 
*                  and _getch.
*
* 24 Aug. 1999 - Added code to change the mouse cursor to the hourglass while the current piece is getting 
*                  written.
*              - Temporarily removed the dynamically sized transfer buffer, and replaced it with a hard coded 
*                  64K buffer.  This will remove a disk not present error caused by no media being present in 
*                  the destination when size_xfer_buffer is called.
*
* 03 Sep. 1999 - Completely removed the call to size_xfer_buffer, and permanently fixed the buffer size at 
*                  64K.
*              - Identified the sections that need modification to support slicing to sections of a 
*                  predefined size.
*
* 05 Sep. 1999 - Added an argument to slice.  It's called portion_size and if it's 0, then we're slicing to 
*                  removable media.  Otherwise we're either slicing to portions of a specified size, or a 
*                  specified number of portions.
*              - Changed the code that allocates the destination filename to allocated enough space properly 
*                  if slicing to sized portions.
*              - Changed the code that checks to see if the destination is a removable drive to work properly 
*                  if slicing to sized portions.
*              - Changed the code that prompts the user for the next disk to work properly if slicing to sized
*                  portions.
*              - Changed the code that sets the number of bytes available on the destination drive to work 
*                  properly if slicing to sized portions.
*              - Removed the spl_src variable.  It was a holdover from from the original splice32.bat stuff.
*
* 18 Sep. 1999 - Removed the marker comments left over from the conversion to slicing to a destination folder 
*                  as well as removable media.
*              - In keeping with the user interface code, a source code width of 110 chars is set here.
*              - All block comments reformatted to 110 chars, and all code within this width.
*
* 22 Sep. 1999 - Added MB_SETFOREGROUND style to all MessageBox calls.
*
* 23 Sep. 1999 - Added MB_ICONERROR style to all message boxes in error handlers.
*              - Changed the code that handles errors while opening the destination file.  If the file
*                  exists, the user is prompted for overwrite confirmation.  If the user elects to over-
*                  write the existing destination, then it gets deleted, and the file open is tried again.
*                  If the user chooses not to overwrite, then slice prompts for another disk.  If an error
*                  occurs while opening the destination, and it's not caused by the destination file already
*                  existing, then the user is asked if he wants to Abort, Retry, Ignore.  If he decides to 
*                  Abort, then slice is cancelled.  If he chooses to Retry or Ignore, then he is prompted
*                  for a new disk.
*              - Added a space check after the user is prompted for the next disk.  If there is less than
*                  (sizeof(fhead32)+1) bytes available on the destination, then the user is prompted to Abort,
*                  Retry, or Ignore.  If he elects to abort, then slice terminates with a return code of 247.
*                  Otherwise, he is prompted for the next disk.  The space check (due to location) has a 
*                  higher priority than the destination file exists check.  So, if the destination file
*                  exists, and is using all of the available space, then the user will have to respond to
*                  the lack of space issue instead of the file already exists issue.
*
* 25 Sep. 1999 - Added a variable that will be 1 larger than dsk_num.  It is this new number that will be
*                  used in all user messages.
*              - In the code that checks to see if there is more than header+1 bytes available on destination
*                  removed the CreateFile call.  This was causing a file overwrite condition to exist when it
*                  shouldn't.
*              - In the code that checks to see if there is more than header+1 bytes available on destination
*                  changed the if(dsk_free<sizeof(fhead32)+1) to while(dsk_free<sizeof(fhead32)+1).  This will
*                  keep prompting the user until a disk with enough space is inserted.
*
* 26 Sep. 1999 - Added code to handle the current destination disk being full due to a copy of the file
*                  already existing there.  Prompts the user to overwrite, if the user says YES, then the
*                  file is deleted, and we can continue.  Otherwise the "Disk Too Full message is displayed".
*                  It looks as though this is finally working the way it should.
*              - The previous fix broke slicing to a destination folder because the destination file only got
*                  opened if the destination was removable.  This is now fixed.
*
* 05 Oct. 1999 - Source editing.  Comment reformatting, etc.
*
* 28 Oct. 1999 - If you have already sliced a large file into sections in a folder (or even off to disk), and
*                  you decide to reslice, for any reason, the same file into the same location, you will be
*                  prompted for each and every overwrite.  To avoid this, I added a global boolean variable
*                  confirm_del.  When it is TRUE, all overwrites (during this slice) are confirmed.  When 
*                  it's FALSE, no overwrites (during this slice) are confirmed; they are just done.  Also, 
*                  added a CANCEL button on the overwrite confirmation messageboxes.  If the user hits CANCEL
*                  the SLICE operation is terminated, and they are sent back to the UI.  This is just in case
*                  they turn on overwrite confirmation, and then decide it's too much hassle.
*
* 28 Oct. 1999 - Working on turning this into a thread function.  Kicking the process priority down works as
*                  far as keeping the disk write from monopolizing the system.  However, it does cause 
*                  problems for the main event processing loop.
*              - Step 1: create a wrapper for the actual slice function.  This is called slice, and the old
*                  slice function is now called slice_thread.  What used to be parameters for slice_thread
*                  are now global variables.  Done, slice operation still works.
*              - Step 2: Enable use of the multi-threaded C RTL.  Added /MT switch to makefile.
*              - Step 3: Specify the __stdcall calling convention for slice_thread.
*              - Step 4: Since linking with LIBCMT.LIB causes a conflict with the standard C RTL, had to
*                  add /NODEFAULTLIB:LIBC.LIB to the makefile.
*              - Step 5: Add the thread creation code, WaitForSingleObject call, GetExitCodeThread call,
*                  and CloseHandle call.
*
* 30 Oct. 1999 - Got the multithreading working.
*              - Removed all of the code and variables used to move the thread priority up and down.
*              - Got rid of the master "Do you want overwrite confirmation turned ON during this SLICE 
*                  operation?" messagebox.
*              - Moved the prototype for slice_thread over to slice32.h.
*              - Implementing newest overwrite control is next.
*
* 31 Oct. 1999 - For the newest overwrite control, I have to delete all of the pre-existing files on the
*                  destination, if the user wants overwrite deletion.  the files have to be enumerated
*                  manually since the DeleteFile function does not work with wildcards.  Newest overwrite
*                  functionality is implemented, and working.
*              - Added code to the slice function to make local copies of the passed parameters.  By doing
*                  this, I can let the user change the settings on the user interface while a slice operation
*                  is executing, and not cause problems.
*
* 01 Nov. 1999 - If the user cancels the slice operation when prompted about overwriting existing files, the
*                  user interface reports that errors were encountered during the slice.  This is a spurious
*                  error message.  Added a special errorcode (246) to indicate the slice was cancelled, and
*                  added code to the cmd_slice button processing in slc32_ui.c to handle it appropriately.
*
* ERRORLEVELS SLICE OPERATION
*               255 = No arguments passed on the command line.
*               254 = Error getting source file attributes.
*               253 = Error opening source file.
*               252 = Error getting source file date/time(s).
*               251 = Error getting source file size.
*               250 = Error allocating buffer to hold destination
*                       filename.
*               249 = Destination is not removable media.
*               248 = Error allocating transfer buffer.
*               247 = Error opening destination file.
*               246 = Slice was cancelled when overwrite was asked
* nmake - to build whole shebang!
************************************************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>                                            /* Has _beginthreadex and _endthreadex */

#include <windows.h>
#include <winnt.h>
#include <winbase.h>                                            /* Includes SearchPath function */

#include "slice32.h"

/************************************************************************************************************
* Global parameters for slice_thread.
************************************************************************************************************/
char *file;                                                     /* File to slice */
char *drv;                                                      /* Slice destination */
unsigned long portion_size;                                     /* Size of each portion */

unsigned long int slice(char *f2slc, char *dest_drive, unsigned long port_size) {
  unsigned long int rtc;                                        /* Slice thread return code */
  unsigned long int sync_rtc;                                   /* WaitForSingleObject return value */
  unsigned long int slc_id;                                     /* Slice thread ID */
  HANDLE            slc_thread;                                 /* Slice thread handle */
  BOOL              tfrtc;                                      /* Boolean return code */
  MSG               messg;

  file=(char *)calloc(strlen(f2slc)+1,sizeof(char));            /* Allocate space to hold a copy of file */
  drv=(char *)calloc(strlen(f2slc)+1,sizeof(char));             /* Allocate space to hold a copy of drv */
  strcpy(file,f2slc);                                           /* Set global parameters from passed args */
  strcpy(drv,dest_drive);
  portion_size=port_size;

  slc_thread=_beginthreadex((void *)NULL,0,slice_thread,NULL,0,&slc_id);

  do {
    tfrtc=PeekMessage(&messg,hwnd_main,0,0,PM_REMOVE);
    if(tfrtc) {
      TranslateMessage(&messg);
      DispatchMessage(&messg);
    }
    sync_rtc=WaitForSingleObject(slc_thread,30);
  } while(sync_rtc!=WAIT_OBJECT_0);

  tfrtc=GetExitCodeThread(slc_thread,&rtc);
  tfrtc=CloseHandle(slc_thread);                                /* Close the thread */

  free(drv);                                                    /* Free local copy of drv */
  free(file);                                                   /* Free local copy of file */

  return(rtc);                                                  /* Return code to UI */
}

unsigned long int __stdcall slice_thread(unsigned long *temp0) {
  unsigned char        *colon;                                  /* Points to : in source name */
  unsigned char        *dir_delim;                              /* Points to \ in source name */
  unsigned char        *src_file;                               /* Bare source filename */
  unsigned char        *dst_file;                               /* Destination filename */
  unsigned char        *xfer_buff;                              /* Buffer for file I/O */
  unsigned char        *del_file;                               /* Preexisting file to delete */
  unsigned char         mb_msg[256];                            /* MessageBox message string */
  unsigned char         lsd_dsk_id;                             /* Disk number lo digit */
  unsigned char         msd_dsk_id;                             /* Disk number hi digit */
  unsigned long int     dsk_num=0;                              /* Current disk number 0-99 */
  unsigned long int     usr_num=1;                              /* User disk number 1-100 */
  unsigned long int     del_ctr;                                /* Used to delete preexisting files */
  unsigned long int     dsk_free;                               /* Free space on current disk */
  unsigned long int     cur_xfer;                               /* Bytes written to current disk */
  unsigned long int     scratch;                                /* Temporary storage */
  unsigned long int     size;                                   /* Source file original size */
  unsigned long int     csize=0;                                /* Current output file size */
  unsigned long int     bytes_rw;                               /* Bytes read or written */
  unsigned long int     errorlevel=0;                           /* ERRORLEVEL return code */
  unsigned long int     rtc;                                    /* Scratch return code */
  unsigned long int     win32_err=0;                            /* GetLastError result */
  unsigned long int     mb_ret=0;                               /* Messagebox return code */
  unsigned long int     buff_size;                              /* Transfer buffer size */
  struct slice32_header fhead32;                                /* 32 bit file header */
  BOOL                  tfrtc;                                  /* Win32 return code */
  BOOL                  ask_del=TRUE;                           /* TRUE=ask at least once before overwrite */
  HANDLE                sl_src;                                 /* Source file */
  HANDLE                sl_dst;                                 /* Destination file */

  /**********************************************************************************************************
  * Initialize rstlast to 0.
  **********************************************************************************************************/
  fhead32.rstlast=0;

  /**********************************************************************************************************
  * Retrieve the system date & time
  **********************************************************************************************************/
  GetLocalTime(&(fhead32.slice_time));

  /**********************************************************************************************************
  * Get the source file attributes
  **********************************************************************************************************/
  fhead32.rstattr=GetFileAttributes(file);
  if(fhead32.rstattr==0xFFFFFFFF) {                             /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error code */
    errorlevel=254;                                             /* Error getting attributes */
    goto SLICE_EXIT;                                            /* Terminate w/errorlevel */
  }

  /**********************************************************************************************************
  * Open the source file 
  **********************************************************************************************************/
  sl_src=CreateFile(file,GENERIC_READ,0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
  if(sl_src==INVALID_HANDLE_VALUE) {                            /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error code */
    errorlevel=253;                                             /* Error opening source file */
    goto SLICE_EXIT;                                            /* Terminate w/errorlevel */
  }

  /**********************************************************************************************************
  * Get the source's Creation, Last Access,and Last write time
  **********************************************************************************************************/
  tfrtc=GetFileTime(sl_src,&(fhead32.creat_time),&(fhead32.acces_time),&(fhead32.write_time));
  if(tfrtc==FALSE) {                                            /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error code */
    errorlevel=252;                                             /* Error getting file date */
    goto SLICE_EXIT;                                            /* Terminate w/errorlevel */
  }

  /**********************************************************************************************************
  * Get the source file's size
  **********************************************************************************************************/
  fhead32.rstlsiz=GetFileSize(sl_src,&(fhead32.rsthsiz));
  if(fhead32.rstlsiz==0xFFFFFFFF) {                             /* If error */
    win32_err=GetLastError();                                   /* Store Win32 error code */
    errorlevel=251;                                             /* Error getting file size */
    goto SLICE_EXIT;                                            /* Terminate w/errorlevel */
  }
  else {                                                        /* If there was no error */
    size=fhead32.rstlsiz;                                       /* Initialize starting size */
  }

  colon=strchr(file,':');                                       /* Point to : if there */
  dir_delim=strrchr(file,'\\');                                 /* Point to last \ if there */

  /**********************************************************************************************************
  * If there is no colon and no directory delimiter, then a bare filename was passed.
  **********************************************************************************************************/
  if((colon==NULL)&&(dir_delim==NULL)) {
    src_file=file;
  }

  /**********************************************************************************************************
  * If there is a colon and no directory delimiter, then the source filename begins immediately after the 
  *   colon.
  **********************************************************************************************************/
  if((colon!=NULL)&&(dir_delim==NULL)) {
    src_file=colon+1;
  }

  /**********************************************************************************************************
  * If there is a directory delimiter, it does not matter if a colon is present or not, the source filename 
  *   begins immediately after the delimiter.
  **********************************************************************************************************/
  if(dir_delim!=NULL) {
    src_file=dir_delim+1;
  }

  memset(fhead32.rstname,'\0',256);                             /* NULL out source filename */
  strcpy(fhead32.rstname,src_file);                             /* Copy the source filename */

  /**********************************************************************************************************
  * Destination filename has to allow for the 2 character destination drive + a potential one character root
  *   directory specification + the terminating NULL.
  **********************************************************************************************************/
  if(portion_size==0) {
    dst_file=(unsigned char *)calloc(strlen(src_file)+4,sizeof(unsigned char));
    if(dst_file==NULL) {                                        /* If error */
      win32_err=1;                                              /* Trigger error message */
      errorlevel=250;                                           /* Dest memory allocation error */
      goto SLICE_EXIT;                                          /* Terminate w/errorlevel */
    }
    else {                                                      /* If there was no error */
      strcpy(dst_file,drv);                                     /* Copy destination drive */
      strcat(dst_file,src_file);                                /* Copy destination filename */
    }
  }
  else {
    dst_file=(unsigned char *)calloc(strlen(src_file)+strlen(drv)+2,sizeof(unsigned char));
    if(dst_file==NULL) {                                        /* If error */
      win32_err=1;                                              /* Trigger error message */
      errorlevel=250;                                           /* Dest memory allocation error */
      goto SLICE_EXIT;                                          /* Terminate w/errorlevel */
    }
    else {                                                      /* If there was no error */
      strcpy(dst_file,drv);                                     /* Copy destination drive */
      strcat(dst_file,src_file);                                /* Copy destination filename */
    }
  }
  /**********************************************************************************************************
  * If the portion size is zero, can only SLICE to removable media.
  **********************************************************************************************************/
  if(portion_size==0) {
    if(GetDriveType(drv)!=DRIVE_REMOVABLE) {
      win32_err=1;                                              /* Trigger error message */
      errorlevel=249;                                           /* Destination not removable */
      goto SLICE_EXIT;                                          /* Terminate w/errorlevel */
    }
  }

  /**********************************************************************************************************
  * Allocate transfer buffer.
  **********************************************************************************************************/
  buff_size=65536;
  xfer_buff=(unsigned char *)calloc(buff_size,sizeof(unsigned char));
  if(xfer_buff==NULL) {                                         /* If error */
    win32_err=1;                                                /* Trigger error message */
    errorlevel=248;                                             /* Error allocating xfer buffer */
    goto SLICE_EXIT;                                            /* Terminate w/errorlevel */
  }

  /**********************************************************************************************************
  * Everything to this point has been initialization.  The real work involved in slicing the file occurs in 
  *   the following loop.
  **********************************************************************************************************/
  while(size>csize) {    
    /********************************************************************************************************
    * Initialize lsd_dsk_id and msd_dsk_id and write them to the last two characters of the destination 
    *   filespec.
    ********************************************************************************************************/
    if(dsk_num<10) {                                            /* If disk number is 0-9 */
      msd_dsk_id='0';                                           /* MSD is 0 */
      lsd_dsk_id=(char)(dsk_num+48);                            /* Convert LSD */
    }
    else {                                                      /* Otherwise */
      msd_dsk_id=(char)((dsk_num/10)+48);                       /* Convert MSD */
      lsd_dsk_id=(char)((dsk_num%10)+48);                       /* Convert LSD */
    }
    dst_file[strlen(dst_file)-1]=lsd_dsk_id;                    /* Store digits into the prompt */
    dst_file[strlen(dst_file)-2]=msd_dsk_id;

    /********************************************************************************************************
    * Prompt user for next disk.
    ********************************************************************************************************/
    if(portion_size==0) {
      memset(mb_msg,'\0',256);
      sprintf(mb_msg,"Please insert disk #%d in drive %s. ",usr_num,drv);
      MessageBox(hwnd_main,mb_msg,"Slice32 - Next Disk",MB_SETFOREGROUND|MB_OK);
    }

    /********************************************************************************************************
    * Before we get any further, let's make sure the disk is not completely full.  If it is, then it's
    *   possible the user left the previous disk in the drive.
    ********************************************************************************************************/
    if(portion_size==0) {
      dsk_free=get_drive_free_bytes(drv);
      while(dsk_free<sizeof(fhead32)+1) {                       /* If no room for header plus one byte*/
        sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
        if(sl_dst==INVALID_HANDLE_VALUE) {                      /* If the file create fails */
          if(GetLastError()==ERROR_FILE_EXISTS) {               /* Because the file already exists */
            if(ask_del==TRUE) {                                 /* If we haven't asked before */
              memset(mb_msg,'\0',256);                          /* Then ask now */
              sprintf(mb_msg,"Do you want to overwrite preexisting files?");
              mb_ret=MessageBox(hwnd_main,mb_msg,"Slice32 - Overwrite?",\
                                MB_ICONEXCLAMATION|MB_SETFOREGROUND|MB_OKCANCEL);
            }
            else {                                              /* Otherwise, we have asked before */
              mb_ret=IDOK;                                      /* Act like the user hit OK */
            }
            if(mb_ret==IDOK ) {                                 /* If overwrite confirmed */
              ask_del=FALSE;                                    /* Don't ask again during this slice */
              del_file=(unsigned char *)calloc(strlen(dst_file)+1,sizeof(unsigned char));
              strcpy(del_file,dst_file);
              for(del_ctr=dsk_num; del_ctr<100; del_ctr++) {
                tfrtc=DeleteFile(del_file);                     /* Delete the file */
                del_file[strlen(del_file)-1]=(char)((del_ctr%10)+48);
                del_file[strlen(del_file)-2]=(char)((del_ctr/10)+48);
              }
              free(del_file);
              break;
            }
            if(mb_ret==IDCANCEL) {                              /* If the user CANCELLED */
              win32_err=0;                                      /* Turn off win32_err message printing */
              errorlevel=246;                                   /* Set Cancel code */
              goto SLICE_EXIT;                                  /* And terminate */
            }
          }          
        }
        else {                                                  /* Delete file avoids spurious file exists */
          tfrtc=CloseHandle(sl_dst);                            /* Close the file so it can be deleted */
          tfrtc=DeleteFile(dst_file);                           /* Delete the file */
        }
        memset(mb_msg,'\0',256);
        sprintf(mb_msg,"%s is not ready.",drv);
        mb_ret=MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",\
                          MB_ICONERROR|MB_SETFOREGROUND|MB_ABORTRETRYIGNORE);
        if(mb_ret==IDABORT) {                                   /* If user elects to Abort */
          errorlevel=247;                                       /* Error opening dest */
          goto SLICE_EXIT;                                      /* Terminate w/errorlevel */
        }
        else {                                                  /* If user elects to retry or ignore */
          memset(mb_msg,'\0',256);                              /* Prompt for a new disk */
          sprintf(mb_msg,"Please insert disk #%d in drive %s. ",usr_num,drv);
          MessageBox(hwnd_main,mb_msg,"Slice32 - Next Disk",MB_SETFOREGROUND|MB_OK);
        }
        dsk_free=get_drive_free_bytes(drv);
      }

      sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
      while(sl_dst==INVALID_HANDLE_VALUE) {                     /* If error */
        win32_err=GetLastError();                               /* Store Win32 error */
        if(win32_err==ERROR_FILE_EXISTS) {                      /* If the destination file exists */
          if(ask_del==TRUE) {                                   /* If we haven't asked before */
            memset(mb_msg,'\0',256);                            /* Then ask now */
            sprintf(mb_msg,"Do you want to overwrite preexisting files?");
            mb_ret=MessageBox(hwnd_main,mb_msg,"Slice32 - Overwrite?",\
                              MB_ICONEXCLAMATION|MB_SETFOREGROUND|MB_OKCANCEL);
          }
          else {                                                /* Otherwise, we have asked before */
            mb_ret=IDOK;                                        /* Act like the user hit OK */
          }
          if(mb_ret==IDOK) {                                    /* If user wants to overwrite */
            ask_del=FALSE;                                      /* Don't ask again during this slice */
            del_file=(unsigned char *)calloc(strlen(dst_file)+1,sizeof(unsigned char));
            strcpy(del_file,dst_file);
            for(del_ctr=dsk_num; del_ctr<100; del_ctr++) {
              tfrtc=DeleteFile(del_file);                       /* Delete the file */
              del_file[strlen(del_file)-1]=(char)((del_ctr%10)+48);
              del_file[strlen(del_file)-2]=(char)((del_ctr/10)+48);
            }
            free(del_file);
            sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
            break;
          }
          else if(mb_ret==IDCANCEL) {                           /* If the user CANCELLED */
            win32_err=0;                                        /* Turn off win32_err message printing */
            errorlevel=246;                                     /* Set Cancel code */
            goto SLICE_EXIT;                                    /* And terminate */
          }
          else {                                                /* If user doesn't want overwrite or cancel */
            memset(mb_msg,'\0',256);                            /* Prompt for a new disk */
            sprintf(mb_msg,"Please insert disk #%d in drive %s. ",usr_num,drv);
            MessageBox(hwnd_main,mb_msg,"Slice32 - Next Disk",MB_SETFOREGROUND|MB_OK);
            sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
          }
        }
        else {                                                  /* If error other than file already exists */
          memset(mb_msg,'\0',256);
          sprintf(mb_msg,"Error while opening %s.  Error Code = %ld",dst_file,win32_err);
          mb_ret=MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",\
                            MB_ICONERROR|MB_SETFOREGROUND|MB_ABORTRETRYIGNORE);
          if(mb_ret==IDABORT) {                                 /* If user elects to Abort */
            errorlevel=247;                                     /* Error opening dest */
            goto SLICE_EXIT;                                    /* Terminate w/errorlevel */
          }
          else {                                                /* If user elects to retry or ignore */
            memset(mb_msg,'\0',256);                            /* Prompt for a new disk */
            sprintf(mb_msg,"Please insert disk #%d in drive %s. ",usr_num,drv);
            MessageBox(hwnd_main,mb_msg,"Slice32 - Next Disk",MB_SETFOREGROUND|MB_OK);
            sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
          }
        }
      }
    }
    /********************************************************************************************************
    * If we're not working on removable media, we still need to open the destination file, and provide
    *   overwrite verification.
    ********************************************************************************************************/
    else {
      sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
      while(sl_dst==INVALID_HANDLE_VALUE) {                     /* If error */
        win32_err=GetLastError();                               /* Store Win32 error */
        if(win32_err==ERROR_FILE_EXISTS) {                      /* If the destination file exists */
          if(ask_del==TRUE) {                                   /* If we haven't asked before */
            memset(mb_msg,'\0',256);                            /* Then ask now */
            sprintf(mb_msg,"Do you want to overwrite preexisting files?");
            mb_ret=MessageBox(hwnd_main,mb_msg,"Slice32 - Overwrite?",\
                              MB_ICONEXCLAMATION|MB_SETFOREGROUND|MB_OKCANCEL);
          }
          else {                                                /* Otherwise, we have asked before */
            mb_ret=IDOK;                                        /* Act like the user hit OK */
          }
          if(mb_ret==IDOK) {                                    /* If user wants to overwrite */
            ask_del=FALSE;                                      /* Don't ask again during this slice */
            del_file=(unsigned char *)calloc(strlen(dst_file)+1,sizeof(unsigned char));
            strcpy(del_file,dst_file);
            for(del_ctr=dsk_num; del_ctr<100; del_ctr++) {
              tfrtc=DeleteFile(del_file);                       /* Delete the file */
              del_file[strlen(del_file)-1]=(char)((del_ctr%10)+48);
              del_file[strlen(del_file)-2]=(char)((del_ctr/10)+48);
            }
            free(del_file);
            sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
            break;
          }
          else if(mb_ret==IDCANCEL) {                           /* If the user CANCELLED */
            win32_err=0;                                        /* Turn off win32_err message printing */
            errorlevel=246;                                     /* Set Cancel code */
            goto SLICE_EXIT;                                    /* And terminate */
          }
          else {                                                /* If user doesn't want overwrite or cancel */
            memset(mb_msg,'\0',256);                            /* Prompt for a new disk */
            sprintf(mb_msg,"Please insert disk #%d in drive %s. ",usr_num,drv);
            MessageBox(hwnd_main,mb_msg,"Slice32 - Next Disk",MB_SETFOREGROUND|MB_OK);
            sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
          }
        }
        else {                                                  /* If error other than file already exists */
          memset(mb_msg,'\0',256);
          sprintf(mb_msg,"Error while opening %s.  Error Code = %ld",dst_file,win32_err);
          mb_ret=MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",\
                            MB_ICONERROR|MB_SETFOREGROUND|MB_ABORTRETRYIGNORE);
          if(mb_ret==IDABORT) {                                 /* If user elects to Abort */
            errorlevel=247;                                     /* Error opening dest */
            goto SLICE_EXIT;                                    /* Terminate w/errorlevel */
          }
          else {                                                /* If user elects to retry or ignore */
            memset(mb_msg,'\0',256);                            /* Prompt for a new disk */
            sprintf(mb_msg,"Please insert disk #%d in drive %s. ",usr_num,drv);
            MessageBox(hwnd_main,mb_msg,"Slice32 - Next Disk",MB_SETFOREGROUND|MB_OK);
            sl_dst=CreateFile(dst_file,GENERIC_WRITE,0,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
          }
        }
      }
    }
    win32_err=0;

    /********************************************************************************************************
    * Create SPLICE32.EXE on destination.
    ********************************************************************************************************/
    if(dsk_num==0) {                                            /* Only necessary on first disk */
      rtc=make_splice32(drv);
    }

    /********************************************************************************************************
    * Get the number of bytes available on the destination drive and subtract the size of the header.
    ********************************************************************************************************/
    if(portion_size==0) {
      dsk_free=get_drive_free_bytes(drv);
    }
    else {
      dsk_free=portion_size;
    }
    dsk_free-=sizeof(fhead32);

    /********************************************************************************************************
    * If there is more, or just enough, space on the destination disk than is needed for the data left to 
    *   write, then this is the last disk in the set. 
    ********************************************************************************************************/
    if(dsk_free>=(size-csize)) {
      fhead32.rstlast=1;
    }

    /********************************************************************************************************
    * Write the header out to the destination file.
    ********************************************************************************************************/
    tfrtc=WriteFile(sl_dst,&fhead32,sizeof(fhead32),&bytes_rw,NULL);

    /********************************************************************************************************
    * Fill the current disk in buff_size chunks, if possible. Otherwise, write out the exact number of bytes
    *   needed.
    ********************************************************************************************************/
    cur_xfer=0;
    while((dsk_free>cur_xfer)&&(size>csize)) {
      if((dsk_free-cur_xfer)>=buff_size) {
        if((size-csize)>=buff_size) {
          tfrtc=ReadFile(sl_src,xfer_buff,buff_size,&bytes_rw,NULL);
          tfrtc=WriteFile(sl_dst,xfer_buff,buff_size,&bytes_rw,NULL);
          cur_xfer+=buff_size;
          csize+=buff_size;                                     /* Increment bytes written */
        }
        else {
          tfrtc=ReadFile(sl_src,xfer_buff,(size-csize),&bytes_rw,NULL);
          tfrtc=WriteFile(sl_dst,xfer_buff,(size-csize),&bytes_rw,NULL);
          cur_xfer+=(size-csize);
          csize+=(size-csize);                                  /* Increment bytes written */
        }
      }
      else {
        if((size-csize)>=(dsk_free-cur_xfer)) {
          tfrtc=ReadFile(sl_src,xfer_buff,(dsk_free-cur_xfer),&bytes_rw,NULL);
          tfrtc=WriteFile(sl_dst,xfer_buff,(dsk_free-cur_xfer),&bytes_rw,NULL);
          scratch=(dsk_free-cur_xfer);
          cur_xfer+=scratch;
          csize+=scratch;                                       /* Increment bytes written */
        }
        else {
          tfrtc=ReadFile(sl_src,xfer_buff,(size-csize),&bytes_rw,NULL);
          tfrtc=WriteFile(sl_dst,xfer_buff,(size-csize),&bytes_rw,NULL);
          cur_xfer+=(size-csize);
          csize+=(size-csize);                                  /* Increment bytes written */
        }
      }
      memset(xfer_buff,'\0',buff_size);
    }
    /********************************************************************************************************
    * Flush the buffers for the destination file out to disk so it can be ejected without data loss.
    ********************************************************************************************************/
    tfrtc=FlushFileBuffers(sl_dst);
    tfrtc=CloseHandle(sl_dst);                                  /* Close this disk file */
    dsk_num++;                                                  /* Increment disk number */
    usr_num++;                                                  /* Increment user disk number */
  }                                                             /* Finish with this disk */

  SLICE_EXIT:                                                   /* Begin SLICE cleanup code */
  switch(errorlevel) {
    case   0:
    case 246:                                                   /* Cancel code */
    case 247: free(xfer_buff);                                  /* Free transfer buffer */
    case 248: if(win32_err!=0) {
                memset(mb_msg,'\0',256);
                sprintf(mb_msg,"Error allocating transfer buffer\n");
                MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",MB_ICONERROR|MB_SETFOREGROUND|MB_OK);
                win32_err=0;
              }
    case 249: if(win32_err!=0) {
                memset(mb_msg,'\0',256);
                sprintf(mb_msg,"Destination: %s is not removable\n",drv);
                MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",MB_ICONERROR|MB_SETFOREGROUND|MB_OK);
                win32_err=0;
              }
               free(dst_file);                                  /* Free destination filename */
    case 250: if(win32_err!=0) {
                memset(mb_msg,'\0',256);
                sprintf(mb_msg,"Error allocating destination name buffer\n");
                MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",MB_ICONERROR|MB_SETFOREGROUND|MB_OK);
                win32_err=0;
              }
    case 251: if(win32_err!=0) {
                memset(mb_msg,'\0',256);
                sprintf(mb_msg,"Error getting size of %s = %ld\n",\
                        file,win32_err);
                MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",MB_ICONERROR|MB_SETFOREGROUND|MB_OK);
                win32_err=0;
              }
    case 252: if(win32_err!=0) {
                memset(mb_msg,'\0',256);
                sprintf(mb_msg,"Error getting date/time(s) for %s = %ld\n",\
                        file,win32_err);
                MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",MB_ICONERROR|MB_SETFOREGROUND|MB_OK);
                win32_err=0;
              }
              tfrtc=CloseHandle(sl_src);                        /* Close source file */
    case 253: if(win32_err!=0) {
                memset(mb_msg,'\0',256);
                sprintf(mb_msg,"Error opening %s = %ld\n",file,win32_err);
                MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",MB_ICONERROR|MB_SETFOREGROUND|MB_OK);
                win32_err=0;
              }
    case 254: if(win32_err!=0) {
                memset(mb_msg,'\0',256);
                sprintf(mb_msg,"Error getting attributes for %s = %ld\n",\
                        file,win32_err);
                MessageBox(hwnd_main,mb_msg,"Slice32 - Slice Error",MB_ICONERROR|MB_SETFOREGROUND|MB_OK);
                win32_err=0;
              }
    default : _endthreadex(errorlevel);                         /* Exit */
  }
}

