#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	Uninstall the Integration API
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# look for a program in the path
#
look_for_pgm ()
{
	pgm=$1

	l=`echo $PATH | sed -e 's/:/ /g'`
	for j in $l
	do
		[ -d $j -a -x $j/$pgm ] && return 0
	done

	return 1
}

# ------------------------------------------------------------------------
# delete a group
#
delete_grp ()
{
	GRP="$1"

	in_windows && return

	if look_for_pgm groupdel
	then
		groupdel $GRP >/dev/null 2>&1
	elif look_for_pgm rmgroup
	then
		rmgroup $GRP >/dev/null 2>&1
	elif look_for_pgm pw
	then
		pw groupdel $GRP >/dev/null 2>&1
	fi
}

# ------------------------------------------------------------------------
# delete a user
#
delete_usr ()
{
	USR="$1"

	in_windows && return

	if look_for_pgm userdel
	then
		userdel -r $USR >/dev/null 2>&1 ||
		userdel    $USR >/dev/null 2>&1
	elif look_for_pgm rmuser
	then
		rmuser -y $USR >/dev/null 2>&1
	elif look_for_pgm pw
	then
		pw userdel $USR -r >/dev/null 2>&1 ||
		pw userdel $USR >/dev/null 2>&1
	fi
}


# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# Return the user name to use
#
usrname ()
{
	echo "${VSI_USRNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# Return the group name to use
#
grpname ()
{
	echo "${VSI_GRPNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -u     keep user entry"
	echo "  -g     keep group entry"
	echo "  -f     force uninstall (no prompt)"
	echo "  -o     only remove non-installed files & directories"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D     debug uninstall"
}

# ------------------------------------------------------------------------
# unsetup all symlinks
#
unsetup_symlinks ()
{
	if [ -d "$VSIFAX/api/cpp" ]
	then
	(
		cd "$VSIFAX/api/cpp"

		if [ -d lib ]
		then
		(
			cd lib

			l=`ls *.a 2>/dev/null`
			if [ "$l" != "" ]
			then
				for i in $l
				do
					rm -f /usr/lib/$i
				done
			fi
		)
		fi

		if [ -d include ]
		then
		(
			cd include

			l=`ls *.h 2>/dev/null`
			if [ "$l" != "" ]
			then
				for i in $l
				do
					rm -f /usr/include/$i
				done
			fi
		)
		fi
	)
	fi
}

# ------------------------------------------------------------------------
# main()
#
USRNAME=`usrname`
GRPNAME=`grpname`
PRDNAME=`prdname`

[ -f /etc/$PRDNAME.sh ] && . /etc/$PRDNAME.sh
if [ x"$VSIFAX" = x"" ]
then
	echo "$PRDNAME is not installed. Nothing to remove."
	exit 0
fi


# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

DEBUG=
DELETE_USR=yes
DELETE_GRP=yes
FORCE=no
ONLY=no

case "$1" in
	'-?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | -helpall | --helpall)
		helpall
		exit 0
		;;

	'-V' | -version | --version)
		[ -x "$VSIFAX/bin/vfxadmin" ] && echo "$PGM: `vfxadmin version`"
		exit 0
		;;
esac

ARGS=`getopt "Dugfo" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-x
		shift
		;;

	-u)
		DELETE_USR=no
		shift
		;;

	-g)
		DELETE_GRP=no
		shift
		;;

	-f)
		FORCE=yes
		shift
		;;

	-o)
		ONLY=yes
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

[ "$DEBUG" != "" ] && set $DEBUG

# ------------------------------------------------------------------------
# check if we should check if OK
#
if [ "$FORCE" = "no" ]
then
	echo ""
	echo "This will completely uninstall the `pkgname` Integration API from $VSIFAX."
	getyesorno -n "" "Do you want to proceed?" || exit 0
fi

# ------------------------------------------------------------------------
# check that user is root
#
check_root

# ------------------------------------------------------------------------
# now all uninstall functions
#
unsetup_symlinks

# ------------------------------------------------------------------------
# now remove all installed files & directories
#
if [ "$ONLY" != "yes" ]
then
	rm -rf "$VSIFAX"/api

	rm -f "$VSIFAX"/lbin/vsi-faxapi-*
	l=`cd "$VSIFAX"/lbin; ls`
	[ "$l" = "" ] && rmdir "$VSIFAX"/lbin

	# ----------------------------------------------------------------
	# check if anything else left in $VSIFAX
	#
	l=`cd "$VSIFAX"; ls`
	[ "$l" = "" ] && rmdir "$VSIFAX"
fi

# ------------------------------------------------------------------------
# remove any system stuff
#
#[ "$DELETE_USR" = "yes" ] && delete_usr $USRNAME
#[ "$DELETE_GRP" = "yes" ] && delete_grp $GRPNAME
