package com.vsi.xmlf;

import java.util.Vector;

/**
 * This class encapsulates the common attributes of all
 * XML-F request objects.
 *
 */
public abstract
class Request implements XmlfObject {

	protected Account _account = null;
	protected String _svc_ref  = null;
	protected String _app_ref  = null;
	protected String _cmd_ref  = null;
	protected ResponseFormat _rsp_fmt  = ResponseFormat.XML;

	protected Vector _rsp_listeners = new Vector();
	protected TransportManager _transport_mgr = null;
	

	/**
	 * Assignes the values/attributes in the XML data to this object.
	 *
	 * @param  root_element  the XML data to assign from
	 */
	public void setFromXml (XmlElement root_element) {

		setResponseFormat(ResponseFormat.XML);
		setServiceReference(null);
		setApplicationReference(null);
		setCommandReference(null);
		setAccount(null);

		if (root_element == null) {
			return;
		}

		setTag(root_element.getName());
		
		setResponseFormat(ResponseFormat.stringToObject(
			root_element.getAttribute("response-format")));

		XmlElement xml[] = root_element.getSubElements();

		setServiceReference(
			XmlElement.getData(xml, "service-reference"));
		setApplicationReference(
			XmlElement.getData(xml, "application-reference"));
		setCommandReference(
			XmlElement.getData(xml, "command-reference"));

		XmlElement account_element = XmlElement.findElement(
						xml, "account");
		if (account_element != null) {
			setAccount(new Account(account_element));
		}
	}


	/**
	 * Gets the <code>Account</code> object associated with the request.
	 */
	public Account getAccount () {
		return (_account);
	}


	/**
	 * Gets the service reference associated with the request.
	 *
	 * @return  the service-reference element
	 */
	public synchronized String getServiceReference () {
		return (_svc_ref);
	}


	/**
	 * Gets the application reference associated with the request.
	 *
	 * @return  the application-reference element
	 */
	public String getApplicationReference () {
		return (_app_ref);
	}


	/**
	 * Gets the command reference associated with the request.
	 *
	 * @return  the command-reference element
	 */
	public String getCommandReference () {
		return (_cmd_ref);
	}


	/**
	 * Gets the response format for the request.
	 *
	 * @return  the response-format attribute
	 */
	public ResponseFormat getResponseFormat () {
		return (_rsp_fmt);
	}


	/**
	 * Sets the <code>Account</code> object associated with the request.
	 *
	 * @param   account  the <code>Account</code> object to assign.
	 */
	public void setAccount (Account account) {
		_account = account;
	}


	/**
	 * Sets the service reference associated with the request.
	 *
	 * @param  svc_ref  the application refernce to assign.
	 */
	public synchronized void setServiceReference (String svc_ref) {
		_svc_ref = svc_ref;
	}


	/**
	 * Sets the application reference associated with the request.
	 *
	 * @param  app_ref  the application refernce to assign.
	 */
	public void setApplicationReference (String app_ref) {
		_app_ref = app_ref;
	}


	/**
	 * Sets the command reference associated with the request.
	 *
	 * @param  cmd_ref  the command refernce to assign.
	 */
	public void setCommandReference (String cmd_ref) {
		_cmd_ref = cmd_ref;
	}


	/**
	 * Assigns a value to the response format.
	 *
	 * @param  rsp_fmt  the value for the response format.
	 */
	public void setResponseFormat (ResponseFormat rsp_fmt) {
		_rsp_fmt = rsp_fmt;
	}


	/**
	 * Send the request to the specified URL using the default 
	 * Transportmanager.
	 *
	 * @param  url_spec  the URL of the fax server
	 * @return  the Response object from the server
	 */
	public Response send (String url_spec) {

		if (_transport_mgr == null) {
			_transport_mgr = TransportManager.getDefault();		
		}

		return (_transport_mgr.send(url_spec, this));
	}


	/**
	 * Send the request asynchronously to the specified URL using 
	 * the default Transportmanager.
	 *
	 * @param  url_spec  the URL of the fax server
	 */
	public void backgroundSend (String url_spec) {

		if (_transport_mgr == null) {
			_transport_mgr = TransportManager.getDefault();		
		}
	
		_transport_mgr.backgroundSend (url_spec, this);
	}


       /**
        * Add a response event listener.
        *
        * @param  l  The ResponseListener to add.
        */
       public synchronized void addResponseListener (ResponseListener l) {

               if (!_rsp_listeners.contains(l)) {
                       _rsp_listeners.addElement(l);
               }
       }


       /**
        * Remove a response event listener.
        *
        * @param  l  The ResponseListener to remove.
        */
       public synchronized void removeResponseListener (ResponseListener l) {

               if (!_rsp_listeners.contains(l)) {
                       _rsp_listeners.removeElement(l);
               }
       }


        /**
         * Notify listening objects of response events.
         *
	 * @param  the ResponseEvent to notify for
         */
        public void fireResponseEvent (Response rsp) {

                /*
                 * Make a copy of the listener object vector so that it cannot
                 * be changed while we are firing events
                 */
                Vector v;
                synchronized(this) {
                        if (_rsp_listeners.size() < 1) {
                                return;
                        }

                        v = (Vector)_rsp_listeners.clone();
                }

                /*
                 * Create the event object
                 */
                ResponseEvent evt = new ResponseEvent(this, rsp);


                /*
                 * Fire the event to all listeners
                 */
                int count = v.size();
                for (int i = 0; i < count; i++) {
                        ResponseListener l = (ResponseListener)v.elementAt(i);
                        l.response(evt);
                }
        }
}
