package com.vsi.xmlf;

/**
 * This class is an abstract class which is used to encapsulate a
 * XMLF body element.
 *
 */
public abstract
class Body implements XmlfObject {

	protected String _content_type = null;
	protected String _content_transfer_encoding = null;

	protected String _tag = "body";


	/**
	 * No args constructor.
	 */
	public Body () {
	}

	
	/**
	 * Sub-classes must implement this. This method returns the
	 * data for the body element.
	 *
	 * @return  the data to be inlined as the body element
	 */
	public abstract String getData ();


	/**
	 * Gets the filename for the body element.
	 *
	 * @return the filename for the body element
	 */
	public abstract String getFilename ();


	/**
	 * Gets the XML tag for this object. This method is implemented by
         * all classes implementing the XmlfObject interface. The tag is
         * utilized in the toXml() method.
	 *
	 * @return  the tag for this XML object
	 * @see     #setTag
	 * @see     #toXml
	 */
	public synchronized String getTag () {
		return (_tag);
	}


	/**
	 * Sets the XML tag for this object. This method is implemented by
         * all classes implementing the XmlfObject interface.
         *
         * @param  tag  the XML tag name for this object
	 * @see    #getTag
	 */
	public synchronized void setTag (String tag) {
		_tag = tag;
	}


	/**
	 * Gets a string containing the XML for this object. Use
	 * <code>toXml(String indent)</code> to output indented XML.
	 *
	 * @return  a String containing the XML for this object
	 * @see     #toXml(String indent)
	 */
	public String toXml () {
		return (toXml(""));
	}


	/**
	 * Gets a string containing the indented XML for this object. Use
	 * <code>toXml()</code> to output non-formatted XML.
	 *
	 * @return  a String containing the indented XML for this object
	 * @see     #toXml()
	 */
	public String toXml (String indent) {

		StringBuffer xml = new StringBuffer();

		if (indent == null)
			indent = "";

		xml.append(indent + "<" + getTag());
		if (getContentTransferEncoding() != null) {
			xml.append(" content-transfer-encoding=\"" +
				getContentTransferEncoding() + "\"");
		}

		if (getContentType() != null) {
			xml.append(" content-type=\"" +
				getContentType() + "\"");
		}

		xml.append(" filename=\"" + getFilename() + "\">\n");
		xml.append(getData() + "\n");
		xml.append(indent + "</"  + getTag()      + ">\n");
		return (xml.toString());	
	}


	/**
	 * Gets the content-transfer-encoding assigned to the Body object.
	 *
	 * @return  the content-transfer-encoding for this body element
	 */
	public String getContentTransferEncoding () {
		return (_content_transfer_encoding);
	}


	/**
	 * Gets the content-type assigned to the Body object.
	 *
	 * @return  the content-type for this body element
	 */
	public String getContentType () {
		return (_content_type);
	}


	/**
	 * Assigns encoding as the content-transfer-encoding for
	 * the Body object.
	 *
	 * @param  encoding  the value to assign to the
	 * 		     content-transfer-encoding.
	 */
	public void setContentTransferEncoding (String encoding) {
		_content_transfer_encoding = encoding;
	}


	/**
	 * Assigns content_type as the content-type of the Body object.
	 *
	 * @param  content_type the value to assign to the content-type.
	 */
	public void setContentType (String content_type) {
		_content_type = content_type;
	}
}

