/*------------------------------------------------------------------------
 * Program to manipulate the person entries in a phone-book
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"csLh:f:F:au:l:x:t:o:WU:H:Z:?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options]\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -s           Use system phone-book\n");
	fprintf(fp, "  -n           Use client phone-book\n");
	fprintf(fp, "  -c           Use local  phone-book\n");
	fprintf(fp, "\n");
	fprintf(fp, "  -h on|off    Display header\n");
	fprintf(fp, "  -f flds      Specify field list\n");
	fprintf(fp, "  -F spec      Output format (csv,pipe,eval)\n");
	fprintf(fp, "\n");
	fprintf(fp, "  -L           List all entries\n");
	fprintf(fp, "  -a           Add entry\n");
	fprintf(fp, "  -u alias     Update entry for alias\n");
	fprintf(fp, "  -l alias     List   entry for alias\n");
	fprintf(fp, "  -x alias     Delete entry for alias\n");
	fprintf(fp, "\n");
	fprintf(fp, "  -t tag=val   Add tag entry\n");
	fprintf(fp, "  -o order     Sort order for list\n");
	fprintf(fp, "\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
}

/*------------------------------------------------------------------------
 * display a PERSON entry
 */
static void
display_person (V_TAG *per_tags)
{
	int count = VSI_Tag_Count(per_tags, 0);
	int i;

	for (i=0; i<count; i++)
	{
		const char *tag_name = VSI_Tag_Get_Name_By_Index(per_tags, i,
			0, 0);
		const char *tag_data = VSI_Tag_Get_Data_By_Index(per_tags, i,
			0, 0);

		if (*tag_data != 0)
			printf("%s = %s\n", tag_name, tag_data);
	}
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	char		old_key[V_TAG_DATALEN];
	V_TAG *		tag_list	= 0;
	V_TAG *		per_tags	= 0;
	const char *	pgmname		= argv[0];
	const char *	pbtype		= V_PB_TYPE_DFLT;
	const char *	order		= V_PB_SORT_DFLT;
	char		palias[V_TAG_DATALEN];
	int		cmd		= 0;
	const char *	flds		= 0;
	int		do_hdr		= -1;
	int		do_dump		= FALSE;
	int		delim		= ',';
	int		quotes		= 1;
	int		c;
	int		rc;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		const char *opt = VSI_Opt_Argstr();

		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'c':
			/*------------------------------------------------
			 * use local  phone-book
			 */
			pbtype = V_PB_TYPE_LCL;
			break;

		case 'n':
			/*------------------------------------------------
			 * use client phone-book
			 */
			pbtype = V_PB_TYPE_USR;
			break;

		case 's':
			/*------------------------------------------------
			 * use server phone-book
			 */
			pbtype = V_PB_TYPE_SYS;
			break;

		case 'f':
			/*------------------------------------------------
			 * specify field list
			 */
			flds = opt;
			do_dump = TRUE;
			break;

		case 'h':
			/*------------------------------------------------
			 * header option
			 */
			if (strcmp(opt, "on") == 0)
			{
				do_hdr = TRUE;
			}
			else if (strcmp(opt, "off") == 0)
			{
				do_hdr = FALSE;
			}
			else
			{
				fprintf(stderr,
					"%s: Invalid header option %s\n",
					pgmname, opt);
				return (EXIT_FAILURE);
			}
			break;

		case 'F':
			/*------------------------------------------------
			 * specify dump format
			 */
			do_dump = TRUE;
			if (strcmp(opt, "csv") == 0)
			{
				delim  = ',';
				quotes = 1;
			}
			else if (strcmp(opt, "pipe") == 0)
			{
				delim  = '|';
				quotes = 0;
			}
			else if (strcmp(opt, "eval") == 0)
			{
				delim  = 0;
				quotes = 0;
			}
			else
			{
				fprintf(stderr, "%s: Invalid dump format %s\n",
					pgmname, opt);
				return (EXIT_FAILURE);
			}
			break;

		case 'L':
			/*------------------------------------------------
			 * list all entries
			 */
			cmd = c;
			break;

		case 'a':
			/*------------------------------------------------
			 * add an entry
			 */
			cmd = c;
			break;

		case 'u':
		case 'l':
		case 'x':
			/*------------------------------------------------
			 * update/list/delete an antry
			 */
			cmd = c;
			strcpy(palias, opt);
			break;

		case 't':
			/*------------------------------------------------
			 * process tag in form "tag=value"
			 */
			tag_list = VSI_Tag_Add_Str(tag_list, opt, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 'o':
			/*------------------------------------------------
			 * sort order
			 */
			order = opt;
			break;

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * already processed
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * check if a cmd was given
	 */
	if (cmd == 0)
	{
		fprintf(stderr, "%s: No command specified\n", pgmname);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * adjust header option
	 *
	 * If not specified, default is:
	 *	do_dump == 0	on
	 *	do_dump == 1	off
	 */
	if (do_hdr == -1)
		do_hdr = ! do_dump;

	/*----------------------------------------------------------------
	 * check if header requested for dump option
	 */
	if (do_dump && do_hdr)
	{
		if (delim != 0)
		{
			/*------------------------------------------------
			 * only do header if not eval format
			 */
			char fld_hdr[BUFSIZ];

			VSI_Tag_Fields(V_TAG_PER_TBLNAME, fld_hdr, delim, 0);
			printf("%s\n", fld_hdr);
		}
	}

	/*----------------------------------------------------------------
	 * perform requested function
	 */
	switch (cmd)
	{
	case 'L':
		/*--------------------------------------------------------
		 * list all entries
		 */
		*old_key = 0;
		while (TRUE)
		{
			per_tags = VSI_PB_Person_Get(vptr, pbtype, old_key,
				order, V_DBCMD_NEXT, msgbuf);
			if (per_tags == 0)
				break;

			if (do_dump)
			{
				char str[BUFSIZ];

				rc = VSI_Tag_To_Str(per_tags, str, flds,
					delim, quotes, msgbuf);
				if (rc)
					fprintf(stderr, "%s: %s\n",
						pgmname, msgbuf);
				else
					printf("%s\n", str);
			}
			else
			{
				display_person(per_tags);
				printf("\n");
				rc = 0;
			}

			VSI_Tag_Get_Data_By_Name(per_tags, V_TAG_PER_PALIAS,
				old_key, 0);
			strcat(old_key, " ");

			VSI_Tag_Free(per_tags, 0);

			if (rc)
				break;
		}
		break;

	case 'a':
		/*--------------------------------------------------------
		 * add an entry
		 */
		rc = VSI_PB_Person_Add(vptr, pbtype, 0, tag_list,
			msgbuf);
		break;

	case 'u':
		/*--------------------------------------------------------
		 * update an entry
		 */
		rc = VSI_PB_Person_Upd(vptr, pbtype, palias, tag_list,
			msgbuf);
		break;

	case 'l':
		/*--------------------------------------------------------
		 * list an entry
		 */
		per_tags = VSI_PB_Person_Get(vptr, pbtype, palias,
			V_PB_SORT_ALIAS, V_DBCMD_EQUAL, msgbuf);
		if (per_tags == 0)
		{
			rc = -1;
		}
		else
		{
			if (do_dump)
			{
				char str[BUFSIZ];

				rc = VSI_Tag_To_Str(per_tags, str, flds,
					delim, quotes, msgbuf);
				if (rc)
					fprintf(stderr, "%s: %s\n",
						pgmname, msgbuf);
				else
					printf("%s\n", str);
			}
			else
			{
				display_person(per_tags);
				rc = 0;
			}
			VSI_Tag_Free(per_tags, 0);
		}
		break;

	case 'x':
		/*--------------------------------------------------------
		 * delete an entry
		 */
		rc = VSI_PB_Person_Upd(vptr, pbtype, palias, tag_list,
			msgbuf);
		break;
	}

	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	/*----------------------------------------------------------------
	 * free the tag list
	 */
	tag_list = VSI_Tag_Free(tag_list, 0);

	/*----------------------------------------------------------------
	 * check if any errors
	 */
	if (rc)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	return (EXIT_SUCCESS);
}
