/*------------------------------------------------------------------------
 * Program to list OLOG (output-log) entries
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"rdgat:u:h:f:F:WU:H:Z:?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] [reqid]\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -r               Display in reverse order\n");
	fprintf(fp, "  -a               Show all entries\n");
	fprintf(fp, "  -g               Show group entries\n");
	fprintf(fp, "  -d               Show detail entries\n");
	fprintf(fp, "  -t tag<op>value  Show entries matching tag\n");
	fprintf(fp, "                   Ops: \"=\", \"!=\", \">\", \"<\", \">=\", \"<=\"\n");
	fprintf(fp, "  -u user          Show entries for user only\n");
	fprintf(fp, "  -h on|off        Display header\n");
	fprintf(fp, "  -f flds          Specify field list\n");
	fprintf(fp, "  -F spec          Output format (csv,pipe,eval)\n");
	fprintf(fp, "  -U username      Username to login as\n");
	fprintf(fp, "  -H hostname      Hostname to connect to\n");
	fprintf(fp, "reqid              request-id to display\n");
}

/*------------------------------------------------------------------------
 * extract a name with a default
 */
static const char *
tag_name (V_TAG *tags, const char *name, const char *dflt)
{
	const char *p = VSI_Tag_Get_Data_By_Name(tags, name, 0, 0);

	if (p == 0 || *p == 0)
	{
		if (dflt != 0)
			p = dflt;
		else
			p = "<none>";
	}
	return (p);
}

/*------------------------------------------------------------------------
 * display a reg OLOG entry
 */
static void
display_olog_reg (V_TAG *tags, int do_hdr)
{
	char time_buf[12];

	const char *seqno	= tag_name(tags, V_TAG_OLOG_SEQNO,      0);
	const char *subtime	= tag_name(tags, V_TAG_OLOG_SUBMITTIME, 0);
	const char *numpgs	= tag_name(tags, V_TAG_OLOG_NUMPGS,     "0");
	const char *clientid	= tag_name(tags, V_TAG_OLOG_CLIENTID,   0);
	const char *tofnum	= tag_name(tags, V_TAG_OLOG_TOFNUM,     0);
	const char *aresult	= tag_name(tags, V_TAG_OLOG_ARESSTR,    0);
	const char *rresult	= tag_name(tags, V_TAG_OLOG_RRESSTR,    0);
	const char *numatts	= tag_name(tags, V_TAG_OLOG_NUMATTS,    0);
	const char *done	= tag_name(tags, V_TAG_OLOG_DONE,       0);

	static int  first_time	= 1;

	/*----------------------------------------------------------------
	 * display header if first call
	 */
	if (do_hdr && first_time)
	{
		first_time = 0;
		printf(
"Req-id   Submitted   npg  Clientid  Phone number      Attmpt  Result  att\n");
	}

	/*----------------------------------------------------------------
	 * convert submit time from "yyyy/mm/dd hh:mm:ss"
	 * to "mm/dd hh:mm"
	 */
	sprintf(time_buf, "%11.11s", subtime+5);

	/*----------------------------------------------------------------
	 * display record
	 */
	done = (VSI_Util_Str_To_Bool(done, 0) ? "*" : "");

	printf("%-6s  %s  %3s  %-8.8s  %-16.16s  %-6.6s  %-6.6s  %2s%s\n",
		seqno,
		time_buf,
		numpgs,
		clientid,
		tofnum,
		aresult,
		rresult,
		numatts,
		done);
}

/*------------------------------------------------------------------------
 * display a grp OLOG entry
 */
static void
display_olog_grp (V_TAG *tags, int do_hdr)
{
	char time_buf[12];

	const char *gseqno	= tag_name(tags, V_TAG_OLOG_GSEQNO,     0);
	const char *subtime	= tag_name(tags, V_TAG_OLOG_SUBMITTIME, 0);
	const char *numpgs	= tag_name(tags, V_TAG_OLOG_NUMPGS,     "0");
	const char *clientid	= tag_name(tags, V_TAG_OLOG_CLIENTID,   0);
	const char *num_mems	= tag_name(tags, V_TAG_OLOG_NUMMEMS,    "0");
	const char *num_queued	= tag_name(tags, V_TAG_OLOG_NUMQUEUED,  "0");
	const char *num_sent	= tag_name(tags, V_TAG_OLOG_NUMSENT,    "0");
	const char *num_failed	= tag_name(tags, V_TAG_OLOG_NUMFAILED,  "0");
	const char *done	= tag_name(tags, V_TAG_OLOG_DONE,       0);

	static int  first_time	= 1;

	/*----------------------------------------------------------------
	 * display header if first call
	 */
	if (do_hdr && first_time)
	{
		first_time = 0;
		printf(
"Req-id   Submitted   npg  Clientid  mems  qued  sent  fail  status\n");
	}

	/*----------------------------------------------------------------
	 * convert submit time from "yyyy/mm/dd hh:mm:ss"
	 * to "mm/dd hh:mm"
	 */
	sprintf(time_buf, "%11.11s", subtime+5);

	/*----------------------------------------------------------------
	 * display record
	 */
	done = (VSI_Util_Str_To_Bool(done, 0) ? "done" : "active");

	printf("%-6s  %s  %3s  %-8.8s  %4s  %4s  %4s  %4s  %s\n",
		gseqno,
		time_buf,
		numpgs,
		clientid,
		num_mems,
		num_queued,
		num_sent,
		num_failed,
		done);
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	V_TAG *		tag_list	= 0;
	V_TAG *		comp_tags	= 0;
	const char *	pgmname		= argv[0];
	char		seqno_buf[V_TAG_DATALEN];
	const char *	reqid		= 0;
	const char *	flds		= 0;
	int		all		= FALSE;
	int		grp		= FALSE;
	int		detail		= FALSE;
	int		do_dump		= FALSE;
	int		do_hdr		= -1;
	int		delim		= ',';
	int		quotes		= 1;
	int		dbcmd		= V_DBCMD_PREV;
	int		opt_index;
	int		c;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		const char *opt = VSI_Opt_Argstr();

		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'r':
			/*------------------------------------------------
			 * display in reverse order (forward)
			 */
			dbcmd = V_DBCMD_NEXT;
			break;

		case 'a':
			/*------------------------------------------------
			 * show all entries
			 */
			all = TRUE;
			break;

		case 'g':
			/*------------------------------------------------
			 * show group entries
			 */
			grp = TRUE;
			break;

		case 'd':
			/*------------------------------------------------
			 * show detail entries
			 */
			detail = TRUE;
			break;

		case 't':
			/*------------------------------------------------
			 * show entries matching tag
			 */
			comp_tags = VSI_Tag_Add_Str(comp_tags, opt, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr,
					"%s: Invalid match tag string %s\n",
					pgmname, opt);
				return (EXIT_FAILURE);
			}
			break;

		case 'u':
			/*------------------------------------------------
			 * show tags for user only
			 */
			comp_tags = VSI_Tag_Add_Comp(comp_tags,
				V_TAG_OLOG_CLIENTID, opt, V_COMP_EQ, 0);
			break;

		case 'h':
			/*------------------------------------------------
			 * header option
			 */
			if (strcmp(opt, "on") == 0)
			{
				do_hdr = TRUE;
			}
			else if (strcmp(opt, "off") == 0)
			{
				do_hdr = FALSE;
			}
			else
			{
				fprintf(stderr,
					"%s: Invalid header option %s\n",
					pgmname, opt);
				return (EXIT_FAILURE);
			}
			break;

		case 'f':
			/*------------------------------------------------
			 * specify field list
			 */
			flds = opt;
			do_dump = TRUE;
			break;

		case 'F':
			/*------------------------------------------------
			 * specify dump format
			 */
			do_dump = TRUE;
			if (strcmp(opt, "csv") == 0)
			{
				delim  = ',';
				quotes = 1;
			}
			else if (strcmp(opt, "pipe") == 0)
			{
				delim  = '|';
				quotes = 0;
			}
			else if (strcmp(opt, "eval") == 0)
			{
				delim  = 0;
				quotes = 0;
			}
			else
			{
				fprintf(stderr, "%s: Invalid dump format %s\n",
					pgmname, opt);
				return (EXIT_FAILURE);
			}
			break;

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * already processed
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * check if reqid specified
	 */
	*seqno_buf = 0;

	opt_index = VSI_Opt_Index();
	if (opt_index < argc)
	{
		dbcmd = V_DBCMD_EQUAL;
		reqid   = argv[opt_index++];
		strcpy(seqno_buf, reqid);
	}

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * adjust header option
	 *
	 * If not specified, default is:
	 *	do_dump == 0	on
	 *	do_dump == 1	off
	 */
	if (do_hdr == -1)
		do_hdr = ! do_dump;

	/*----------------------------------------------------------------
	 * check if header requested for dump option
	 */
	if (do_dump && do_hdr)
	{
		if (delim != 0)
		{
			/*------------------------------------------------
			 * only do header if not eval format
			 */
			char fld_hdr[BUFSIZ];

			VSI_Tag_Fields(V_TAG_OLOG_TBLNAME, fld_hdr, delim, 0);
			printf("%s\n", fld_hdr);
		}
	}

	/*----------------------------------------------------------------
	 * loop through all records or do a specific reqid
	 */
	if (reqid != 0)
	{
		/*------------------------------------------------
		 * get list of status tags
		 */
		tag_list = VSI_Server_Olog_Get(vptr, seqno_buf, grp, dbcmd,
			msgbuf);

		if (tag_list == 0)
		{
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		}
		else
		{
			/*------------------------------------------------
			 * display olog info
			 */
			if (do_dump)
			{
				char str[BUFSIZ];

				VSI_Tag_To_Str(tag_list, str, flds, delim,
					quotes, msgbuf);
				printf("%s\n", str);
			}
			else
			{
				if (grp)
					display_olog_grp(tag_list, do_hdr);
				else
					display_olog_reg(tag_list, do_hdr);
			}
	
			/*------------------------------------------------
			 * free the tag list
			 */
			tag_list = VSI_Tag_Free(tag_list, 0);
		}
	}
	else
	{
		V_LIST * olog_list;
		V_LIST * ol;

		while (TRUE)
		{
			/*------------------------------------------------
			 * get list of status tags
			 */
			olog_list = VSI_Server_Olog_List(vptr, seqno_buf,
				grp, dbcmd, comp_tags, detail, all, 25, msgbuf);
			if (olog_list == 0)
				break;

			/*------------------------------------------------
			 * do each entry
			 */
			for (ol=olog_list; ol; ol=VSI_List_Next(ol))
			{
				tag_list = VSI_List_Tags(ol);

				if (do_dump)
				{
					char str[BUFSIZ];

					VSI_Tag_To_Str(tag_list, str, flds,
						delim, quotes, msgbuf);
					printf("%s\n", str);
				}
				else
				{
					if (grp)
						display_olog_grp(tag_list, do_hdr);
					else
						display_olog_reg(tag_list, do_hdr);
				}
			}
	
			/*------------------------------------------------
			 * free the olog list
			 */
			olog_list = VSI_List_Free(olog_list, 0);
		}
	}

	VSI_Tag_Free(comp_tags, 0);
	
	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	return (EXIT_SUCCESS);
}
