/*------------------------------------------------------------------------
 * Header for the VSIFAX C/C++ Fax API
 *------------------------------------------------------------------------*/

#ifndef VSI_H
#define VSI_H

#include <stdio.h>
#include <sys/types.h>

/*========================================================================
 * This part is for ANSI compatibility.  For a few years, we cannot rely
 * on having ANSI compilers around, but we want to try hard to use their
 * features transparently when possible.
 *
 *------------------------------------------------------------------------
 * If the compiler thinks it is ANSI, turn on all of the features only if
 * the respective enabling macro is undefined (it very well could be set
 * to 0 which disables it even on an ANSI compiler).
 *
 *	__STDC__	1 if ANSI compliant, but often 0 for a compiler
 *			that is mostly ANSI but not fully (say, because
 *			it permits extensions). We just use the existence
 *			of this rather than its value.
 *
 *	__GNUC__	GNU compilers are always ANSI
 *	__cplusplus	.. so are C++
 *	_MSC_VER	any modern Microsoft compiler
 */
#if defined(__STDC__)    || \
    defined(__GNUC__)    || \
    defined(__cplusplus) || \
    defined(_MSC_VER)    || \
    defined(_WIN32)

#  ifndef V_STDC_PROTOTYPES
#    define	V_STDC_PROTOTYPES	1
#  endif

#  ifndef V_STDC_CONST
#    define	V_STDC_CONST		1
#  endif

#endif

/*------------------------------------------------------------------------
 * For external function declarations, we like to use a form that either
 * uses prototypes if possible or doesn't if they are not supported.
 */
#if defined(V_NO_PROTOTYPES) && ! defined(__cplusplus)
#  ifdef V_STDC_PROTOTYPES
#    undef V_STDC_PROTOTYPES
#  endif
#endif

#if defined(V_USE_PROTOTYPES)
#  ifndef V_STDC_PROTOTYPES
#    define V_STDC_PROTOTYPES		1
#  endif
#endif

#if V_STDC_PROTOTYPES
#  define P(args)	args
#else
#  define P(args) 	()
#endif

/*------------------------------------------------------------------------
 * The "const" type qualifier is a no-op if not supported.
 */
#if defined(V_NO_CONST) && ! defined(__cplusplus)
#  ifdef V_STDC_CONST
#    undef V_STDC_CONST
#  endif
#endif

#if defined(V_USE_CONST)
#  ifndef V_STDC_CONST
#    define V_STDC_CONST 1
#  endif
#endif

#if ! V_STDC_CONST && ! defined(__cplusplus)
#  define const		/* nothing */
#endif

/*
 * End of ANSI compatability section
 *========================================================================*/

/*------------------------------------------------------------------------
 * Unix / Windows differences
 */
#if defined(_MSC_VER) || defined(_WIN32)
#else
#  define __stdcall		/* nothing */
#  define __cdecl		/* nothing */
#  define __declspec(x)		/* nothing */
#endif

/*------------------------------------------------------------------------
 * C++ encapsulation
 */
#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------------
 * misc defines used everywhere
 */
#ifndef TRUE
#define TRUE		1
#endif

#ifndef FALSE
#define FALSE		0
#endif

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0
#endif

#ifndef EXIT_FAILURE
#define EXIT_FAILURE	1
#endif

/*------------------------------------------------------------------------
 * Environment variables used by VSI-FAX for default values
 */
#define V_ENVVAR_USERID		"VSIUSER"	/* VSI client id	*/
#define V_ENVVAR_USERNAME	"VSINAME"	/* VSI username		*/
#define V_ENVVAR_LANGUAGE	"VSILANG"	/* VSI language		*/
#define V_ENVVAR_LOCALE		"VSILOCALE"	/* VSI locale		*/
#define V_ENVVAR_TIMEZONE	"VSITZ"		/* VSI timezone		*/
#define V_ENVVAR_USERHOME	"VSIHOME"	/* VSI home directory	*/
#define V_ENVVAR_DATEFMT	"VSIDATE"	/* VSI date format	*/
#define V_ENVVAR_TIMEFMT	"VSITIME"	/* VSI time format	*/
#define V_ENVVAR_HOSTNAME	"VSIHOST"	/* VSI server hostname	*/
#define V_ENVVAR_OPTIONS	"VSIOPTS"	/* VSI options		*/
#define V_ENVVAR_TEMPDIR	"VSITEMP"	/* VSI temp directory	*/
#define V_ENVVAR_HOMEDIR	"VSIHOME"	/* VSI home directory	*/

/*------------------------------------------------------------------------
 * data lengths
 */
#define V_TAG_NAMELEN		32		/* max length of name	*/
#define V_TAG_DATALEN		256		/* max length of data	*/

#define V_MSG_LEN		256		/* length of msg buffer	*/

#define V_MAX_PATH_LEN		256		/* max pathname length	*/
#define V_MAX_FILE_LEN		128		/* max filename length	*/

/*------------------------------------------------------------------------
 * special tag values
 */
#define V_FILE_DUMMY		"*dummy*"	/* dummy filename for att */
#define V_FILE_NO_OFN		"*none*"	/* no OFN entry generated */

/*------------------------------------------------------------------------
 * VSI system objects
 */
typedef void	V_CTX;		/* opaque ptr to VSI context object	*/
typedef void	V_SYS;		/* opaque ptr to VSI connect object	*/

/*------------------------------------------------------------------------
 * VSI list of tags
 */
typedef struct v_tag	V_TAG;			/* VSI tag structure	*/
struct v_tag
{
	V_TAG *		t_next;			/* ptr to next tag	*/
	V_TAG *		t_prev;			/* ptr to prev tag	*/

	char		t_name[V_TAG_NAMELEN];	/* tag name		*/
	char		t_data[V_TAG_DATALEN];	/* tag data		*/
	int		t_comp;			/* tag compare op	*/
};

/*------------------------------------------------------------------------
 * VSI list of tag-lists
 */
typedef struct v_list	V_LIST;			/* VSI list structure	*/
struct v_list
{
	V_LIST *	l_next;			/* ptr to next entry	*/
	V_LIST *	l_prev;			/* ptr to prev entry	*/

	V_TAG *		l_tags;			/* ptr to tag list	*/
};

/*========================================================================
 * function prototypes
 *
 * Note that in all calls that take a msgbuf argument, a NULL pointer
 * may be passed, indicating you are not interested in any error messages.
 * If a pointer is passed, it is assumed to point to a buffer that is at
 * least V_MSG_LEN bytes long.
 */

/*------------------------------------------------------------------------
 * V_LIST functions
 */
#if VSI_NOMACROS

extern V_LIST *		__stdcall VSI_List_Next
							P((V_LIST *l));

extern V_LIST *		__stdcall VSI_List_Prev
							P((V_LIST *l));

extern V_TAG *		__stdcall VSI_List_Tags
							P((V_LIST *l));

#else

#define VSI_List_Next(l)	((l) == 0 ? 0 : (l)->l_next)
#define VSI_List_Prev(l)	((l) == 0 ? 0 : (l)->l_prev)
#define VSI_List_Tags(l)	((l) == 0 ? 0 : (l)->l_tags)

#endif

extern V_LIST *		__stdcall VSI_List_Free
							P((V_LIST *list,
							char *msgbuf));

extern int		__stdcall VSI_List_Count
							P((const V_LIST *list,
							char *msgbuf));

extern const V_LIST *	__stdcall VSI_List_Find_By_Index
							P((const V_LIST *list,
							int index,
							char *msgbuf));

extern const V_TAG *	__stdcall VSI_List_Get_Tags
							P((const V_LIST *list,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_List_Add
							P((V_LIST *list,
							V_TAG *tags,
							char *msgbuf));

/*------------------------------------------------------------------------
 * V_TAG functions
 */
#if VSI_NOMACROS

extern V_TAG *		__stdcall VSI_Tag_Next
							P((V_TAG *tags));

extern V_TAG *		__stdcall VSI_Tag_Prev
							P((V_TAG *tags));

extern char *		__stdcall VSI_Tag_Name
							P((V_TAG *tags));

extern char *		__stdcall VSI_Tag_Data
							P((V_TAG *tags));

#else

#define VSI_Tag_Next(t)		((t) == 0 ? 0 : (t)->t_next)
#define VSI_Tag_Prev(t)		((t) == 0 ? 0 : (t)->t_prev)
#define VSI_Tag_Name(t)		((t) == 0 ? 0 : (t)->t_name)
#define VSI_Tag_Data(t)		((t) == 0 ? 0 : (t)->t_data)

#endif

extern V_TAG *		__stdcall VSI_Tag_Free
							P((V_TAG *tags,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add
							P((V_TAG *tags,
							const char *name,
							const char *data,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Comp
							P((V_TAG *tags,
							const char *name,
							const char *data,
							int comp,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Str
							P((V_TAG *tags,
							const char *str,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Ini_Sys
							P((V_TAG * tags,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Ini_Usr
							P((V_TAG * tags,
							char *msgbuf));

extern int		__stdcall VSI_Tag_Count
							P((const V_TAG *tags,
							char *msgbuf));

extern const V_TAG *	__stdcall VSI_Tag_Find_By_Index
							P((const V_TAG *tags,
							int index,
							char *msgbuf));

extern const V_TAG *	__stdcall VSI_Tag_Find_By_Name
							P((const V_TAG *tags,
							const char *name,
							char *msgbuf));

extern const V_TAG *	__stdcall VSI_Tag_Find_By_Data
							P((const V_TAG *tags,
							const char *data,
							char *msgbuf));

extern const char *	__stdcall VSI_Tag_Get_Name_By_Index
							P((const V_TAG *tags,
							int index,
							char *name_buf,
							char *msgbuf));

extern const char *	__stdcall VSI_Tag_Get_Name_By_Data
							P((const V_TAG *tags,
							const char *data,
							char *name_buf,
							char *msgbuf));

extern const char *	__stdcall VSI_Tag_Get_Data_By_Index
							P((const V_TAG *tags,
							int index,
							char *data_buf,
							char *msgbuf));

extern const char *	__stdcall VSI_Tag_Get_Data_By_Name
							P((const V_TAG *tags,
							const char *name,
							char *data_buf,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Char
							P((V_TAG *list,
							const char *name,
							const char *str,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Bool
							P((V_TAG *list,
							const char *name,
							int bval,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Word
							P((V_TAG *list,
							const char *name,
							int wval,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Add_Time
							P((V_TAG *list,
							const char *name,
							time_t tval,
							char *msgbuf));

extern const char *	__stdcall VSI_Tag_Get_Char
							P((const V_TAG *list,
							const char *name,
							char *msgbuf));

extern int		__stdcall VSI_Tag_Get_Bool
							P((const V_TAG *list,
							const char *name,
							char *msgbuf));

extern int		__stdcall VSI_Tag_Get_Word
							P((const V_TAG *list,
							const char *name,
							char *msgbuf));

extern time_t		__stdcall VSI_Tag_Get_Time
							P((const V_TAG *list,
							const char *name,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Load
							P((V_TAG *tags,
							const char *path,
							char *msgbuf));

extern int		__stdcall VSI_Tag_Save
							P((const V_TAG *tags,
							const char *path,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Append
							P((V_TAG *orig,
							const V_TAG *other,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Copy
							P((const V_TAG *tags,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_Prepend
							P((V_TAG *orig,
							const V_TAG *other,
							char *msgbuf));

extern int		__stdcall VSI_Tag_To_Str
							P((const V_TAG *tags,
							char *str,
							const char *flds,
							int delim,
							int quotes,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Tag_From_Str
							P((const char *str,
							const char *flds,
							int delim,
							char *msgbuf));

extern int		__stdcall VSI_Tag_Fields
							P((const char *tbl,
							char *flds,
							int delim,
							char *msgbuf));

extern int		__stdcall VSI_Tag_Delete_Files
							P((const V_TAG *tags,
							const char *dir,
							char *msgbuf));

/*------------------------------------------------------------------------
 * data/string conversion utility functions
 */
extern char *		__stdcall VSI_Util_Time_To_Str
							P((time_t timeval,
							char *timestr,
							char *msgbuf));

extern char *		__stdcall VSI_Util_Word_To_Str
							P((int wordval,
							char *wordstr,
							char *msgbuf));

extern char *		__stdcall VSI_Util_Bool_To_Str
							P((int boolval,
							char *boolstr,
							char *msgbuf));

extern time_t		__stdcall VSI_Util_Str_To_Time
							P((const char *timestr,
							char *msgbuf));

extern int		__stdcall VSI_Util_Str_To_Word
							P((const char *wordstr,
							char *msgbuf));

extern int		__stdcall VSI_Util_Str_To_Bool
							P((const char *boolstr,
							char *msgbuf));

/*------------------------------------------------------------------------
 * misc utility functions
 */
extern int		__stdcall VSI_Util_Version
							P((char *version,
							char *build,
							char *msgbuf));

extern int		__stdcall VSI_Util_Crypt
							P((const char *str,
							char *buf,
							char *msgbuf));

extern int		__stdcall VSI_Util_Encrypt
							P((const char *str,
							const char *key,
							char *buf,
							char *msgbuf));

extern int		__stdcall VSI_Util_Decrypt
							P((const char *str,
							const char *key,
							char *buf,
							char *msgbuf));

extern const char *	__stdcall VSI_Util_Temp_Dir
							P((char *buf));

extern const char *	__stdcall VSI_Util_Temp_File
							P((char *buf,
							const char *prefix,
							const char *dir,
							const char *ext));

extern const char *	__stdcall VSI_Util_Home_Dir
							P((char *buf));

extern void		__stdcall VSI_Util_Sleep
							P((int ms));

/*------------------------------------------------------------------------
 * debugging
 */
extern void		__cdecl   VSI_Debug_Printf
							P((FILE *fp,
							const char *fmt,
							...));

/*------------------------------------------------------------------------
 * cmdline utility functions
 */
typedef struct
{
	char	cmdbuf[5120];		/* cmd line buffer	*/
	char *	nextp;
	int	argc;			/* arg count		*/
	char *	argv[100];		/* room for 100 args	*/
	char **	pargv;
	char *	pstart;	
} Cmdline;

extern void		__stdcall VSI_Cmdline_Parse
							P((Cmdline *cp,
							const char *str));

extern int		__stdcall VSI_Cmdline_argc
							P((Cmdline *cp));

extern char **		__stdcall VSI_Cmdline_argv
							P((Cmdline *cp));

/*------------------------------------------------------------------------
 * replacement for UNIX getopt() routine
 */
extern int		__stdcall VSI_Opt_Get
							P((int argc,
							char **argv,
							const char *optstr,
							char *msgbuf));

extern int		__stdcall VSI_Opt_Index		P((void));
extern int		__stdcall VSI_Opt_Option	P((void));
extern const char *	__stdcall VSI_Opt_Argstr	P((void));
extern void		__stdcall VSI_Opt_Reset		P((void));

/*------------------------------------------------------------------------
 * time conversion routines
 *
 * The format string consists of a string with various values
 * specified by a % followed by a format specifier.
 * This format follows the conventions for the ANSI routine strftime().
 * See the strftime(3C) man page for details.
 *
 * The language is affected by the LC_CTYPE env variable and the
 * timezone is affected by the TZ environment variable.
 * See the setlocale(3C) and ctime(3C) man pages for details.
 *
 * The format characters used are as follows:
 *
 *	%a	Abbreviated weekday name.
 *
 *	%A	Full weekday name.
 *
 *	%b	Abbreviated month name.
 *
 *	%B	Full month name.
 *
 *	%c	Appropriate date & time representation.
 *
 *	%C	Date & time representation as produced by ctime(3C).
 *
 *	%C	Century number (the year divided by 100 and truncated
 *		to an integer as a decimal number [1,99]);
 *		single digits are preceded by 0.
 *
 *	%d	Day of month [1,31]; single digits are preceded by 0.
 *
 *	%D	Date as %m/%d/%y.
 *
 *	%e	Day of month [1,31]; single digits are preceded by a space.
 *
 *	%g	Week-based year within century [00,99].
 *
 *	%G	Week-based year, including the century [0000,9999].
 *
 *	%h	Abbreviated month name.
 *
 *	%H	Hour (24-hour clock) [0,23]; single digits are preceded by 0.
 *
 *	%I	Hour (12-hour clock) [1,12]; single digits are preceded by 0.
 *
 *	%j	Day number of year [1,366];  single digits are preceded by 0.
 *
 *	%k	Hour (24-hour clock) [0,23]; single digits preceded by a blank.
 *
 *	%l	Hour (12-hour clock) [1,12]; single digits preceded by a blank.
 *
 *	%m	Month number [1,12]; single digits are preceded by 0.
 *
 *	%M	Minute [00,59]; leading 0 is permitted but not required.
 *
 *	%p	Either AM or PM.
 *
 *	%r	Appropriate time representation in 12-hour clock format with %p.
 *
 *	%R	Time as %H:%M.
 *
 *	%S	Seconds [00,61]; the range of values is [00,61] rather
 *		than [00,59] to allow for the occasional leap second and
 *		even more occasional double leap second.
 *
 *	%T	Time as %H:%M:%S.
 *
 *	%u	Weekday as a decimal number [1,7], with 1 representing Monday.
 *
 *	%U	Week number of year as a decimal number [00,53],
 *		with Sunday as the first day of week 1.
 *
 *	%V	The ISO 8601 week number as a decimal number [01,53].
 *		In the ISO 8601 week-based system, weeks begin on a
 *		Monday and week 1 of the year is the week that includes
 *		both January 4th and the first Thursday of the year.
 *		If the first Monday of January is the 2nd, 3rd, or 4th,
 *		the preceding days are part of the last week of the
 *		preceding year.
 *
 *	%w	Weekday as a decimal number [0,6], with 0 representing Sunday.
 *
 *	%W	Week number of year as a decimal number [00,53],
 *		with Monday as the first day of week 1.
 *
 *	%x	Appropriate date representation.
 *
 *	%X	Appropriate time representation.
 *
 *	%y	Year within century [00,99].
 *
 *	%Y	Year, including the century.
 *
 *	%Z	Time zone name or abbreviation, or "" if no TZ info exists.
 *
 *	Examples (data is "2002/05/20 12:48:16 pm"),
 *	LC_CTYPE is "C", and the TZ is "PST8PDT":
 *
 *	%C			"Mon May 20 12:48:16 PDT 2002"
 *	%m/%d/%y %I:%M %p 	"05/20/02 12:48 PM"
 *
 * These codes are also used in the following tags:
 *	V_TAG_ENV_DATEFMT
 *	V_TAG_ENV_TIMEFMT
 *	V_TAG_OLOG_DATEFMT
 *	V_TAG_OLOG_TIMEFMT
 *	V_TAG_USER_DATEFMT
 *	V_TAG_USER_TIMEFMT
 */
extern char *		__stdcall VSI_Time_Format_Val
							P((time_t t,
							const char *fmt,
							char *buf,
							int buflen,
							char *msgbuf));

extern char *		__stdcall VSI_Time_Format_Str
							P((const char *str,
							const char *fmt,
							char *buf,
							int buflen,
							char *msgbuf));

/*------------------------------------------------------------------------
 * memory functions
 *
 * These calls provide for extensive debugging of memory allocations.
 * If the compile-time variable V_DEBUG_MEMORY is set, then the filename
 * and line number is stored for each allocation made, which makes
 * for maximum bebugging info at the expense of executable size.
 *
 * The following environment variables are queried at run-time:
 *
 *	V_MALLOC_NOFREE		if set, don't actually free any data
 *				(allows catching a free of already freed data)
 *				[value is ignored]
 *
 *	V_MALLOC_CHAIN		if set, output the malloc chain on error
 *				[value is ignored]
 *
 *	V_MALLOC_DUMP=nnn	if set, dump data along with the chain
 *				[value is max chars per entry, 0 == all]
 *
 *	V_MALLOC_ABORT		if set, abort if error occurs (after all msgs)
 *				[value is ignored]
 *
 *	V_MALLOC_NOSTDERR	if set, output msgs via OutputDebugString()
 *				(Windows only)
 *				[value is ignored]
 *
 *	V_MALLOC_NULLCHK	if set, check for frees of a NULL pointer
 *				[value is ignored]
 *
 *	V_MALLOC_CHKALLOC	if set, check chain at each allocate
 *				[value is ignored]
 *
 *	V_MALLOC_CHKFREE	if set, check chain at each free
 *				[value is ignored]
 *
 *	V_MALLOC_DMPALLOC	if set, dump alloc info for each alloc
 *				[value is ignored]
 *
 *	V_MALLOC_DMPFREE	if set, dump free info for each free
 *				[value is ignored]
 *
 *	V_MALLOC_ATEXIT		if set, outputs a usage summary at exit time
 *				[value is ignored]
 *
 *	V_MALLOC_DBGFILE	if set, names a debug file for msgs
 *				[default is stderr]
 *
 */
#ifdef V_DEBUG_MEMORY

#define VSI_Memory_Malloc(s)	VSI_Memory_Debug_Malloc(s, \
						__FILE__, __LINE__)
#define VSI_Memory_Calloc(n,s)	VSI_Memory_Debug_Calloc(n, s, \
						__FILE__, __LINE__)
#define VSI_Memory_Realloc(p,s)	VSI_Memory_Debug_Realloc(p, s, \
						__FILE__, __LINE__)
#define VSI_Memory_Free(p)	VSI_Memory_Debug_Free(p, \
						__FILE__, __LINE__)
#define VSI_Memory_Strdup(s)	VSI_Memory_Debug_Strdup(s, \
						__FILE__, __LINE__)

#else

extern void *		__stdcall VSI_Memory_Malloc
							P((int size));

extern void *		__stdcall VSI_Memory_Calloc
							P((int num,
							int size));

extern void *		__stdcall VSI_Memory_Realloc
							P((void *ptr,
							int size));

extern void		__stdcall VSI_Memory_Free
							P((void *ptr));

extern char *		__stdcall VSI_Memory_Strdup
							P((const char *str));

#endif /* V_DEBUG_MEMORY */

extern void *		__stdcall VSI_Memory_Debug_Malloc
							P((int size,
							const char *file,
							int line));

extern void *		__stdcall VSI_Memory_Debug_Calloc
							P((int num,
							int size,
							const char *file,
							int line));

extern void *		__stdcall VSI_Memory_Debug_Realloc
							P((void *ptr,
							int size,
							const char *file,
							int line));

extern void		__stdcall VSI_Memory_Debug_Free
							P((void *ptr,
							const char *file,
							int line));

extern char *		__stdcall VSI_Memory_Debug_Strdup
							P((const char *str,
							const char *file,
							int line));

extern int		__stdcall VSI_Memory_Count
							P((void));

extern void		__stdcall VSI_Memory_Dump
							P((FILE *fp,
							int dump_data,
							int maxlen));

/*------------------------------------------------------------------------
 * TIFF file functions
 */
extern V_TAG *		__stdcall VSI_TIFF_Header
							P((const char *path,
							char *msgbuf));

extern int		__stdcall VSI_TIFF_Copy
							P((const char *out_path,
							const char *inp_path,
							const char *pg_range,
							const char *resolution,
							int vsitags,
							char *msgbuf));

extern int		__stdcall VSI_TIFF_Merge
							P((const char *out_path,
							const V_TAG *files,
							const char *resolution,
							int vsitags,
							char *msgbuf));

/*------------------------------------------------------------------------
 * whoami functions (takes either a V_CTX or a V_SYS pointer)
 */
extern int		__stdcall VSI_WhoAmI
							P((void *ptr,
							char *clientid,
							char *hostname,
							char *msgbuf));

/*------------------------------------------------------------------------
 * context functions
 */
extern V_CTX *		__stdcall VSI_Context_Load
							P((int argc,
							char **argv,
							const char *pgmname,
							char *msgbuf));

extern V_CTX *		__stdcall VSI_Context_Load_Cmdline
							P((const char *cmdline,
							const char *pgmname,
							char *msgbuf));

extern int		__stdcall VSI_Context_Free
							P((V_CTX *vctx,
							char *msgbuf));

extern int		__stdcall VSI_Context_Update
							P((V_CTX *vctx,
							const char *clientid,
							const char *passwd,
							const char *hostname,
							char *msgbuf));

/*------------------------------------------------------------------------
 * server functions
 */
extern V_TAG *		__stdcall VSI_Server_Query
							P((int timeout,
							const char *netmask,
							char *msgbuf));

extern int		__stdcall VSI_Server_Get_Last_Error
							P((V_SYS *vptr,
							char *msgbuf));

extern V_SYS *		__stdcall VSI_Server_Login_By_Context
							P((V_CTX *vctx,
							char *msgbuf));

extern V_SYS *		__stdcall VSI_Server_Login
							P((const char *clientid,
							const char *passwd,
							const char *hostname,
							const char *pgmname,
							char *msgbuf));

extern int		__stdcall VSI_Server_Logout
							P((V_SYS *vptr,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Server_Info
							P((V_SYS *vptr,
							char *msgbuf));

extern int		__stdcall VSI_Server_Submit
							P((V_SYS *vptr,
							char *reqid_buf,
							const V_TAG *tags,
							char *msgbuf));

extern int		__cdecl   VSI_Server_Submit_Tags
							P((V_SYS *vptr,
							char *reqid_buf,
							...));
#if 0		/* document actual args */
							P((V_SYS *vptr,
							char *reqid_buf,
							const char *tagname,
							const char *tagdata,
							...
							0,
							char *msgbuf));
#endif

extern int		__stdcall VSI_Server_Image
							P((V_SYS *vptr,
							const char *filename,
							const V_TAG *tags,
							char *msgbuf));

extern int		__stdcall VSI_Server_Modify
							P((V_SYS *vptr,
							const char *reqid,
							const V_TAG *tags,
							char *msgbuf));

extern int		__stdcall VSI_Server_Cancel
							P((V_SYS *vptr,
							const char *reqid,
							char *msgbuf));

extern int     __stdcall VSI_Server_Cancel_Group
						P((V_SYS *vptr,
						const char *reqid,
						char *msgbuf));

extern V_TAG *		__stdcall VSI_Server_Status
							P((V_SYS *vptr,
							const char *reqid,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Server_Olog_Get
							P((V_SYS *vptr,
							char *reqid_buf,
							int grp,
							int dbcmd,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_Server_Olog_List
							P((V_SYS *vptr,
							char *reqid_buf,
							int grp,
							int dbcmd,
							V_TAG *tags,
							int detail,
							int all,
							int num,
							char *msgbuf));

extern int		__stdcall VSI_Server_Olog_Delete
							P((V_SYS *vptr,
							const char *reqid,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Server_Ilog_Get
							P((V_SYS *vptr,
							char *reqid_buf,
							int dbcmd,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_Server_Ilog_List
							P((V_SYS *vptr,
							char *reqid_buf,
							int dbcmd,
							V_TAG *tags,
							int num,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_Server_File_List
							P((V_SYS *vptr,
							const char *clientid,
							char *msgbuf));

extern int		__stdcall VSI_Server_File_Retrieve
							P((V_SYS *vptr,
							const char *clientid,
							const char *filename,
							const char *path,
							int archive,
							char *msgbuf));

extern int		__stdcall VSI_Server_File_Delete
							P((V_SYS *vptr,
							const char *clientid,
							const char *filename,
							char *msgbuf));

extern int		__stdcall VSI_Server_File_Route
							P((V_SYS *vptr,
							const char *clientid,
							const char *filename,
							const char *to_id,
							const char *comment,
							const char *costcode,
							int forward,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Server_Resource
							P((V_SYS *vptr,
							const char *res_type,
							char *msgbuf));

extern time_t		__stdcall VSI_Server_Newmsg
							P((V_SYS *vptr,
							const char *clientid,
							int msgtype,
							time_t since,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_Server_Newmsg_Get
							P((V_SYS *vptr,
							const char *clientid,
							int msgtype,
							time_t since,
							char *msgbuf));

extern int		__stdcall VSI_Server_Delegate_Set
							P((V_SYS *vptr,
							const char *clientid,
							int addmask,
							int delmask,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Server_Delegate_List
							P((V_SYS *vptr,
							int mask,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_Server_User_List
							P((V_SYS *vptr,
							int all,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_Server_User_Delegate_List
							P((V_SYS *vptr,
							char *msgbuf));

/*------------------------------------------------------------------------
 * user profile functions
 */
extern V_TAG *		__stdcall VSI_Server_Profile_Get
							P((V_SYS *vptr,
							char *msgbuf));

extern int		__stdcall VSI_Server_Profile_Set
							P((V_SYS *vptr,
							const V_TAG *tags,
							char *msgbuf));

/*------------------------------------------------------------------------
 * phone-book person functions
 */
extern V_TAG *		__stdcall VSI_PB_Person_Get
							P((V_SYS *vptr,
							const char *pbtype,
							char *key,
							const char *order,
							int dbcmd,
							char *msgbuf));

extern int		__stdcall VSI_PB_Person_Add
							P((V_SYS *vptr,
							const char *pbtype,
							char *palias_buf,
							const V_TAG *tags,
							char *msgbuf));

extern int		__stdcall VSI_PB_Person_Upd
							P((V_SYS *vptr,
							const char *pbtype,
							const char *palias,
							const V_TAG *tags,
							char *msgbuf));

extern int		__stdcall VSI_PB_Person_Del
							P((V_SYS *vptr,
							const char *pbtype,
							const char *palias,
							char *msgbuf));

/*------------------------------------------------------------------------
 * phone-book group functions
 */
extern V_TAG *		__stdcall VSI_PB_Group_Get
							P((V_SYS *vptr,
							const char *pbtype,
							char *galias,
							int dbcmd,
							char *msgbuf));

extern int		__stdcall VSI_PB_Group_Add
							P((V_SYS *vptr,
							const char *pbtype,
							const V_TAG *tags,
							char *msgbuf));

extern int		__stdcall VSI_PB_Group_Upd
							P((V_SYS *vptr,
							const char *pbtype,
							const V_TAG *tags,
							char *msgbuf));

extern int		__stdcall VSI_PB_Group_Del
							P((V_SYS *vptr,
							const char *pbtype,
							const char *galias,
							char *msgbuf));

/*------------------------------------------------------------------------
 * phone-book member functions
 */
extern V_TAG *		__stdcall VSI_PB_Member_Get
							P((V_SYS *vptr,
							const char *pbtype,
							int dbcmd,
							char *msgbuf));

extern int		__stdcall VSI_PB_Member_Add
							P((V_SYS *vptr,
							const char *pbtype,
							const char *palias,
							const char *galias,
							char *msgbuf));

extern int		__stdcall VSI_PB_Member_Del
							P((V_SYS *vptr,
							const char *pbtype,
							const char *palias,
							const char *galias,
							char *msgbuf));

/*------------------------------------------------------------------------
 * phone-book list functions
 */
extern V_LIST *		__stdcall VSI_PB_Person_List
							P((V_SYS *vptr,
							const char *pbtype,
							const char *order,
							const char *key,
							const char *galias,
							int dbcmd,
							int num,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_PB_Person_List_By_Tags
							P((V_SYS *vptr,
							const char *pbtype,
							const char *order,
							const V_TAG *tags,
							int dbcmd,
							int num,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_PB_Group_List
							P((V_SYS *vptr,
							const char *pbtype,
							const char *galias,
							int dbcmd,
							int num,
							char *msgbuf));

extern V_LIST *		__stdcall VSI_PB_Group_List_By_Member
							P((V_SYS *vptr,
							const char *pbtype,
							const char *palias,
							int dbcmd,
							int num,
							char *msgbuf));

/*------------------------------------------------------------------------
 * phone-book extended list functions
 */
extern V_LIST *		__stdcall VSI_PB_Person_List_Extended
							P((V_SYS *vptr,
							const char *pbtype,
							const char *order,
							const char *beg,
							const char *end,
							const char *galias,
							int offset,
							int num,
							char *msgbuf));

extern V_TAG *		__stdcall VSI_PB_Types_List
							P((V_SYS *vptr,
							char *msgbuf));

/*------------------------------------------------------------------------
 * phone-book utilities
 */
extern V_SYS *      __stdcall VSI_PB_Local_Login
							P((const char *clientid,
							const char *pgmname,
							char *msgbuf));

/*========================================================================
 * tag lists
 *
 * tag type (in comment) is as follows:
 *
 *	S	arbitrary string
 *	N	resource name
 *	F	Filename
 *	E	Enum (one of a string list)
 *	W	Word (numeric)
 *	B	Bool
 *	T	Time
 *========================================================================*/

/*========================================================================
 * envelope tags
 */
#define V_TAG_ENV_TBLNAME	"env"		/* env tag table name	*/

/*------------------------------------------------------------------------
 * destination tags
 */
#define V_TAG_ENV_TONAME	"tnm"		/* S To name		*/
#define V_TAG_ENV_TOCOMP	"tco"		/* S To company		*/
#define V_TAG_ENV_TOFNUM	"tfn"		/* S To fax-number	*/
#define V_TAG_ENV_TOVNUM	"tvn"		/* S To voice-number	*/
#define V_TAG_ENV_TOINFO	"tin"		/* S To info		*/
#define V_TAG_ENV_TOMAIL	"tem"		/* S To email-address	*/
#define V_TAG_ENV_TOMFMT	"tft"		/* E To mail format	*/
#define V_TAG_ENV_TOSUBJ	"tsb"		/* S To subject		*/
#define V_TAG_ENV_TOCVR		"tcv"		/* S To cover file	*/

/*------------------------------------------------------------------------
 * icc tags
 */
#define V_TAG_ENV_ICC		"icc"		/* N Internal CC w/cvr	*/
#define V_TAG_ENV_NCC		"ncc"		/* N Internal CC wo/cvr	*/

/*------------------------------------------------------------------------
 * cover tags
 */
#define V_TAG_ENV_COVERPAGE	"cvr"		/* N Coverpage name	*/
#define V_TAG_ENV_COVERFILE	"cvf"		/* F Coverpage file	*/
#define V_TAG_ENV_FRNAME	"fnm"		/* S From name		*/
#define V_TAG_ENV_FRCOMP	"fco"		/* S From company	*/
#define V_TAG_ENV_FRFNUM	"ffn"		/* S From fax-number	*/
#define V_TAG_ENV_FRVNUM	"fvn"		/* S From voice-number	*/
#define V_TAG_ENV_ADDR1		"fa1"		/* S From addr-line-1	*/
#define V_TAG_ENV_ADDR2		"fa2"		/* S From addr-line-2	*/
#define V_TAG_ENV_ADDR3		"fa3"		/* S From addr-line-3	*/
#define V_TAG_ENV_COUNTRY	"fcn"		/* S From country	*/
#define V_TAG_ENV_EMAIL		"fem"		/* S From email-address	*/
#define V_TAG_ENV_ORUI		"oui"		/* B Override user-info	*/
#define V_TAG_ENV_SUBJECT	"sub"		/* S Subject		*/
#define V_TAG_ENV_NOTEFILE	"ntf"		/* F Notefile		*/
#define V_TAG_ENV_SIGFILE	"sig"		/* F Signature file	*/
#define V_TAG_ENV_COMPNAME	"cmp"		/* N Company name	*/
#define V_TAG_ENV_APPREF	"arf"		/* S App reference	*/
#define V_TAG_ENV_COSTCODE	"cst"		/* S Cost code		*/

#define V_TAG_ENV_TAG1		"tg1"		/* S User tag 1		*/
#define V_TAG_ENV_TAG2		"tg2"		/* S User tag 2		*/
#define V_TAG_ENV_TAG3		"tg3"		/* S User tag 3		*/
#define V_TAG_ENV_TAG4		"tg4"		/* S User tag 4		*/

/*------------------------------------------------------------------------
 * send parameter tags
 */
#define V_TAG_ENV_CLIENTID	"cli"		/* N ClientID		*/
#define V_TAG_ENV_SENDTIME	"stm"		/* T Send time		*/
#define V_TAG_ENV_EXPMINS	"exp"		/* W Expire minutes	*/
#define V_TAG_ENV_NOTIFY	"not"		/* E Notify mode	*/
#define V_TAG_ENV_ARCHIVE	"arc"		/* B Archive fax	*/
#define V_TAG_ENV_RETRY		"ret"		/* N Retry method	*/
#define V_TAG_ENV_PRIORITY	"pri"		/* N Priority (l,m,h,u)	*/
#define V_TAG_ENV_QUEUE		"que"		/* N Queue		*/
#define V_TAG_ENV_GROUPNAME	"grp"		/* N Group name		*/
#define V_TAG_ENV_GROUPFILE	"grf"		/* F Group file		*/
#define V_TAG_ENV_MAILADDR	"mad"		/* S Email-address	*/
#define V_TAG_ENV_TNP		"tnp"		/* N TNP name		*/
#define V_TAG_ENV_GNP		"gnp"		/* N GNP name		*/
#define V_TAG_ENV_LNP		"lnp"		/* N LNP name		*/
#define V_TAG_ENV_SENDRES	"res"		/* N Send resolution	*/
#define V_TAG_ENV_TMPLATE	"tpl"		/* N template to use	*/
#define V_TAG_ENV_SKELETON	"skl"		/* S Sendmail skeleton	*/
#define V_TAG_ENV_DEFPGL	"pgl"		/* N Default page len	*/
#define V_TAG_ENV_DEFLND	"lnd"		/* B Landscape		*/
#define V_TAG_ENV_TTI		"tti"		/* S TTI string		*/
#define V_TAG_ENV_TSI		"tsi"		/* S TSI string		*/
#define V_TAG_ENV_PAGEHDR	"pgh"		/* S Page header	*/
#define V_TAG_ENV_DATEFMT	"dtf"		/* S Date format	*/
#define V_TAG_ENV_TIMEFMT	"tmf"		/* S Time format	*/
#define V_TAG_ENV_AREACODE	"aco"		/* S Areacode		*/
#define V_TAG_ENV_PREFIX	"pre"		/* S Dialstring prefix	*/
#define V_TAG_ENV_SUFFIX	"suf"		/* S Dialstring suffix	*/
#define V_TAG_ENV_SENDERID	"sid"		/* S SenderID		*/
#define V_TAG_ENV_GDR		"gdr"		/* N GDR name		*/
#define V_TAG_ENV_CVTTYPE	"cvt"		/* N Convert type	*/
#define V_TAG_ENV_EJID		"eji"		/* S External job id	*/
#define V_TAG_ENV_LCR		"lcr"		/* B Allow LCR		*/
#define V_TAG_ENV_ACB		"acb"		/* B Allow call batch	*/
#define V_TAG_ENV_ALB		"alb"		/* B Allow load balance	*/
#define V_TAG_ENV_MAILFMT	"mft"		/* E Mail format	*/

/*------------------------------------------------------------------------
 * These overlay files are applied to all attachment pages
 */
#define V_TAG_ENV_FORMSRV	"ovs"		/* N Ovl file (server)	*/
#define V_TAG_ENV_FORMLCL	"ovl"		/* F Ovl file (local)	*/

/*------------------------------------------------------------------------
 * file tags
 */
#define V_TAG_ENV_FLDRSRV	"fds"		/* N Folder   (server)	*/
#define V_TAG_ENV_FLDRLCL	"fdl"		/* N Folder   (local)	*/

#define V_TAG_ENV_FILESRV	"fls"		/* N Att file (server)	*/
#define V_TAG_ENV_FILELCL	"fll"		/* F Att file (local)	*/
#define V_TAG_ENV_FILESPU   "fsp"		/* F Att pickup file (server) */

#define V_TAG_ENV_OVLSRV	"fvs"		/* N Ovl file (server)	*/
#define V_TAG_ENV_OVLLCL	"fvl"		/* F Ovl file (local)	*/

#define V_TAG_ENV_FSGSRV	"fgs"		/* N Sig file (server)	*/
#define V_TAG_ENV_FSGLCL	"fsg"		/* F Sig file (local)	*/

#define V_TAG_ENV_ORIGPATH	"ofn"		/* S File orig name	*/
#define V_TAG_ENV_FILETYPE	"ftp"		/* N File type		*/
#define V_TAG_ENV_FILERES	"frs"		/* N File resolution	*/
#define V_TAG_ENV_PGLEN		"fpl"		/* N File page len	*/
#define V_TAG_ENV_PGLAND	"fln"		/* B File landscape	*/
#define V_TAG_ENV_PGRANGE	"fpg"		/* S File page range	*/
#define V_TAG_ENV_FILEOPTS	"fcv"		/* S File conv opts	*/

/*------------------------------------------------------------------------
 * dest-specific file tags
 */
#define V_TAG_ENV_DEST_FILELCL	"dll"		/* F Att file (local)	*/

#define V_TAG_ENV_DEST_FSGSRV	"dgs"		/* N Ovl file (server)	*/
#define V_TAG_ENV_DEST_FSGLCL	"dsg"		/* F Ovl file (local)	*/

#define V_TAG_ENV_DEST_OVLSRV	"dvs"		/* N Sig file (server)	*/
#define V_TAG_ENV_DEST_OVLLCL	"dvl"		/* F Sig file (local)	*/

#define V_TAG_ENV_DEST_ORIGPATH	"dfn"		/* S File orig name	*/
#define V_TAG_ENV_DEST_FILETYPE	"dtp"		/* N File type		*/
#define V_TAG_ENV_DEST_FILERES	"drs"		/* N File resolution	*/
#define V_TAG_ENV_DEST_PGLEN	"dpl"		/* N File page len	*/
#define V_TAG_ENV_DEST_PGLAND	"dln"		/* B File landscape	*/
#define V_TAG_ENV_DEST_PGRANGE	"dpg"		/* S File page range	*/
#define V_TAG_ENV_DEST_FILEOPTS	"dcv"		/* S File conv opts	*/

/*------------------------------------------------------------------------
 * user info tags
 */
#define V_TAG_ENV_UI_COMPANY	"ucp"		/* S Company name	*/
#define V_TAG_ENV_UI_FPHONE	"ufp"		/* S Company fax-num	*/
#define V_TAG_ENV_UI_VPHONE	"uvp"		/* S Company voice-num	*/
#define V_TAG_ENV_UI_ADDR1	"ua1"		/* S Company addr-1	*/
#define V_TAG_ENV_UI_ADDR2	"ua2"		/* S Company addr-2	*/
#define V_TAG_ENV_UI_ADDR3	"ua3"		/* S Company addr-3	*/
#define V_TAG_ENV_UI_COUNTRY	"uct"		/* S Company country	*/
#define V_TAG_ENV_UI_EMAIL	"uem"		/* S Company email-addr	*/

/*------------------------------------------------------------------------
 * These tags get expanded to one or more dest lists
 */
#define V_TAG_ENV_PALIAS	"pal"		/* S Person alias	*/
#define V_TAG_ENV_GALIAS	"gal"		/* S Group alias	*/

/*------------------------------------------------------------------------
 * pseudo tags not actually in envelope
 */
#define V_TAG_ENV_TAGFILE	"tgf"		/* F Tag file		*/
#define V_TAG_ENV_USEDEF	"udf"		/* B Use user defaults	*/

/*========================================================================
 * olog tag names
 */
#define V_TAG_OLOG_TBLNAME	"olog"		/* olog tag table name	*/

#define V_TAG_OLOG_SEQNO	"seq"		/* W Reg sequence num	*/
#define V_TAG_OLOG_CLIENTID	"cli"		/* N ClientID		*/
#define V_TAG_OLOG_REQTYPE	"typ"		/* E Request type	*/
#define V_TAG_OLOG_GSEQNO	"gse"		/* W Grp sequence num	*/
#define V_TAG_OLOG_MEMNO	"mem"		/* W Member number	*/
#define V_TAG_OLOG_SESSIONID	"ses"		/* W SessionID		*/
#define V_TAG_OLOG_REFERENCE	"ref"		/* W Cmd ref number	*/
#define V_TAG_OLOG_SENDERID	"sid"		/* S SenderID		*/
#define V_TAG_OLOG_GDR		"gdr"		/* N GDR name		*/
#define V_TAG_OLOG_CVTTYPE	"cvt"		/* N Convert type	*/
#define V_TAG_OLOG_HOSTNAME	"hst"		/* S Server hostname	*/
#define V_TAG_OLOG_EJID		"eji"		/* S Ext JobID		*/

#define V_TAG_OLOG_FWDHOST	"thn"		/* S Fwd to   hostname	*/
#define V_TAG_OLOG_FWDSEQ	"tsq"		/* W Fwd to   seq num	*/
#define V_TAG_OLOG_FRMHOST	"fhn"		/* S Fwd from hostname	*/
#define V_TAG_OLOG_FRMSEQ	"fsq"		/* W Fwd from seq num	*/

#define V_TAG_OLOG_LCR		"lcr"		/* B Allow LCR		*/
#define V_TAG_OLOG_ACB		"acb"		/* B Allow call batch	*/
#define V_TAG_OLOG_ALB		"alb"		/* B Allow load balance	*/

#define V_TAG_OLOG_ATTEMPT	"att"		/* W Attempt number	*/
#define V_TAG_OLOG_NUMATTS	"nat"		/* W Number of attempts	*/
#define V_TAG_OLOG_CURATT	"cat"		/* W Cur attempt num	*/
#define V_TAG_OLOG_ACTIVE	"act"		/* E Active mode	*/
#define V_TAG_OLOG_DONE		"don"		/* B Done		*/
#define V_TAG_OLOG_REPORT	"rpt"		/* B Report flag	*/
#define V_TAG_OLOG_CATEGORY	"ctg"		/* W Category		*/

#define V_TAG_OLOG_NUMMEMS	"nmm"		/* W Num members	*/
#define V_TAG_OLOG_NUMQUEUED	"nmq"		/* W Num members queued	*/
#define V_TAG_OLOG_NUMSENT	"nms"		/* W Num members sent	*/
#define V_TAG_OLOG_NUMFAILED	"nmf"		/* W Num members failed	*/
#define V_TAG_OLOG_NEXTMEM	"nxt"		/* W Next mem to send	*/

#define V_TAG_OLOG_NUMPGS	"npg"		/* W Num pages		*/
#define V_TAG_OLOG_NPGATT	"npa"		/* W Num pages this att	*/
#define V_TAG_OLOG_NPAGES	"nps"		/* W Num pages sent	*/
#define V_TAG_OLOG_NAKS		"nak"		/* W Num pages bad	*/
#define V_TAG_OLOG_NFILES	"nfl"		/* W Num files		*/

#define V_TAG_OLOG_ARESULT	"are"		/* W Attempt result	*/
#define V_TAG_OLOG_RRESULT	"rre"		/* W Job result		*/

#define V_TAG_OLOG_ARESSTR	"ars"		/* N Attempt result	*/
#define V_TAG_OLOG_RRESSTR	"rrs"		/* N Job result		*/

#define V_TAG_OLOG_RETRY	"ret"		/* N Retry method	*/
#define V_TAG_OLOG_NOTIFY	"not"		/* E Notify mode	*/
#define V_TAG_OLOG_PRILEVEL	"prl"		/* W Priority (0-100)	*/
#define V_TAG_OLOG_PRIORITY	"pri"		/* N Priority (l,m,h,u)	*/

#define V_TAG_OLOG_STATTIME	"sat"		/* T Status time	*/
#define V_TAG_OLOG_SUBMITTIME	"sbt"		/* T Submit time	*/
#define V_TAG_OLOG_SENDTIME	"stm"		/* T Send   time	*/
#define V_TAG_OLOG_STIME	"sti"		/* T Start  time	*/
#define V_TAG_OLOG_ETIME	"eti"		/* T End    time	*/
#define V_TAG_OLOG_ELAPSED	"ela"		/* W Elapsed secs	*/
#define V_TAG_OLOG_EXPMINS	"exp"		/* W Expire minutes	*/

#define V_TAG_OLOG_TONAME	"tnm"		/* S To name		*/
#define V_TAG_OLOG_TOCOMP	"tco"		/* S To company		*/
#define V_TAG_OLOG_TOFNUM	"tfn"		/* S To fax-number	*/
#define V_TAG_OLOG_TOVNUM	"tvn"		/* S To voice-number	*/
#define V_TAG_OLOG_TOINFO	"tin"		/* S To info		*/
#define V_TAG_OLOG_TOMAIL	"tem"		/* S To email-address	*/
#define V_TAG_OLOG_TOMFMT	"tft"		/* E To mail format	*/

#define V_TAG_OLOG_FRNAME	"fnm"		/* S From name		*/
#define V_TAG_OLOG_FRCOMP	"fco"		/* S From company	*/
#define V_TAG_OLOG_FRFNUM	"ffn"		/* S From fax-number	*/
#define V_TAG_OLOG_FRVNUM	"fvn"		/* S From voice-number	*/
#define V_TAG_OLOG_ADDR1	"fa1"		/* S From addr-line-1	*/
#define V_TAG_OLOG_ADDR2	"fa2"		/* S From addr-line-2	*/
#define V_TAG_OLOG_ADDR3	"fa3"		/* S From addr-line-3	*/
#define V_TAG_OLOG_COUNTRY	"fcn"		/* S From country	*/
#define V_TAG_OLOG_EMAIL	"fem"		/* S From email-addr	*/
#define V_TAG_OLOG_ORUI		"oui"		/* B Override user-info	*/

#define V_TAG_OLOG_USRTAG1	"tg1"		/* S User tag 1		*/
#define V_TAG_OLOG_USRTAG2	"tg2"		/* S User tag 2		*/
#define V_TAG_OLOG_USRTAG3	"tg3"		/* S User tag 3		*/
#define V_TAG_OLOG_USRTAG4	"tg4"		/* S User tag 4		*/

#define V_TAG_OLOG_SUBJECT	"sub"		/* S Subject		*/
#define V_TAG_OLOG_COVERPAGE	"cvr"		/* N Coverpage name	*/
#define V_TAG_OLOG_COMPNAME	"cmp"		/* N Company name	*/
#define V_TAG_OLOG_APPREF	"arf"		/* S App reference	*/
#define V_TAG_OLOG_COSTCODE	"cst"		/* S Cost code		*/

#define V_TAG_OLOG_QUEUE	"que"		/* N Requested queue	*/
#define V_TAG_OLOG_ASGQUE	"asq"		/* N Assigned queue	*/
#define V_TAG_OLOG_ACTQUE	"atq"		/* N Actual queue	*/
#define V_TAG_OLOG_SENDRES	"res"		/* N Send resolution	*/
#define V_TAG_OLOG_TMPLATE	"tpl"		/* N template to use	*/
#define V_TAG_OLOG_SKELETON	"skl"		/* S Sendmail skeleton	*/
#define V_TAG_OLOG_DEFPGL	"pgl"		/* N Default page len	*/
#define V_TAG_OLOG_DEFLND	"lnd"		/* B Landscape		*/
#define V_TAG_OLOG_DIALED	"dia"		/* S Dialed number	*/
#define V_TAG_OLOG_PREFIX	"pre"		/* S Dialstring prefix	*/
#define V_TAG_OLOG_SUFFIX	"suf"		/* S Dialstring suffix	*/
#define V_TAG_OLOG_AREACODE	"aco"		/* S Areacode		*/

#define V_TAG_OLOG_TNP		"tnp"		/* N TNP name		*/
#define V_TAG_OLOG_GNP		"gnp"		/* N GNP name		*/
#define V_TAG_OLOG_LNP		"lnp"		/* N LNP name		*/
#define V_TAG_OLOG_TTI		"tti"		/* S TTI string		*/
#define V_TAG_OLOG_TSI		"tsi"		/* S TSI string		*/
#define V_TAG_OLOG_CSI		"csi"		/* S CSI called		*/
#define V_TAG_OLOG_PAGEHDR	"pgh"		/* S Page header	*/
#define V_TAG_OLOG_DATEFMT	"dtf"		/* S Date format	*/
#define V_TAG_OLOG_TIMEFMT	"tmf"		/* S Time format	*/
#define V_TAG_OLOG_MAILADDR	"mad"		/* S Email-address	*/

#define V_TAG_OLOG_ERRMSG	"err"		/* S Error message	*/
#define V_TAG_OLOG_SNDARCH	"sar"		/* B Fax archived	*/

/*========================================================================
 * faxhdr tag names
 */
#define V_TAG_HDR_TBLNAME	"hdr"		/* faxhdr tag table name */

#define V_TAG_HDR_SEQNO		"seq"		/* W Recv sequence num	*/
#define V_TAG_HDR_OSEQNO	"osq"		/* W Orig sequence num	*/
#define V_TAG_HDR_FILENAME	"fil"		/* S Filename		*/
#define V_TAG_HDR_CLIENTID	"cli"		/* N ClientID		*/
#define V_TAG_HDR_TSI		"tsi"		/* S TSI of sender	*/
#define V_TAG_HDR_STIME		"sti"		/* T Start time		*/
#define V_TAG_HDR_ETIME		"eti"		/* T End   time		*/
#define V_TAG_HDR_ELAPSED	"ela"		/* W Elapsed secs	*/
#define V_TAG_HDR_NUMPGS	"npg"		/* W Number of pages	*/
#define V_TAG_HDR_DIDEXT	"did"		/* W DID extension	*/
#define V_TAG_HDR_RES		"res"		/* N Resolution		*/
#define V_TAG_HDR_PGL		"pgl"		/* N Page length	*/
#define V_TAG_HDR_PNL		"pnl"		/* W Page lines		*/
#define V_TAG_HDR_QUEUE		"que"		/* N Queue recvd on	*/
#define V_TAG_HDR_RCVTIME	"vtt"		/* T Recv time		*/
#define V_TAG_HDR_RCVFAXEMA	"vfe"		/* S Recv fax email	*/
#define V_TAG_HDR_RCVNTYEMA	"vne"		/* S Recv notify email	*/
#define V_TAG_HDR_RTEFROM	"rtf"		/* N Routed from	*/
#define V_TAG_HDR_RTEBY		"rtb"		/* N Routed by		*/
#define V_TAG_HDR_RTETIME	"rtt"		/* T Route time		*/
#define V_TAG_HDR_RTEFAXEMA	"rfe"		/* S Route fax email	*/
#define V_TAG_HDR_RTENTYEMA	"rne"		/* S Route notify email	*/
#define V_TAG_HDR_COSTCODE	"cst"		/* S Cost code		*/
#define V_TAG_HDR_CALLERID	"cid"		/* S Caller ID		*/
#define V_TAG_HDR_COMMENT	"cmt"		/* S Comment		*/
#define V_TAG_HDR_FSOURCE	"fss"		/* E File source	*/
#define V_TAG_HDR_FSTATUS	"fst"		/* E File status	*/
#define V_TAG_HDR_RSTATUS	"rst"		/* W Recv status	*/
#define V_TAG_HDR_FILESIZE	"fsz"		/* W File size		*/
#define V_TAG_HDR_RCVARCH	"rar"		/* B File archived	*/
#define V_TAG_HDR_DELETED	"del"		/* B File deleted	*/
#define V_TAG_HDR_DELETER	"der"		/* N Deleted by		*/
#define V_TAG_HDR_DELTIME	"dtm"		/* T Time deleted	*/
#define V_TAG_HDR_DELWHY	"dwy"		/* E Why deleted	*/

/*========================================================================
 * ilog tag names
 */
#define V_TAG_ILOG_TBLNAME	"ilog"		/* ilog tag table name */

#define V_TAG_ILOG_SEQNO	V_TAG_HDR_SEQNO
#define V_TAG_ILOG_OSEQNO	V_TAG_HDR_OSEQNO
#define V_TAG_ILOG_FILENAME	V_TAG_HDR_FILENAME
#define V_TAG_ILOG_CLIENTID	V_TAG_HDR_CLIENTID
#define V_TAG_ILOG_TSI		V_TAG_HDR_TSI
#define V_TAG_ILOG_STIME	V_TAG_HDR_STIME
#define V_TAG_ILOG_ETIME	V_TAG_HDR_ETIME
#define V_TAG_ILOG_ELAPSED	V_TAG_HDR_ELAPSED
#define V_TAG_ILOG_NUMPGS	V_TAG_HDR_NUMPGS
#define V_TAG_ILOG_DIDEXT	V_TAG_HDR_DIDEXT
#define V_TAG_ILOG_RES		V_TAG_HDR_RES
#define V_TAG_ILOG_PGL		V_TAG_HDR_PGL
#define V_TAG_ILOG_PNL		V_TAG_HDR_PNL
#define V_TAG_ILOG_QUEUE	V_TAG_HDR_QUEUE
#define V_TAG_ILOG_RCVTIME	V_TAG_HDR_RCVTIME
#define V_TAG_ILOG_RCVFAXEMA	V_TAG_HDR_RCVFAXEMA
#define V_TAG_ILOG_RCVNTYEMA	V_TAG_HDR_RCVNTYEMA
#define V_TAG_ILOG_RTEFROM	V_TAG_HDR_RTEFROM
#define V_TAG_ILOG_RTEBY	V_TAG_HDR_RTEBY
#define V_TAG_ILOG_RTETIME	V_TAG_HDR_RTETIME
#define V_TAG_ILOG_RTEFAXEMA	V_TAG_HDR_RTEFAXEMA
#define V_TAG_ILOG_RTENTYEMA	V_TAG_HDR_RTENTYEMA
#define V_TAG_ILOG_COSTCODE	V_TAG_HDR_COSTCODE
#define V_TAG_ILOG_CALLERID	V_TAG_HDR_CALLERID
#define V_TAG_ILOG_COMMENT	V_TAG_HDR_COMMENT
#define V_TAG_ILOG_FSOURCE	V_TAG_HDR_FSOURCE
#define V_TAG_ILOG_FSTATUS	V_TAG_HDR_FSTATUS
#define V_TAG_ILOG_RSTATUS	V_TAG_HDR_RSTATUS
#define V_TAG_ILOG_FILESIZE	V_TAG_HDR_FILESIZE
#define V_TAG_ILOG_RCVARCH	V_TAG_HDR_RCVARCH
#define V_TAG_ILOG_DELETED	V_TAG_HDR_DELETED
#define V_TAG_ILOG_DELETER	V_TAG_HDR_DELETER
#define V_TAG_ILOG_DELTIME	V_TAG_HDR_DELTIME
#define V_TAG_ILOG_DELWHY	V_TAG_HDR_DELWHY

/*========================================================================
 * Server-info field tags
 */
#define V_TAG_SRVI_TBLNAME	"srvi"		/* srvi tag table name	*/

#define V_TAG_SRVI_HOSTNAME	"hst"		/* S Server hostname	*/
#define V_TAG_SRVI_STARTUP	"stu"		/* T Server start time	*/
#define V_TAG_SRVI_SCHEDTIME	"sct"		/* T Server curr  time	*/
#define V_TAG_SRVI_STATE	"ste"		/* W Server state	*/

#define V_TAG_SRVI_VERSION	"ver"		/* S Server version	*/
#define V_TAG_SRVI_BUILD	"bld"		/* S Server build	*/
#define V_TAG_SRVI_BLDINFO	"bif"		/* S Server build info	*/
#define V_TAG_SRVI_RUNINFO	"rif"		/* S Server run   info	*/

#define V_TAG_SRVI_SERIAL	"ser"		/* S Server serial num	*/
#define V_TAG_SRVI_LICMODE	"lcm"		/* W License mode	*/
#define V_TAG_SRVI_NUMUSERS	"nmu"		/* W Num client lics	*/
#define V_TAG_SRVI_NUMPORTS	"nmp"		/* W Num modem  lics	*/
#define V_TAG_SRVI_SRVEXPDT	"sxt"		/* T server expire date	*/

#define V_TAG_SRVI_TTFL		"ttl"		/* W Time to fax (low)	*/
#define V_TAG_SRVI_TTFM		"ttm"		/* W Time to fax (med)	*/
#define V_TAG_SRVI_TTFH		"tth"		/* W Time to fax (high)	*/
#define V_TAG_SRVI_TTFU		"ttu"		/* W Time to fax (urg)	*/

#define V_TAG_SRVI_GMTOFFSET	"gmt"		/* W Server gmtoffset	*/

/*========================================================================
 * person tag names
 */
#define V_TAG_PER_TBLNAME	"per"		/* person tag table name */

#define V_TAG_PER_PALIAS	"pal"		/* S Person alias	*/
#define V_TAG_PER_LNAME		"las"		/* S Last name		*/
#define V_TAG_PER_FNAME		"fir"		/* S First name		*/
#define V_TAG_PER_COMPANY	"com"		/* S Company		*/
#define V_TAG_PER_NOTES		"not"		/* S Notes		*/

#define V_TAG_PER_FCNTRY	"fcn"		/* S Fax country code	*/
#define V_TAG_PER_FAREA		"far"		/* S Fax area code	*/
#define V_TAG_PER_FLOCAL	"flo"		/* S Fax local num	*/
#define V_TAG_PER_FEXT		"fex"		/* S Fax extension	*/

#define V_TAG_PER_VCNTRY	"vcn"		/* S Voice country code	*/
#define V_TAG_PER_VAREA		"var"		/* S Voice area code	*/
#define V_TAG_PER_VLOCAL	"vlo"		/* S Voice local num	*/
#define V_TAG_PER_VEXT		"vex"		/* S Voice extension	*/

#define V_TAG_PER_PCNTRY	"pcn"		/* S Pager country code	*/
#define V_TAG_PER_PAREA		"par"		/* S Pager area code	*/
#define V_TAG_PER_PLOCAL	"plo"		/* S Pager local num	*/
#define V_TAG_PER_PEXT		"pex"		/* S Pager extension	*/

#define V_TAG_PER_CCNTRY	"ccn"		/* S Cell country code	*/
#define V_TAG_PER_CAREA		"car"		/* S Cell area code	*/
#define V_TAG_PER_CLOCAL	"clo"		/* S Cell local num	*/
#define V_TAG_PER_CEXT		"cex"		/* S Cell extension	*/

#define V_TAG_PER_ADDR1		"ad1"		/* S Address line 1	*/
#define V_TAG_PER_ADDR2		"ad2"		/* S Address line 2	*/
#define V_TAG_PER_ADDR3		"ad3"		/* S Address line 3	*/
#define V_TAG_PER_COUNTRY	"cou"		/* S Country		*/

#define V_TAG_PER_EMAIL		"ema"		/* S Email-address	*/
#define V_TAG_PER_INFO		"inf"		/* S Info		*/

#define V_TAG_PER_USER1		"us1"		/* S User tag 1		*/
#define V_TAG_PER_USER2		"us2"		/* S User tag 2		*/

#define V_TAG_PER_NAME		"nam"		/* S Full name		*/
#define V_TAG_PER_FAX		"fax"		/* S Full fax number	*/
#define V_TAG_PER_VOICE		"voi"		/* S Full voice number	*/
#define V_TAG_PER_PAGER		"pag"		/* S Full pager number	*/
#define V_TAG_PER_CELL		"cel"		/* S Full cell number	*/

#define V_TAG_PER_ACOLEN	"aln"		/* W Areacode len	*/
#define V_TAG_PER_LCLLEN	"lln"		/* W Local num len	*/

/*========================================================================
 * group tag names
 */
#define V_TAG_GRP_TBLNAME	"grp"		/* group tag table name */

#define V_TAG_GRP_GALIAS	"gal"		/* S Group alias	*/
#define V_TAG_GRP_GDESC		"gde"		/* S Group description	*/

/*========================================================================
 * member tag names
 */
#define V_TAG_MEM_TBLNAME	"mem"		/* member tag table name */

#define V_TAG_MEM_PALIAS	"pal"		/* S Person alias	*/
#define V_TAG_MEM_GALIAS	"gal"		/* S Group alias	*/

/*========================================================================
 * USER field tags
 */
#define V_TAG_USER_TBLNAME	"user"		/* user tag table name	*/

/*------------------------------------------------------------------------
 * User ID fields
 */
#define V_TAG_USER_LOGNAME	"osn"		/* S O/S login name	*/
#define V_TAG_USER_CLIENTID	"cli"		/* N ClientID		*/
#define V_TAG_USER_PASSWD	"pwd"		/* S Password		*/
#define V_TAG_USER_USERNAME	"usr"		/* S User name		*/
#define V_TAG_USER_LANGUAGE	"lng"		/* S Language		*/
#define V_TAG_USER_LOCALE	"lcl"		/* S Locale		*/
#define V_TAG_USER_TIMEZONE	"tmz"		/* S Timezone		*/
#define V_TAG_USER_DATEFMT	"dtf"		/* S Date format	*/
#define V_TAG_USER_TIMEFMT	"tmf"		/* S Time format	*/
#define V_TAG_USER_LASTLGN	"lgn"		/* T Last login time	*/
#define V_TAG_USER_OSLOGIN	"osl"		/* B Allow O/S login	*/
#define V_TAG_USER_DIDEXT	"did"		/* S DID extension	*/
#define V_TAG_USER_CALCDID	"cdd"		/* B Calculate DID	*/
#define V_TAG_USER_DIDLEN	"ddl"		/* W DID length		*/
#define V_TAG_USER_LICENSED	"lic"		/* B User is licensed	*/
#define V_TAG_USER_USERTYPE	"utp"		/* E User type		*/
#define V_TAG_USER_USRMODE	"umd"		/* E User mode		*/
#define V_TAG_USER_MAKEPROF	"mkp"		/* B Make a profile	*/
#define V_TAG_USER_ISPROFILE	"isp"		/* B User is a profile	*/
#define V_TAG_USER_PARENTID	"pid"		/* N ParentID		*/

/*------------------------------------------------------------------------
 * profile inherit/edit fields
 */
#define V_TAG_USER_INHERIT	"inh"		/* S Inherit fld list	*/
#define V_TAG_USER_USREDIT	"ued"		/* S Editable fld list	*/
#define V_TAG_USER_INHFLDS	"ifd"		/* S Merged fld list	*/
#define V_TAG_USER_USRFLDS	"ufd"		/* S Updated fld list	*/

/*------------------------------------------------------------------------
 * coverpage fields
 */
#define V_TAG_USER_COVERPAGE	"cvr"		/* N Coverpage name	*/
#define V_TAG_USER_USECVR	"ucv"		/* B Use coverpage	*/
#define V_TAG_USER_FRNAME	"fnm"		/* S From name		*/
#define V_TAG_USER_FRCOMP	"fco"		/* S From company	*/
#define V_TAG_USER_FRFNUM	"ffn"		/* S From fax-number	*/
#define V_TAG_USER_FRVNUM	"fvn"		/* S From voice-number	*/
#define V_TAG_USER_ADDR1	"fa1"		/* S From addr line 1	*/
#define V_TAG_USER_ADDR2	"fa2"		/* S From addr line 2	*/
#define V_TAG_USER_ADDR3	"fa3"		/* S From addr line 3	*/
#define V_TAG_USER_COUNTRY	"fcn"		/* S From country	*/
#define V_TAG_USER_EMAIL	"fem"		/* S From email-address	*/
#define V_TAG_USER_ORUI		"oui"		/* B Override user-info	*/
#define V_TAG_USER_USRTAG1	"tg1"		/* S User tag 1		*/
#define V_TAG_USER_USRTAG2	"tg2"		/* S User tag 2		*/
#define V_TAG_USER_USRTAG3	"tg3"		/* S User tag 3		*/
#define V_TAG_USER_USRTAG4	"tg4"		/* S User tag 4		*/

/*------------------------------------------------------------------------
 * sendparam fields
 */
#define V_TAG_USER_MAXPRI	"mxp"		/* N Max priority	*/
#define V_TAG_USER_PRIORITY	"pri"		/* N Priority		*/
#define V_TAG_USER_NOTIFY	"not"		/* E Notify mode	*/
#define V_TAG_USER_MAILADDR	"mad"		/* S email-address	*/
#define V_TAG_USER_RETRY	"ret"		/* N Retry method	*/
#define V_TAG_USER_QUEUE	"que"		/* N queue		*/
#define V_TAG_USER_SENDRES	"res"		/* N Send resolution	*/
#define V_TAG_USER_TMPLATE	"tpl"		/* N template to use	*/
#define V_TAG_USER_SKELETON	"skl"		/* S Sendmail skeleton	*/
#define V_TAG_USER_DEFPGL	"pgl"		/* N Default page len	*/
#define V_TAG_USER_DEFLND	"lnd"		/* B Landscape		*/
#define V_TAG_USER_TSI		"tsi"		/* S TSI string		*/
#define V_TAG_USER_TTI		"tti"		/* S TTI string		*/
#define V_TAG_USER_COSTCODE	"cst"		/* S Cost code		*/
#define V_TAG_USER_LNP		"lnp"		/* N LNP name		*/
#define V_TAG_USER_TNP		"tnp"		/* N TNP name		*/
#define V_TAG_USER_GNP		"gnp"		/* N GNP name		*/
#define V_TAG_USER_AREACODE	"aco"		/* S Areacode		*/
#define V_TAG_USER_PREFIX	"pre"		/* S Dialstring prefix	*/
#define V_TAG_USER_SUFFIX	"suf"		/* S Dialstring suffix	*/
#define V_TAG_USER_PAGEHDR	"pgh"		/* S Page header	*/
#define V_TAG_USER_USEPGH	"upg"		/* B Use page header	*/
#define V_TAG_USER_LCR		"lcr"		/* B Allow LCR		*/
#define V_TAG_USER_ACB		"acb"		/* B Allow call batch	*/
#define V_TAG_USER_ALB		"alb"		/* B Allow load balance	*/
#define V_TAG_USER_ARCHIVE	"arc"		/* B Archive sent faxes	*/

/*------------------------------------------------------------------------
 * sent actions fields
 */
#define V_TAG_USER_SNDFAX	"sfx"		/* E Sent mail fax mode	*/
#define V_TAG_USER_SNDFAXEMA	"sfe"		/* S Sent email addr	*/
#define V_TAG_USER_SNDFMT	"sfm"		/* E Sent mail format	*/
#define V_TAG_USER_SNDPRTNTY	"spn"		/* E Sent prt nty mode	*/
#define V_TAG_USER_SNDPRTFAX	"spf"		/* E Sent prt fax mode	*/
#define V_TAG_USER_SPRTSTR	"pst"		/* N Sent printer name	*/
#define V_TAG_USER_SNDNTYTPOK	"sto"		/* S Sent OK  template	*/
#define V_TAG_USER_SNDNTYTPNG	"stn"		/* S Sent bad template	*/
#define V_TAG_USER_SNDNTYTPGP	"stg"		/* S Sent grp template	*/
#define V_TAG_USER_SNDSUBTP	"ssp"		/* S Sent sub template	*/
#define V_TAG_USER_SNDARCH	"sar"		/* B Archive sent faxes	*/

/*------------------------------------------------------------------------
 * recv actions fields
 */
#define V_TAG_USER_RCVNTY	"vnt"		/* B Recv email notify	*/
#define V_TAG_USER_RCVNTYEMA	"vne"		/* S Recv email addr	*/
#define V_TAG_USER_RCVNTYTP	"vtp"		/* N Recv nty template	*/
#define V_TAG_USER_RCVSUBTP	"vsp"		/* S Recv sub template	*/
#define V_TAG_USER_RCVFAX	"vfx"		/* B Recv email fax	*/
#define V_TAG_USER_RCVFAXEMA	"vfe"		/* S Recv fax email	*/
#define V_TAG_USER_RCVFMT	"vfm"		/* E Recv mail format	*/
#define V_TAG_USER_RCVPRT	"vpr"		/* B Recv print fax	*/
#define V_TAG_USER_RPRTSTR	"pvt"		/* N Recv printer name	*/
#define V_TAG_USER_RCVARCH	"var"		/* B Archive recv faxes	*/

/*------------------------------------------------------------------------
 * route actions fields
 */
#define V_TAG_USER_RNP		"rnp"		/* N RNP name		*/
#define V_TAG_USER_RTENTY	"rnt"		/* B Route email nty	*/
#define V_TAG_USER_RTENTYEMA	"rne"		/* S Route nty email	*/
#define V_TAG_USER_RTENTYTP	"rtp"		/* N Route nty template	*/
#define V_TAG_USER_RTESUBTP	"rsp"		/* S Route nty sub tpl	*/
#define V_TAG_USER_RTEFAX	"aem"		/* B Route email fax	*/
#define V_TAG_USER_RTEFAXEMA	"rfe"		/* S Route fax email	*/
#define V_TAG_USER_RTEFMT	"rfm"		/* E Route mail format	*/
#define V_TAG_USER_RTEPRT	"apr"		/* B Route print fax	*/
#define V_TAG_USER_PRTSTR	"prt"		/* N Route printer name	*/
#define V_TAG_USER_AUTODEL	"adl"		/* B Auto-delete fax	*/

/*------------------------------------------------------------------------
 * misc client fields
 */
#define V_TAG_USER_CLRONSEND	"cos"		/* B Clear on send	*/
#define V_TAG_USER_DISPSTAT	"dsp"		/* B Display status dlg	*/
#define V_TAG_USER_WEBINSTALL	"wbi"		/* B Allow web install	*/
#define V_TAG_USER_AUTHMASK	"amk"		/* W Default auth mask	*/

/*------------------------------------------------------------------------
 * directory fields
 */
#define V_TAG_USER_PARENTDIRS	"prd"		/* B Use parent dirs	*/
#define V_TAG_USER_DIRECTORIES	"dir"		/* S Directory list	*/

/*========================================================================
 * phone-book types
 */
#define V_PB_TYPE_LCL		"lcl"		/* client local  PB	*/
#define V_PB_TYPE_USR		"usr"		/* client server PB	*/
#define V_PB_TYPE_SYS		"sys"		/* system server PB	*/
#define V_PB_TYPE_EXT		"ext"		/* system extern PB	*/

#define V_PB_TYPE_DFLT		V_PB_TYPE_USR	/* default type		*/

#define V_PB_GROUP_ALL		"all"		/* use all groups	*/

/*========================================================================
 * phone-book sort orders
 */
#define V_PB_SORT_ALIAS		"alias"		/* alias      order	*/
#define V_PB_SORT_LNAME		"lname"		/* last name  order	*/
#define V_PB_SORT_FNAME		"fname"		/* first name order	*/
#define V_PB_SORT_COMPANY	"comp"		/* company    order	*/

#define V_PB_SORT_DFLT		V_PB_SORT_ALIAS	/* default sort order	*/

/*========================================================================
 * database cmds
 */
#define V_DBCMD_NEXT		1		/* read next  entry	*/
#define V_DBCMD_PREV		2		/* read prev  entry	*/
#define V_DBCMD_FIRST		3		/* read first entry	*/
#define V_DBCMD_LAST		4		/* read last  entry	*/
#define V_DBCMD_EQUAL		5		/* read rec   == key	*/
#define V_DBCMD_GTEQ		6		/* read next  >= key	*/

/*========================================================================
 * compare operations
 */
#define V_COMP_EQ		0		/* ==			*/
#define V_COMP_NE		1		/* !=			*/
#define V_COMP_GT		2		/* >			*/
#define V_COMP_LT		3		/* <			*/
#define V_COMP_GE		4		/* >=			*/
#define V_COMP_LE		5		/* <=			*/
#define V_COMP_SS		6		/* substring		*/

/*========================================================================
 * resource names
 */
#define V_RES_ENVTAGS		"envtags"	/* envelope tags	*/

#define V_RES_FILETYPES		"filetypes"	/* file types		*/
#define V_RES_PGLENGTHS		"pglengths"	/* page lengths		*/
#define V_RES_PRIORITIES	"priorities"	/* priorities		*/
#define V_RES_RESOLUTIONS	"resolutions"	/* resolutions		*/
#define V_RES_RETRYS		"retries"	/* retry methods	*/
#define V_RES_COVERS		"covers"	/* coverpages		*/
#define V_RES_QUEUES		"queues"	/* queues		*/
#define V_RES_GROUPS		"groups"	/* server groups	*/
#define V_RES_TNPS		"tnps"		/* user TNP scripts	*/
#define V_RES_GNPS		"gnps"		/* user GNP scripts	*/
#define V_RES_LNPS		"lnps"		/* user LNP scripts	*/
#define V_RES_RNPS		"rnps"		/* user RNP scripts	*/
#define V_RES_FOLDERS		"folders"	/* server folders	*/
#define V_RES_FORMS		"forms"		/* server overlays	*/
#define V_RES_IMAGES		"images"	/* server images	*/
#define V_RES_SRVFILES		"srvfiles"	/* server files		*/
#define V_RES_ROUTES		"routes"	/* users		*/
#define V_RES_FAXFILES		"faxfiles"	/* received faxes	*/
#define V_RES_ARCFILES		"arcfiles"	/* archived faxes	*/
#define V_RES_PRINTERS		"printers"	/* printer names	*/
#define V_RES_TEMPLATES		"templates"	/* template files	*/
#define V_RES_COMPANIES		"companies"	/* company names	*/

#define V_RES_GDRS		"gdrs"		/* GDR names		*/

/*========================================================================
 * olog aresult codes
 *
 * tags:
 *	V_TAG_OLOG_ARESULT
 */
#define	V_ARESULT_NORMAL	 1		/* normal termination	*/
#define	V_ARESULT_CANCEL_IDLE	 2		/* cancelled while idle	*/
#define	V_ARESULT_CANCEL_WORK	 3		/* cancelled while busy	*/
#define	V_ARESULT_BUSY		 4		/* other end busy	*/
#define	V_ARESULT_LINE_DROP	 5		/* line dropped		*/
#define	V_ARESULT_NOT_FAX	 6		/* not a fax		*/
#define	V_ARESULT_NO_ANSWER	 7		/* no answer		*/
#define	V_ARESULT_VOICE		 8		/* voice answered	*/
#define	V_ARESULT_NO_DIAL	 9		/* no dial tone		*/
#define	V_ARESULT_REM_REJ	10		/* remote rejected	*/
#define	V_ARESULT_FIM_FAILURE	11		/* FIM failure		*/
#define	V_ARESULT_BAD_REQ	12		/* bad request file	*/
#define	V_ARESULT_UNDERRUN	15		/* output underrun	*/
#define	V_ARESULT_BAD_MODEM	16		/* problems with modem	*/
#define	V_ARESULT_TIMEOUT	17		/* heard nothing from modem */
#define	V_ARESULT_FIM_DIED	18		/* FIM just died	*/
#define	V_ARESULT_FIM_UNAVAIL	19		/* FIM no longer avail	*/
#define	V_ARESULT_SCHED_ERROR	20		/* scheduler error	*/
#define	V_ARESULT_NO_FILE	21		/* cannot open file	*/
#define	V_ARESULT_BAD_FILE	22		/* bad file format	*/
#define	V_ARESULT_CANCEL_EXPWORK 23		/* force expire	job active*/
#define	V_ARESULT_CANCEL_EXPIDLE 24		/* force expire job idle */
#define V_ARESULT_CANCEL_USRREQ	25		/* user requested termination */
#define V_ARESULT_FORWARD	26		/* job was forwarded	*/
#define V_ARESULT_QUEUED	31		/* queued		*/
#define V_ARESULT_SENDING	32		/* sending		*/
#define V_ARESULT_SUBMITTED	33		/* submitted		*/
#define V_ARESULT_IMGFAIL	34		/* image failure	*/
#define V_ARESULT_LPWAIT	35		/* waiting for LNP	*/
#define V_ARESULT_TPWAIT	36		/* waiting for TNP	*/
#define V_ARESULT_NORSP		37		/* no rsp from fwd host	*/
#define V_ARESULT_NOCARRIER	38		/* no carrier detected	*/
#define V_ARESULT_EXTERR	39		/* external error	*/
#define V_ARESULT_EXTUNK	40		/* external unknown	*/
#define V_ARESULT_LOADED	41		/* job is loaded	*/

/*========================================================================
 * olog rresult codes
 *
 * tags:
 *	V_TAG_OLOG_RRESULT
 */
#define	V_RRESULT_NORMAL	1000		/* normal termination	*/
#define	V_RRESULT_RESCHED	1001		/* rescheduled		*/
#define	V_RRESULT_EXP_TIME	1002		/* expired - max time	*/
#define	V_RRESULT_EXP_ATTEMPTS	1003		/* expired - max attempts */
#define	V_RRESULT_CANCEL	1004		/* cancelled while sending */
#define	V_RRESULT_FAIL		1005		/* various other failures */
#define	V_RRESULT_MODIFIED	1006		/* job was modified	*/
#define V_RRESULT_QUEUED	1007		/* queued		*/
#define V_RRESULT_SUBMITTED	1008		/* submitted		*/
#define V_RRESULT_SENDING	1009		/* sending		*/
#define V_RRESULT_USRTERM	1010		/* user terminated	*/
#define V_RRESULT_IMGFAIL	1011		/* imaging failed	*/
#define V_RRESULT_EXPIRED	1012		/* expired		*/
#define V_RRESULT_FORWARD	1013		/* job was forwarded	*/
#define V_RRESULT_CONNECTING	1014		/* connecting		*/
#define V_RRESULT_DISCONNECTING	1015		/* disconnecting	*/
#define V_RRESULT_UNFWD		1016		/* job was un-forwarded	*/
#define V_RRESULT_EXTERR	1017		/* external error	*/
#define V_RRESULT_EXTUNK	1018		/* external unknown	*/
#define V_RRESULT_LOADED	1019		/* job is loaded	*/
#define V_RRESULT_LOADBAL	1020		/* job is load-balanced	*/

/*========================================================================
 * olog disposition values
 *
 * tags:
 *	V_TAG_OLOG_ACTIVE
 */
#define V_ENUM_ACTIVE_NOTRDY	"notrdy"	/* in submitted state	*/
#define V_ENUM_ACTIVE_READY	"ready"		/* in queued state	*/
#define V_ENUM_ACTIVE_EXPIRED	"expired"	/* done - expired	*/
#define V_ENUM_ACTIVE_DONE	"done"		/* done - other		*/

/*========================================================================
 * notification modes
 *
 * tags:
 *	V_TAG_ENV_NOTIFY
 *	V_TAG_OLOG_NOTIFY
 *	V_TAG_USER_NOTIFY
 *	V_TAG_USER_SNDFAX
 *	V_TAG_USER_SNDPRTNTY
 *	V_TAG_USER_SNDPRTFAX
 */
#define V_ENUM_NOTIFY_BOTH	"okfail"	/* always notify	*/
#define V_ENUM_NOTIFY_OK	"ok"		/* notify if successful	*/
#define V_ENUM_NOTIFY_FAIL	"fail"		/* notify if failed	*/
#define V_ENUM_NOTIFY_NONE	"none"		/* never notify		*/
#define V_ENUM_NOTIFY_EACH	"each"		/* notify each attempt	*/

#define V_ENUM_NOTIFY_DEFAULT	V_ENUM_NOTIFY_FAIL

/*========================================================================
 * mail formats
 *
 * tags:
 *	V_TAG_ENV_TOMFMT
 *	V_TAG_ENV_MAILFMT
 *	V_TAG_OLOG_TOMFMT
 *	V_TAG_USER_SNDFMT
 *	V_TAG_USER_RCVFMT
 *	V_TAG_USER_RTEFMT
 */
#define V_ENUM_MAILFMT_NORM	"norm"		/* cvr & fax as one TIF	*/
#define V_ENUM_MAILFMT_CVRFAX	"cvrfax"	/* cvr & fax as sep TIF	*/
#define V_ENUM_MAILFMT_PAGES	"pages"		/* each page as sep TIF	*/
#define V_ENUM_MAILFMT_GIF	"gif"		/* each page as sep GIF	*/
#define V_ENUM_MAILFMT_PDF	"pdf"		/* cvr & fax as one PDF	*/
#define V_ENUM_MAILFMT_CVRPDF	"cvrpdf"	/* cvr & fax as sep PDF	*/

#define V_ENUM_MAILFMT_DEFAULT	V_ENUM_MAILFMT_NORM

/*========================================================================
 * Request types
 *
 * tags:
 *	V_TAG_OLOG_REQTYPE
 */
#define V_ENUM_REQTYPE_REG	"regular"	/* regular job		*/
#define V_ENUM_REQTYPE_GRP	"group"		/* member of a group	*/

/*========================================================================
 * File status types
 *
 * tags:
 *	V_TAG_HDR_FSTATUS
 */
#define V_ENUM_FSTATUS_NEW	"new"		/* new (unread)		*/
#define V_ENUM_FSTATUS_READ	"read"		/* file has been read	*/
#define V_ENUM_FSTATUS_CUR	"cur"		/* file is current	*/
#define V_ENUM_FSTATUS_DEL	"del"		/* File was deleted	*/
#define V_ENUM_FSTATUS_FWD	"fwd"		/* File was forwarded	*/
#define V_ENUM_FSTATUS_CPY	"cpy"		/* File was copied	*/

/*========================================================================
 * File source types
 *
 * tags:
 *	V_TAG_HDR_FSOURCE
 */
#define V_ENUM_FSOURCE_RCVD	"rcvd"		/* File was received	*/
#define V_ENUM_FSOURCE_ROUTED	"routed"	/* File was routed	*/

/*========================================================================
 * File delete reasons
 *
 * tags:
 *	V_TAG_HDR_DELWHY
 */
#define V_ENUM_DELWHY_DELAUTO	"delauto"	/* Auto-delete		*/
#define V_ENUM_DELWHY_DELREQ	"delreq"	/* User request		*/
#define V_ENUM_DELWHY_DELROUTE	"delroute"	/* Routed		*/
#define V_ENUM_DELWHY_DELRNP	"delrnp"	/* RNP delete		*/
#define V_ENUM_DELWHY_DELPURGE	"delpurge"	/* Purged		*/

/*========================================================================
 * User types
 *
 * tags:
 *	V_TAG_USER_USERTYPE
 */
#define V_ENUM_USERTYPE_IND	"ind"		/* Individual		*/
#define V_ENUM_USERTYPE_DEPT	"dept"		/* Department		*/
#define V_ENUM_USERTYPE_GROUP	"grp"		/* Group		*/
#define V_ENUM_USERTYPE_SYSTEM	"system"	/* System		*/

/*========================================================================
 * User modes
 *
 * tags:
 *	V_TAG_USER_USRMODE
 */
#define V_ENUM_USERMODE_REG	"reg"		/* Regular		*/
#define V_ENUM_USERMODE_ADMIN	"admin"		/* Administrator	*/

/*========================================================================
 * License modes
 *
 * tags:
 *	V_TAG_SRVI_LICMODE
 */
#define V_LICMODE_NONE		0		/* none			*/
#define V_LICMODE_REAL		1		/* real - permanent	*/
#define V_LICMODE_DEMO		2		/* demo - temp/perm	*/
#define V_LICMODE_EVAL		3		/* eval - relative date	*/
#define V_LICMODE_DATE		4		/* eval - fixed date	*/
#define V_LICMODE_TEMP		6		/* real - fixed date	*/

/*========================================================================
 * File Convert types
 *
 * tags:
 *	V_TAG_ENV_CVTTYPE
 *	V_TAG_OLOG_CVTTYPE
 */
#define V_CVTTYPE_TIF		"tif"		/* convert to TIF	*/
#define V_CVTTYPE_MIME		"mim"		/* encapsulate as MIME	*/

#define V_CVTTYPE_DEFAULT	V_CVTTYPE_TIF

/*========================================================================
 * Priorities
 *
 * These are actually a resource, but the list is static.
 *
 * tags:
 *	V_TAG_ENV_PRIORITY
 *	V_TAG_OLOG_PRIORITY
 *	V_TAG_USER_MAXPRI
 *	V_TAG_USER_PRIORITY
 */
#define V_PRIORITY_LOW		"l"		/* low    priority	*/
#define V_PRIORITY_MEDIUM	"m"		/* medium priority	*/
#define V_PRIORITY_HIGH		"h"		/* high   priority	*/
#define V_PRIORITY_URGENT	"u"		/* urgent priority	*/

#define V_PRIORITY_DEFAULT	V_PRIORITY_MEDIUM

/*========================================================================
 * Resolutions
 *
 * These are actually a resource, but the list is static.
 *
 * tags:
 *	V_TAG_ENV_SENDRES
 *	V_TAG_ENV_FILERES
 *	V_TAG_ENV_DEST_FILERES
 *	V_TAG_OLOG_SENDRES
 *	V_TAG_HDR_RES
 *	V_TAG_ILOG_RES
 *	V_TAG_USER_SENDRES
 */
#define V_RESOLUTION_STD	"std"		/* standard ( 98 x 204)	*/
#define V_RESOLUTION_FINE	"fine"		/* fine     (196 x 204)	*/

#define V_RESOLUTION_DEFAULT	V_RESOLUTION_FINE

/*========================================================================
 * Page lengths
 *
 * These are actually a resource, but the list is static.
 *
 * tags:
 *	V_TAG_ENV_PGLEN
 *	V_TAG_ENV_DEST_PGLEN
 */
#define V_PGLEN_LETTER		"letter"	/* letter size		*/
#define V_PGLEN_LEGAL		"legal"		/* legal  size		*/
#define V_PGLEN_A4		"a4"		/* a4     size		*/

#define V_PGLEN_DEFAULT		V_PGLEN_LETTER

/*========================================================================
 * auth mask bits
 *
 * tags:
 *	V_TAG_USER_AUTHMASK
 */
#define V_AUTH_LOGIN		0x0001		/* login		*/
#define V_AUTH_RECVFAX		0x0002		/* access rcvd faxes	*/

/* these are not-yet-implemented */

#define V_AUTH_COVERS		0x0004		/* access covers	*/
#define V_AUTH_PB		0x0008		/* access phone-book	*/
#define V_AUTH_FILES		0x0010		/* manipulate files	*/

/*========================================================================
 * category types
 *
 * tags:
 *	V_TAG_OLOG_CATEGORY
 */
#define V_CATEGORY_NONE		0		/* No category		*/
#define V_CATEGORY_VFS		1		/* VFS program		*/
#define V_CATEGORY_AUTOSEND	2		/* Autosend file	*/
#define V_CATEGORY_OUTLOOK	3		/* Outlook calls	*/
#define V_CATEGORY_ADMIN	4		/* Admin calls		*/
#define V_CATEGORY_API		5		/* API calls		*/
#define V_CATEGORY_GUI		6		/* GUI client calls	*/
#define V_CATEGORY_LB		7		/* Load-balance calls	*/
#define V_CATEGORY_LCR		8		/* LCR calls		*/

/*========================================================================
 * server states
 *
 * tags:
 *	V_TAG_SRVI_STATE
 */
#define V_STATE_DOWN		0		/* server down		*/
#define V_STATE_INIT		1		/* server coming up	*/
#define V_STATE_STOPPING	2		/* server going down	*/
#define V_STATE_READY		3		/* server ready		*/

/*========================================================================
 * message types
 */
#define V_MSG_ROUTE		1		/* change in route list	*/
#define V_MSG_RCVFAX		2		/* change in rcvd faxes	*/
#define V_MSG_SNTFAX		3		/* change in sent faxes	*/

/*========================================================================
 * fax archive locations
 */
#define V_ARCH_USER		0		/* user's rcvd fax	*/
#define V_ARCH_UARC		1		/* user's arch fax	*/
#define V_ARCH_SENT		2		/* system sent fax	*/
#define V_ARCH_RCVD		3		/* system rcvd fax	*/

/*========================================================================
 * Server error codes (from VSI_Server_Get_Last_Error())
 */
	/*----------------------------------------------------------------
	 * generic errors
	 */
#define EX_G_OK			0	/* No error			*/
#define EX_G_NOMEM		1	/* No memory			*/
#define EX_G_SERVERR		2	/* Server error			*/
#define EX_G_SYSERR		3	/* System error			*/
#define EX_G_NOCONNECT		4	/* Not connected		*/
#define EX_G_IOERR		5	/* I/O error			*/
#define EX_G_USRERR		6	/* User rtn failed		*/

	/*----------------------------------------------------------------
	 * server errors
	 */
#define EX_S_INVUSER		7	/* Invalid user			*/
#define EX_S_INVPW		8	/* Invalid password		*/
#define EX_S_INVALT		9	/* Invalid altroute		*/
#define EX_S_INVHOST		10	/* Invalid hostname		*/
#define EX_S_INVCMD		11	/* Invalid command		*/
#define EX_S_NOPERM		12	/* Permission denied		*/
#define EX_S_TOOMANY		13	/* Logged in too many times	*/
#define EX_S_CMDFAIL		14	/* Command failed		*/
#define EX_S_USEREXISTS		15	/* User already exists		*/
#define EX_S_NOLIC		16	/* No license available		*/
#define EX_S_NORSP		17	/* No response from server	*/
#define EX_S_NOTRUNNING		18	/* Scheduler not running	*/
#define EX_S_VERNA		19	/* Version not available	*/

	/*----------------------------------------------------------------
	 * object errors
	 */
#define EX_E_INVDATA		20	/* Invalid data			*/
#define EX_E_INVFIELD		21	/* Invalid field		*/
#define EX_E_INVOBJ		22	/* Invalid object		*/
#define EX_E_INVINDEX		23	/* Invalid index		*/
#define EX_E_INVTAG		24	/* Invalid tag			*/
#define EX_E_TOOLONG		25	/* Data too long		*/
#define EX_E_TRUNC		26	/* Data was truncated		*/
#define EX_E_TOOMANYDESTS	27	/* Max dests exceeded		*/
#define EX_E_TOOMANYFILES	28	/* Max files exceeded		*/

	/*----------------------------------------------------------------
	 * phone-book errors
	 */
#define EX_P_CTERR		29	/* C-Tree error			*/
#define EX_P_INVPER		30	/* Invalid person		*/
#define EX_P_INVGRP		31	/* Invalid group		*/
#define EX_P_INVMEM		32	/* Invalid member		*/
#define EX_P_DUPALIAS		33	/* Duplicate alias		*/
#define EX_P_PBNOTOPEN		34	/* Phone-book not open		*/
#define EX_P_PBRDONLY		35	/* Phone-book open read only	*/
#define EX_P_LEOF		36	/* Logical EOF encountered	*/
#define EX_P_INVDATA		37	/* Invalid data			*/
#define EX_P_DNE		38	/* Phone-book does not exist	*/
#define EX_P_NOMEM		39	/* No memory			*/
#define EX_P_PBEXISTS		40	/* Phone-book exists		*/
#define EX_P_SYSERR		41	/* System error			*/

/*========================================================================
 * defines for inhflds & usrflds
 *
 * The inhflds & usrflds are char strings with one byte for each field
 * referenced.  The value is the index into the field list.
 */
	/*----------------------------------------------------------------
	 * userid section (00 - 19)
	 */
#define V_USER_PROF_LANGUAGE	1	/* language			*/
#define V_USER_PROF_LOCALE	2	/* locale			*/
#define V_USER_PROF_TIMEZONE	3	/* timezone			*/

#define V_USER_PROF_OSLOGIN	4	/* allow O/S login		*/

#define V_USER_PROF_DIDEXT	5	/* DID extension		*/
#define V_USER_PROF_CALCDID	6	/* TRUE to calculate DID	*/
#define V_USER_PROF_DIDLEN	7	/* length of calculated DID	*/

#define V_USER_PROF_ISPROFILE	8	/* entry is a profile		*/
#define V_USER_PROF_INHERIT	9	/* inherit mode			*/
#define V_USER_PROF_PARENTID	10	/* parent id for inheritence	*/
#define V_USER_PROF_MAKEPROF	11	/* make child a profile		*/

#define V_USER_PROF_DATEFMT	12	/* date format			*/
#define V_USER_PROF_TIMEFMT	13	/* time format			*/

	/*----------------------------------------------------------------
	 * profile inherit/edit section (20 - 29)
	 */
#define V_USER_PROF_USREDIT	21	/* TRUE if user can edit	*/

#define V_USER_PROF_INHFLDS	22	/* list of inherited fields	*/
#define V_USER_PROF_USRFLDS	23	/* list of editable fields	*/

	/*----------------------------------------------------------------
	 * coverpage section (30 - 49)
	 */
#define V_USER_PROF_COVERPAGE	31	/* coverpage name		*/
#define V_USER_PROF_USECVR	32	/* TRUE to use coverpage	*/

#define V_USER_PROF_FRNAME	33	/* sender name			*/
#define V_USER_PROF_FRCOMP	34	/* sender company		*/
#define V_USER_PROF_FRFNUM	35	/* sender fax number		*/
#define V_USER_PROF_FRVNUM	36	/* sender voice number		*/
#define V_USER_PROF_ADDR1	37	/* address line 1		*/
#define V_USER_PROF_ADDR2	38	/* address line 2		*/
#define V_USER_PROF_ADDR3	39	/* address line 3		*/
#define V_USER_PROF_COUNTRY	40	/* country			*/
#define V_USER_PROF_EMAIL	41	/* email address		*/

#define V_USER_PROF_ORUI	42	/* override ui info		*/

#define V_USER_PROF_USRTAG1	43	/* user tag 1			*/
#define V_USER_PROF_USRTAG2	44	/* user tag 2			*/
#define V_USER_PROF_USRTAG3	45	/* user tag 3			*/
#define V_USER_PROF_USRTAG4	46	/* user tag 4			*/

	/*----------------------------------------------------------------
	 * sendparam section (50 - 79)
	 */
#define V_USER_PROF_MAXPRI	51	/* max priority allowed		*/

#define V_USER_PROF_PRIORITY	52	/* send priority		*/
#define V_USER_PROF_NOTIFY	53	/* notify mode			*/
#define V_USER_PROF_MAILADDR	54	/* email address for notifys	*/
#define V_USER_PROF_RETRY	55	/* retry strategy		*/
#define V_USER_PROF_QUEUE	56	/* queue to use			*/
#define V_USER_PROF_SENDRES	57	/* send resolution		*/
#define V_USER_PROF_DEFPGL	58	/* page length			*/
#define V_USER_PROF_DEFLND	59	/* TRUE for landscape mode	*/
#define V_USER_PROF_TSI		60	/* TSI				*/
#define V_USER_PROF_TTI		61	/* TTI				*/
#define V_USER_PROF_COSTCODE	62	/* cost code			*/

#define V_USER_PROF_LNP		63	/* user LNP			*/
#define V_USER_PROF_TNP		64	/* user TNP			*/
#define V_USER_PROF_GNP		65	/* user GNP			*/

#define V_USER_PROF_AREACODE	66	/* area code			*/
#define V_USER_PROF_PREFIX	67	/* phone number prefix		*/
#define V_USER_PROF_SUFFIX	68	/* phone number suffix		*/

#define V_USER_PROF_PAGEHDR	69	/* page header string		*/
#define V_USER_PROF_USEPGH	70	/* TRUE to use page hdr		*/

#define V_USER_PROF_LCR		71	/* TRUE to allow LCR		*/
#define V_USER_PROF_ARCHIVE	72	/* archive sent fax		*/
#define V_USER_PROF_ACB		73	/* allow call-batching		*/
#define V_USER_PROF_ALB		74	/* allow load-balancing		*/

#define V_USER_PROF_TMPLATE	75	/* template to use		*/
#define V_USER_PROF_SKELETON	76	/* attachment skeleton		*/

	/*----------------------------------------------------------------
	 * send actions section (80 - 99)
	 */
#define V_USER_PROF_SNDFAX	80	/* mode to email sent fax	*/
#define V_USER_PROF_SNDFAXEMA	81	/* email name			*/
#define V_USER_PROF_SNDFMT	82	/* mail format			*/

#define V_USER_PROF_SNDPRTNTY	83	/* mode to print status		*/
#define V_USER_PROF_SNDPRTFAX	84	/* mode to print fax		*/
#define V_USER_PROF_SPRTSTR	85	/* send printer string		*/

#define V_USER_PROF_SNDARCH	86	/* TRUE to archive sent fax	*/

#define V_USER_PROF_SNDNTYTPOK	87	/* send OK template		*/
#define V_USER_PROF_SNDNTYTPNG	88	/* send NG template		*/
#define V_USER_PROF_SNDNTYTPGP	89	/* send group template		*/
#define V_USER_PROF_SNDSUBTP	90	/* send subject template	*/

	/*----------------------------------------------------------------
	 * recv actions section (100 - 119)
	 */
#define V_USER_PROF_RCVNTY	101	/* TRUE to email rcvd notify	*/
#define V_USER_PROF_RCVNTYEMA	102	/* email name			*/

#define V_USER_PROF_RCVFAX	103	/* TRUE to email rcvd fax	*/
#define V_USER_PROF_RCVFAXEMA	104	/* email name			*/
#define V_USER_PROF_RCVFMT	105	/* mail format			*/

#define V_USER_PROF_RCVPRT	106	/* TRUE to print rcvd fax	*/
#define V_USER_PROF_RPRTSTR	107	/* printer name			*/

#define V_USER_PROF_RCVARCH	108	/* TRUE to archive rcvd faxes	*/

#define V_USER_PROF_RCVNTYTP	109	/* template			*/
#define V_USER_PROF_RCVSUBTP	110	/* subject template		*/

	/*----------------------------------------------------------------
	 * route actions section (120 - 139)
	 */
#define V_USER_PROF_RNP		121	/* RNP				*/

#define V_USER_PROF_RTENTY	122	/* TRUE to email routed nty	*/
#define V_USER_PROF_RTENTYEMA	123	/* email name			*/

#define V_USER_PROF_RTEFAX	124	/* TRUE to email routed fax	*/
#define V_USER_PROF_RTEFAXEMA	125	/* email name			*/
#define V_USER_PROF_RTEFMT	126	/* mail format			*/

#define V_USER_PROF_RTEPRT	127	/* TRUE to print routed fax	*/
#define V_USER_PROF_PRTSTR	128	/* printer name			*/

#define V_USER_PROF_AUTODEL	129	/* TRUE to delete routed fax	*/

#define V_USER_PROF_RTENTYTP	130	/* template			*/
#define V_USER_PROF_RTESUBTP	131	/* subject template		*/

	/*----------------------------------------------------------------
	 * misc client program fields (140 - 159)
	 */
#define V_USER_PROF_CLRONSEND	141	/* clear on send		*/
#define V_USER_PROF_DISPSTAT	142	/* display status dialog	*/
#define V_USER_PROF_WEBINSTALL	143	/* allow web install		*/
#define V_USER_PROF_AUTHMASK	144	/* default auth mask		*/

	/*----------------------------------------------------------------
	 * directory section (160 - 189)
	 */
#define V_USER_PROF_PARENTDIRS	161	/* use parent directories	*/
#define V_USER_PROF_DIRECTORIES	162	/* available directory list	*/

/*------------------------------------------------------------------------
 * values for inhflds & usrflds bytes
 */
#define V_USER_PROF_UNUSED	'.'	/* entry unused			*/
#define V_USER_PROF_NOCONTROL	0	/* entry not under inh control	*/

#define V_USER_PROF_INH_OFF	'0'	/* inherit is off		*/
#define V_USER_PROF_INH_ON	'1'	/* inherit is on		*/

#define V_USER_PROF_EDIT_OFF	'0'	/* user cannot edit		*/
#define V_USER_PROF_EDIT_ON	'1'	/* user can edit		*/
#define V_USER_PROF_EDIT_RUN	'2'	/* user can change at run time	*/

/*------------------------------------------------------------------------
 * C++ encapsulation
 */
#ifdef __cplusplus
}
#endif

#endif /* VSI_H */
