#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	setup processing for vfs installation
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# add a directory to the path
#
# Note that the directory is ALWAYS added to the end.
#
addpath ()
{
	dir="$1"

	sep=":"
	arg=""
	if in_windows
	then
		sep=";"
		arg="-i"
	fi

	foo=`echo "$PATH" | egrep $arg "^$dir$sep|$sep$dir$sep|$sep$dir$"`
	if [ $? -eq 0 ]
	then
		echo "$PATH"
	else
		echo "$PATH$sep$dir"
	fi

	unset dir sep arg foo
}

# ------------------------------------------------------------------------
# Return the user name to use
#
usrname ()
{
	echo "${VSI_USRNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# Return the group name to use
#
grpname ()
{
	echo "${VSI_GRPNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# Return the legacy product name to use
#
legname ()
{
	echo "vsifax3"
}

# ------------------------------------------------------------------------
# Set owner & group of a file
#
set_usrgrp ()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# get args
	#
	USRNAME="$1"
	GRPNAME="$2"
	FILE="$3"

	# ----------------------------------------------------------------
	# set the owner & group
	#
	chgrp $GRPNAME "$FILE"
	chown $USRNAME "$FILE"
}

# ------------------------------------------------------------------------
# create env files & update system files to source them
#
add_env_files ()
{
	# ----------------------------------------------------------------
	# validate that /etc exists
	#
	[ -d /etc ] || return

	# ----------------------------------------------------------------
	# get package info
	#
	USRNAME=`usrname`
	GRPNAME=`grpname`
	PKGNAME=`pkgname`
	PRDNAME=`prdname`
	LEGNAME=`legname`

	# ----------------------------------------------------------------
	# Create env files
	#
	SH_ENV_FILE=/etc/$PRDNAME.sh
	(
		echo "# setup $PKGNAME environment"
		echo "VSIFAX=\"$VSIFAX\""
		echo "PATH=\"\$PATH:\$VSIFAX/bin\""
		echo 'id=`id | sed -e "s/[^=]*=//" -e "s/(.*//"`'
		echo "[ \$id -eq 0 -o \"\$LOGNAME\" = \"$USRNAME\" ] &&"
		echo "	PATH=\"\$PATH:\$VSIFAX/lbin\""
		echo "unset id"
		echo "[ -d \"\$VSIFAX/obin\" ] && PATH=\"\$VSIFAX/obin:\$PATH\""
		echo "export VSIFAX PATH"
	) >$SH_ENV_FILE

	chmod 775 $SH_ENV_FILE
	set_usrgrp $USRNAME $GRPNAME $SH_ENV_FILE

	SH_LEG_FILE=/etc/$LEGNAME.sh
	rm -f $SH_LEG_FILE
	ln -s `basename $SH_ENV_FILE` $SH_LEG_FILE

	CSH_ENV_FILE=/etc/$PRDNAME.csh
	(
		echo "# setup $PKGNAME environment"
		echo "setenv VSIFAX \"$VSIFAX\""
		echo "set path=(\$path \"\$VSIFAX/bin\")"
		echo 'set id=`id | sed -e "s/[^=]*=//" -e "s/(.*//"`'
		echo "if ( \$id == 0 || \"$LOGNAME\" == \"$USRNAME\" ) \\"
		echo "	set path=(\$path \"\$VSIFAX/lbin\")"
		echo "unset id"
		echo "if (-d \"\$VSIFAX/obin\") set path=(\"\$VSIFAX/obin\" \$path)"
	) >$CSH_ENV_FILE

	chmod 775 $CSH_ENV_FILE
	set_usrgrp $USRNAME $GRPNAME $CSH_ENV_FILE

	CSH_LEG_FILE=/etc/$LEGNAME.csh
	rm -f $CSH_LEG_FILE
	ln -s `basename $CSH_ENV_FILE` $CSH_LEG_FILE

	# ----------------------------------------------------------------
	# Check if /etc/profile or /etc/cshrc need to be added
	#
	DO_ADD=no
	FILES=""
	if [ -f /etc/profile ]
	then
		grep $SH_ENV_FILE /etc/profile >/dev/null 2>&1
		if [ $? -ne 0 ]
		then
			DO_ADD=yes
			FILES="file /etc/profile"
		fi
	fi

	if [ -f /etc/cshrc ]
	then
		grep $CSH_ENV_FILE /etc/cshrc >/dev/null 2>&1
		if [ $? -ne 0 ]
		then
			DO_ADD=yes
			FILES="files /etc/profile and /etc/cshrc"
		fi
	fi

	# ----------------------------------------------------------------
	# Ask if we should update system files
	#
	if false # [ "$DO_ADD" = "yes" -a "$QUICK_INSTALL" != "yes" ]
	then
		echo ""
		echo "The $PKGNAME system requires users to set a few"
		echo "environment variables prior to use."
		echo ""
		echo "The file $SH_ENV_FILE will set these variables for"
		echo "Bourne and Korn shell users and the file $CSH_ENV_FILE"
		echo "will set these variables for C-shell users."
		echo ""
		echo "The $FILES can be updated"
		echo "by this installation procedure to automatically"
		echo "source these files."
		echo ""
		echo "Updating $FILES."

		is_ok "set-env-files-ok" || DO_ADD=no
	fi

	# ----------------------------------------------------------------
	# Check if entry exists in /etc/profile to use the sh env file
	#
	if [ "$DO_ADD" = "yes" ]
	then
		if [ -f /etc/profile ]
		then
			grep $SH_ENV_FILE /etc/profile >/dev/null 2>&1
			if [ $? -ne 0 ]
			then
			(
			echo ""
			echo "# Setup the $PKGNAME environment"
			echo "[ -f $SH_ENV_FILE ] && . $SH_ENV_FILE"
			) >>/etc/profile
			fi
		fi
	fi

	# ----------------------------------------------------------------
	# Check if entry exists in /etc/cshrc to use the csh env file
	#
	if [ "$DO_ADD" = "yes" ]
	then
		if [ -f /etc/cshrc ]
		then
			grep $CSH_ENV_FILE /etc/cshrc >/dev/null 2>&1
			if [ $? -ne 0 ]
			then
			(
			echo ""
			echo "# Setup the $PKGNAME environment"
			echo "if (-f $CSH_ENV_FILE) source $CSH_ENV_FILE"
			) >>/etc/cshrc
			fi
		fi
	fi
}

# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -q     quick install"
	echo "  -s     silent install"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D     debug install"
}

# ------------------------------------------------------------------------
# main()
#
PATH=`addpath "$VSIFAX/bin"`
PATH=`addpath "$VSIFAX/lbin"`

USRNAME=`usrname`
GRPNAME=`grpname`

# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

DEBUG=

case "$1" in
	'-?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | -helpall | --helpall)
		helpall
		exit 0
		;;

	'-V' | -version | --version)
		echo "$PGM: `vadmin version`"
		exit 0
		;;
esac

ARGS=`getopt "Dqs" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-x
		shift
		;;

	-q)
		QUICK_INSTALL=yes	export QUICK_INSTALL
		shift
		;;

	-s)
		SILENT_INSTALL=yes	export SILENT_INSTALL
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

[ "$DEBUG" != "" ] && set $DEBUG

# ------------------------------------------------------------------------
# check that user is root
#
check_root

# ------------------------------------------------------------------------
# create env files if they don't exist
#
add_env_files

exit 0
