#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	Master install script
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# add a directory to the path
#
# Note that the directory is ALWAYS added to the end.
#
addpath ()
{
	dir="$1"

	sep=":"
	arg=""
	if in_windows
	then
		sep=";"
		arg="-i"
	fi

	foo=`echo "$PATH" | egrep $arg "^$dir$sep|$sep$dir$sep|$sep$dir$"`
	if [ $? -eq 0 ]
	then
		echo "$PATH"
	else
		echo "$PATH$sep$dir"
	fi

	unset dir sep arg foo
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# output a prompt & get a response from the user
#
get_user_input ()
{
	id="$1"
	dflt="$2"
	msg="$3"

	ok_id="$id-ok"
	[ "$id" = "" ] && ok_id=""

	CONSOLE=`get_dev_tty`

	while true
	do
		if [ x"$dflt" = x"!" ]
		then
			echonl "Enter $msg: "
		else
			echonl "Enter $msg [$dflt]: "
		fi >$CONSOLE

		response=`read_user_input "$id"`
		[ "$response" = "" -a x"$dflt" != x"!" ] && response="$dflt"

		echo "$msg = \"$response\"" >$CONSOLE
		is_ok "$ok_id" >$CONSOLE && { echo "$response"; break; }

		dflt="$response"
	done
}

# ------------------------------------------------------------------------
# get Windows directory
#
get_windir ()
{
	{
		if [ "$windir" != "" ]
		then
			echo "$windir"

		elif [ "$WINDIR" != "" ]
		then
			echo "$WINDIR"

		elif [ "$SystemRoot" != "" ]
		then
			echo "$SystemRoot"

		elif [ "$SYSTEMROOT" != "" ]
		then
			echo "$SYSTEMROOT"

		elif [ -d "c:/windows" ]
		then
			echo "c:/windows"

		elif [ -d "c:/winnt" ]
		then
			echo "c:/winnt"

		else
			echo "c:/windows"
		fi
	} | sed -e 's,\\\\*,/,g'
}

# ------------------------------------------------------------------------
# get Windows drive
#
get_windrive()
{
	get_windir | sed -e 's,[/\\].*,,'
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get default install directory
#
get_instdir ()
{
	if in_windows
	then
		echo "`get_windrive`/Program Files/VSI-FAX/FaxServer"
	else
		echo "/usr/`prdname`"
	fi
}

# ------------------------------------------------------------------------
# echo a line only if $SILENT_INSTALL != "yes"
#
echons ()
{
	[ "$SILENT_INSTALL" = "yes" ] || echo "$*"
}

# ------------------------------------------------------------------------
# echo a line to the user's tty
#
echotty ()
{
	CONSOLE=`get_dev_tty`
	echo "$*" >$CONSOLE
}

# ------------------------------------------------------------------------
# look for a program in the path
#
look_for_pgm ()
{
	pgm=$1

	l=`echo $PATH | sed -e 's/:/ /g'`
	for j in $l
	do
		[ -d $j -a -x $j/$pgm ] && return 0
	done

	return 1
}

# ------------------------------------------------------------------------
# Display one dot character on the standard output.  This is used
# at various places throughout the installation script to signal
# activity.
#
dot()
{
	echonl "."
}

# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# check if group exists & output the gid is it does
#
find_grp ()
{
	NAME="$1"

	if in_windows
	then
		echo "0"
		return
	fi

	gidline=`cut -d: -f1,2,3 /etc/group 2>/dev/null | grep "^$NAME:"`

	if [ "$gidline" = "" ]
	then
		if look_for_pgm ypcat
		then
			gidline=`ypcat group.byname 2>/dev/null |
			grep "^$NAME:"`
		fi
	fi

	if [ "$gidline" = "" ]
	then
		if look_for_pgm niscat
		then
			gidline=`niscat group.org_dir 2>/dev/null |
			grep "^$NAME:"`
		fi
	fi

	if [ "$gidline" != "" ]
	then
		grpid=`echo "$gidline" | cut -f3 -d:`
		echo "$grpid"
	fi
}

# ------------------------------------------------------------------------
# check if user exists & output the uid if it does
#
find_usr ()
{
	NAME="$1"

	if in_windows
	then
		echo "0"
		return
	fi

	uidline=`cat /etc/passwd 2>/dev/null | grep "^$NAME:"`

	if [ "$uidline" = "" ]
	then
		if look_for_pgm ypcat
		then
			uidline=`ypcat passwd.byname 2>/dev/null |
			grep "^$NAME:"`
		fi
	fi

	if [ "$uidline" = "" ]
	then
		if look_for_pgm niscat
		then
			uidline=`niscat passwd.org_dir 2>/dev/null |
			grep "^$NAME:"`
		fi
	fi

	if [ "$uidline" != "" ]
	then
		usrid=`echo "$uidline" | cut -f3 -d:`
		echo "$usrid"
	fi
}

# ------------------------------------------------------------------------
# create a group
#
make_grp ()
{
	GRP="$1"

	echons ""
	echons "Making group \"$GRP\"."

	if look_for_pgm groupadd
	then
		groupadd $GRP
	elif look_for_pgm mkgroup
	then
		mkgroup $GRP
	elif look_for_pgm pw
	then
		pw groupadd $GRP
	else
		echoerr "Cannot create group $GRP"
		false
	fi || return 1

	echons "Group \"$GRP\" created."
}

# ------------------------------------------------------------------------
# create a user
#
make_usr ()
{
	USR="$1"
	GRP="$2"
	DSC="$3"

	echons ""
	echons "Making user \"$USR\"."

	if look_for_pgm useradd
	then
		useradd -c "$DSC" -g $GRP $USR
	elif look_for_pgm mkuser
	then
		mkuser pgrp="$GRP" gecos="$DSC" $USR
	elif look_for_pgm pw
	then
		pw useradd $USR -c "$DSC" -g $GRP
	else
		echoerr "Cannot create user $USR"
		false
	fi || return 1

	echons "User \"$USR\" created."
}

# ------------------------------------------------------------------------
# Set owner & group of a file
#
set_usrgrp ()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# get args
	#
	USRNAME="$1"
	GRPNAME="$2"
	FILE="$3"

	# ----------------------------------------------------------------
	# set the owner & group
	#
	chgrp $GRPNAME "$FILE"
	chown $USRNAME "$FILE"
}

# ------------------------------------------------------------------------
# function to make a directory hierarchy
#
make_dir ()
{
	PRM="$1"
	USR="$2"
	GRP="$3"
	DIR="$4"

	if in_windows
	then
		mkdir -p "$DIR"
		return
	fi

	case "$DIR" in
	/*)	xd=""	;;
	*)	xd="."	;;
	esac

	save_ifs=$IFS
	IFS=/
	for xi in $DIR
	do
		xd="$xd/$xi"
		if [ ! -d "$xd" ]
		then
			mkdir "$xd"
			if [ $? -ne 0 ]
			then
				IFS="$save_ifs"
				return 1
			else
				chmod $PRM "$xd"
				set_usrgrp $USR $GRP "$xd"
			fi
		fi
	done

	IFS="$save_ifs"
	return 0
}

# ------------------------------------------------------------------------
# Get the permissions of a file
#
get_perms ()
{
	# ----------------------------------------------------------------
	# get each permission part as a separate char
	#
	FILE="$1"

	OPTS=-ld
	in_windows || OPTS="${OPTS}L"
	PERMS=`ls $OPTS $FILE | cut -d' ' -f1 | sed -e 's/\(.\)/\1 /g'`

	# ----------------------------------------------------------------
	# now get all parts as named vars
	#
	set -- $PERMS
	shift

	RUSR=$1; WUSR=$2; XUSR=$3
	RGRP=$4; WGRP=$5; XGRP=$6
	ROTH=$7; WOTH=$8; XOTH=$9

	# ----------------------------------------------------------------
	# now create the perms value
	# Note that we calculate this as a decimal number,
	# but it will be interpreted as an octal number.
	#
	PERMS=0

	case "$RUSR" in
		r)	PERMS=`expr $PERMS +  400`	;;
	esac

	case "$WUSR" in
		w)	PERMS=`expr $PERMS +  200`	;;
	esac

	case "$XUSR" in
		x)	PERMS=`expr $PERMS +  100`	;;
		s)	PERMS=`expr $PERMS + 4100`	;;
		S)	PERMS=`expr $PERMS + 4000`	;;
	esac

	case "$RGRP" in
		r)	PERMS=`expr $PERMS +   40`	;;
	esac

	case "$WGRP" in
		w)	PERMS=`expr $PERMS +   20`	;;
	esac

	case "$XGRP" in
		x)	PERMS=`expr $PERMS +   10`	;;
		s)	PERMS=`expr $PERMS + 2010`	;;
		[SlL])	PERMS=`expr $PERMS + 2000`	;;
	esac

	case "$ROTH" in
		r)	PERMS=`expr $PERMS +    4`	;;
	esac

	case "$WOTH" in
		w)	PERMS=`expr $PERMS +    2`	;;
	esac

	case "$XOTH" in
		x)	PERMS=`expr $PERMS +    1`	;;
		t)	PERMS=`expr $PERMS + 1001`	;;
		T)	PERMS=`expr $PERMS + 1000`	;;
	esac

	echo "$PERMS"
}

# ------------------------------------------------------------------------
# copy a file to the proper directory, creating the directory as needed
#
install_mv()
{
	# ----------------------------------------------------------------
	# Get args
	#
	SRCFIL="$1"		# source file
	DSTDIR="$2"		# destination directory
	USR="$3"		# user  name
	GRP="$4"		# group name

	# ----------------------------------------------------------------
	# If source is a directory, then just create it
	#
	if [ -d "$SRCFIL" ]
	then
		TGTDIR="$VSIFAX/$DSTDIR/$SRCFILE"
		if [ ! -d "$TGTDIR" ]
		then
			make_dir 755 $USR $GRP "$TGTDIR"
			if [ $? -ne 0 ]
			then
				echoerr ""
				echoerr "Cannot make directory $TGTDIR"
				return 1
			fi

			PERMS=`get_perms "$SRCFILE"`
			chmod $PERMS "$TGTDIR"
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# get target directory
	#
	TGTDIR="$VSIFAX/$DSTDIR"

	# ----------------------------------------------------------------
	# Create dest directory if it doesn't exist
	#
	if [ ! -d "$TGTDIR" ]
	then
		make_dir 755 $USR $GRP "$TGTDIR"
		if [ $? -ne 0 ]
		then
			echoerr ""
			echoerr "Cannot make directory $TGTDIR"
			return 1
		fi
	fi

	# ----------------------------------------------------------------
	# get dest filename
	#
	DSTFIL="$TGTDIR/$SRCFIL"

	# ----------------------------------------------------------------
	# delete any existing "old" file to avoid perm problems
	#
	rm -f "$DSTFILE"

	# ----------------------------------------------------------------
	# copy the file
	#
	cp -p "$SRCFIL" "$DSTFIL" || return 1

	# ----------------------------------------------------------------
	# set the file ownerships
	#
	set_usrgrp $USR $GRP "$DSTFIL"

	return 0
}

# ------------------------------------------------------------------------
# Copy over all files
#
copy_files ()
{
	USR="$1"
	GRP="$2"

	dirs=`find . -type d -print | sort`
	for d in $dirs
	do
		[ "$d" = "." ] && continue

		d=`echo "$d" | sed -e 's/..//'`
		(
			cd $d
			files=`echo *`
			[ "$files" = "*" ] && files=""
			for i in . $files
			do
				if [ "$SILENT_INSTALL" != "yes" ]
				then
					if [ "$VERBOSE_INSTALL" != "yes" ]
					then
						dot
					else
						echo "copying $d/$i"
					fi
				fi
				install_mv $i $d $USR $GRP || return 1
			done
		)
	done

	[ "$SILENT_INSTALL" != "yes" -a "$VERBOSE_INSTALL" != "yes" ] && echo ""

	return 0
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# Stop any servers which are running
#
stop_servers ()
{
	if [ -x "$VSIFAX/bin/vfxsched" -o -x "$VSIFAX/bin/vfxsched.exe" ]
	then
		PKGNAME=`pkgname`

		srv_running=no
		adm_running=no

		vfxsched -s check && srv_running=yes
		vadmind  -s check && adm_running=yes

		if [ "$srv_running" = "yes" -o "$adm_running" = "yes" ]
		then
			[ "$QUICK_INSTALL" = "yes" ] ||
			{
		echo ""
		echo "The $PKGNAME servers must be shutdown to do an install."
		echo "Shutting down the $PKGNAME servers."
				is_ok "shutdown-ok"
			}

			if [ $? -eq 0 ]
			then
				VFLAG=
				[ "$SILENT_INSTALL" = "yes" ] && VFLAG=-s
				[ "$srv_running" = "yes" ] &&
					vfxsched $VFLAG stop
				[ "$adm_running" = "yes" ] &&
					vadmind  $VFLAG stop
			else
				echo ""
				echo "Please run this installation again after"
				echo "shutting down the $PKGNAME Servers."
				abort_install
			fi
		fi
	fi
}

# ------------------------------------------------------------------------
# Check if servers should be started
#
start_servers ()
{
	if [ "$QUICK_INSTALL" != "yes" ]
	then
		PKGNAME=`pkgname`

		if [ -x "$VSIFAX/bin/vfxsched" -o \
		     -x "$VSIFAX/bin/vfxsched.exe" ]
		then
			echo ""
			if getyesorno -y "start-admin" \
			"Do you want to start the $PKGNAME Admin Server now?"
			then
				vadmind
			fi

			echo ""
			if getyesorno -y "start-server" \
			"Do you want to start the $PKGNAME Fax Server now?"
			then
				vfxsched
			fi
		else
			if [ -x "$VSIFAX/lbin/vfxpickup" -o \
		     	     -x "$VSIFAX/lbin/vfxpickup.exe" ]
			then
			    echo ""
			    if getyesorno -y "start-vsipickup" \
			    "Do you want to start the $PKGNAME VFS Pickup Service now?"
			    then
				$VSIFAX/lbin/vfxpickup
			    fi

			fi
		fi
	fi
}

# ------------------------------------------------------------------------
# Return the user name to use
#
usrname ()
{
	echo "${VSI_USRNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# Return the group name to use
#
grpname ()
{
	echo "${VSI_GRPNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# list of software bundles
#
# "-n"		display bundle names only
# "-d name"	display description for <name>
#
bundle ()
{
	# ----------------------------------------------------------------
	# check if partial list wanted
	#
	if [ x"$1" = x"-n" ]
	then
		OUT="cut -d: -f1"
	elif [ x"$1" = x"-d" ]
	then
		OUT="grep $2 | cut -d: -f2"
	else
		OUT=cat
	fi

	# ----------------------------------------------------------------
	# here is the list
	#
	sed -e '/^#/d' -e 's/		*/:/g' <<-EOF | eval $OUT
	#
	#bundle		description
	#--------	-----------------
	fxserver	Fax Server
	fxvfsxml	XML Virtual Fax Server
	fxpickup	VFS pickup service
	fxcompat	Compatibility programs
	fxfaxapi	Fax Client API
	fxapicpp	Fax Client API (C/C++)
	fxapixmlf	Fax Client API (XMLF)
	fxapijava	Fax Client API (Java)
	fxbt1034	Brooktrout TR1034 drivers
	fxfodc		Fax on Demand Connector
	EOF
}

# ------------------------------------------------------------------------
# get response dir/file info
#
response_info ()
{
	if in_windows
	then
		RESPONSE_DIR="`get_windrive`/Program Files/Common Files/VSI-FAX Shared"
	else
		RESPONSE_DIR="/var/spool/pkg"
	fi
	RESPONSE_FILE="`prdname`-install.log"

	echo "RESPONSE_DIR=\"$RESPONSE_DIR\""
	echo "SAVE_RESPONSE_FILE=\"$RESPONSE_DIR/$RESPONSE_FILE\""
}

# ------------------------------------------------------------------------
# look for required lib
#
check_lib ()
{
	pgm=$1
	ldconfig -p 2> /dev/null | grep "$pgm" > /dev/null 2>&1
	[ $? -eq 0 ] && return 0

	return 1
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -r            read response file"
	echo "  -q            quick install"
	echo "  -s            silent install"
	echo "  -n            no delete of install files"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D            debug install"
	echo "  -v            verbose mode"
}

# ------------------------------------------------------------------------
# get list of bundles present
#
bundles_present ()
{
	bundle -n |
	while read NAME
	do
		[ -d $NAME ] && echo "$NAME"
	done
}

# ------------------------------------------------------------------------
# get vsi dirs
#
get_vsi_dirs ()
{
	VBASE=$DFLT_VSIFAX

	if [ "$VSIFAX" = "" ]
	then
		[ -f /etc/$PRDNAME.sh ] && . /etc/$PRDNAME.sh
	fi

	if [ "$VSIFAX" = "" ]
	then
		echotty ""
	echotty "Please enter the install directory for the $PKGNAME system."

		VSIFAX=`get_user_input "install-dir" "$VBASE" "install directory"`
	fi

	echo "$VSIFAX"
}

# ------------------------------------------------------------------------
# install a bundle
#
install_bundle ()
{
	prod=$1
	desc=`bundle -d $prod`

	PROD_DIR=`ls | grep -i $prod`
	[ "$PROD_DIR" = "" ] && return 0

	(
		cd $PROD_DIR

		echons ""
		echons "Installing $PKGNAME $desc"

		copy_files $USRNAME $GRPNAME || return 1
	) || return 1

	return 0
}

# ========================================================================
# main()
#
umask 022

PKGNAME=`pkgname`
PRDNAME=`prdname`
USRNAME=`usrname`	export USRNAME
GRPNAME=`grpname`	export GRPNAME

DFLT_VSIFAX=`get_instdir`

eval `response_info`

# ------------------------------------------------------------------------
# set SHELL (just in case some other SHELL is used by root)
#
in_windows || { SHELL=/bin/sh export SHELL; }

# ------------------------------------------------------------------------
# catch SIGINT signals
#
trap abort_install 2

# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

FXCPRM=yes
DEBUG=
QUICK_INSTALL=no
SILENT_INSTALL=no
VERBOSE_INSTALL="no"

if [ "$1" = "-?" -o "$1" = "-help" ]
then
	usage
	exit 0
fi

if [ "$1" = "-??" -o "$1" = "-helpall" ]
then
	helpall
	exit 0
fi

ARGS=`getopt "Dvnqsr" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-x
		shift
		;;

	-q)
		QUICK_INSTALL=yes
		FXCPRM=no
		shift
		;;

	-s)
		SILENT_INSTALL=yes
		QUICK_INSTALL=yes
		FXCPRM=no
		shift
		;;

	-n)
		FXCPRM=no
		shift
		;;

	-r)
		LOAD_RESPONSE_FILE="$SAVE_RESPONSE_FILE"
		shift
		;;

	-v)
		VERBOSE_INSTALL="yes"
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

export QUICK_INSTALL
export SILENT_INSTALL
export VERBOSE_INSTALL
export SAVE_RESPONSE_FILE
export LOAD_RESPONSE_FILE

if [ "$SAVE_RESPONSE_FILE" != "" ]
then
	[ ! -d "$RESPONSE_DIR" ] && mkdir -p "$RESPONSE_DIR"
	touch "$SAVE_RESPONSE_FILE"
	chmod 664 "$SAVE_RESPONSE_FILE"
fi

if [ "$FXCPRM" = "yes" ]
then
	[ -w . ] || FXCPRM=no
fi

if [ "$DEBUG" != "" ]
then
	CMD_SHELL="$SHELL $DEBUG"
	set $DEBUG
else
	CMD_SHELL="$SHELL"
fi
export CMD_SHELL

INSTALL_MODE=update	export INSTALL_MODE

# ------------------------------------------------------------------------
# check that user is root
#
check_root

#
# Check for required libs
#
if in_linux
then
	check_lib "libstdc++.so.5"
	if [ $? -ne 0 ]
	then
echo ""
echo "VSI-FAX Server requires that the 'libstdc++.so.5' library be installed"
echo "on your Operating System to function properly. This library seems"
echo "to be currently missing."
echo ""
echo "Please refer to the instructions provided in the"
echo "'Installing libstdc++.so.5 on Red Hat Enterprise 4 or 5' topic of "
echo "vsireadme.txt or the Installation and Administration guide (vsifax_instadm.pdf"
echo "in the /docs folder of the VSI-FAX CD-ROM)."
echo ""
	getyesorno -n "cont-install-required-lib" \
		"Do you wish to continue with the installation?" ||
		abort_install
	fi
fi

# ------------------------------------------------------------------------
# initial display
#
echons "$PKGNAME System Installation"

# ------------------------------------------------------------------------
# check if user & group exist
#
USRID=`find_usr $USRNAME`
GRPID=`find_grp $GRPNAME`

# ------------------------------------------------------------------------
# Display initial menu message
#
if [ "$QUICK_INSTALL" != "yes" ]
then
  N=1

  if [ "$VSIFAX" = "" ]
  then
    [ $N -eq 1 ] && { echo ""; echo "Please note:"; }
    echo "$N) Before running this installation, please decide where you want"
    echo "   to install the $PKGNAME system."
    echo "   (The default location is $DFLT_VSIFAX)"
    echo ""
    N=`expr $N + 1`
  fi

  if [ "$GRPID" = "" ]
  then
    [ $N -eq 1 ] && { echo ""; echo "Please note:"; }
    echo "$N) You must create a group \"$GRPNAME\"."
    echo "   If you like, the installation program can do this for you."
    echo ""
    N=`expr $N + 1`
  fi

  if [ "$USRID" = "" ]
  then
    [ $N -eq 1 ] && { echo ""; echo "Please note:"; }
    echo "$N) You must create a user \"$USRNAME\"."
    echo "   If you like, the installation program can do this for you."
    echo ""
    N=`expr $N + 1`
  fi

  if [ $N -gt 1 ]
  then
	  getyesorno "cont-install" \
		"Do you wish to continue with the installation?" ||
		abort_install
  fi
fi

# ------------------------------------------------------------------------
# check that the proper group exists
#
if [ "$GRPID" = "" ]
then
	if [ "$QUICK_INSTALL" != "yes" ]
	then
		echo ""
		echo "Group name \"$GRPNAME\" does not exist."
		getyesorno "add-group" "Do you wish to create it?" ||
			abort_install
	fi

	make_grp $GRPNAME || abort_install

	GRPID=`find_grp $GRPNAME`
fi

# ------------------------------------------------------------------------
# check that the proper user exists
#
if [ "$USRID" = "" ]
then
	if [ "$QUICK_INSTALL" != "yes" ]
	then
		echo ""
		echo "User name \"$USRNAME\" does not exist."
		getyesorno "add-user" "Do you wish to create it?" ||
			abort_install
	fi

	make_usr $USRNAME $GRPNAME "VSIFAX Administrator" || abort_install

	USRID=`find_usr $USRNAME`
fi

# ------------------------------------------------------------------------
# Get VSI directories
#
VSIFAX=`get_vsi_dirs`	export VSIFAX

# ------------------------------------------------------------------------
# create top-level directory if it doesn't exist
#
[ -d "$VSIFAX" ] ||
{
	make_dir 755 $USRNAME $GRPNAME "$VSIFAX" || abort_install
}

# ------------------------------------------------------------------------
# Add VSI directories to the path
#
PATH=`addpath "$VSIFAX/bin"`
PATH=`addpath "$VSIFAX/lbin"`

# ------------------------------------------------------------------------
# Stop the servers if needed
#
stop_servers

# ------------------------------------------------------------------------
# Now install all bundles present
#
for pkg in `bundles_present`
do
	if [ -x install/$pkg-preinstall.sh ]
	then
	(
		cd $pkg
		$CMD_SHELL ../install/$pkg-preinstall.sh
	) || abort_install
	fi

	install_bundle $pkg || abort_install
done

# ------------------------------------------------------------------------
# check if any post-processing is to be done
#
for pkg in `bundles_present`
do
	if [ -x install/$pkg-postinstall.sh ]
	then
		$CMD_SHELL install/$pkg-postinstall.sh
	fi
done

# ------------------------------------------------------------------------
# delete installation files
#
if [ "$FXCPRM" = "yes" ]
then
	# ----------------------------------------------------------------
	# delete all bundles
	#
	if [ "$SILENT_INSTALL" != "yes" ]
	then
		echo ""
		echonl "Deleting all temp files "
	fi

	for pkg in `bundles_present` install
	do
		[ "$SILENT_INSTALL" != "yes" ] && dot
		rm -rf $pkg
	done

	[ "$SILENT_INSTALL" != "yes" ] && echo ""

	# ----------------------------------------------------------------
	# now delete this script
	#
	rm -f  install.sh
fi

# ------------------------------------------------------------------------
# check if servers should be started
#
start_servers

# ------------------------------------------------------------------------
# final message to user
#
echons ""
echons "Installation is complete."
echons ""
echons "The complete set of documentation in PDF format is available on your"
echons "$PKGNAME CD-ROM in the \docs folder."
echons "You can also search for and browse the latest technical documentation"
echons "at our online documentation Web site: http://doc.esker.com/vsi61"
echons ""
echons "Thank you for using $PKGNAME!"

exit 0
