#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	configure the server
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# add a directory to the path
#
# Note that the directory is ALWAYS added to the end.
#
addpath ()
{
	dir="$1"

	sep=":"
	arg=""
	if in_windows
	then
		sep=";"
		arg="-i"
	fi

	foo=`echo "$PATH" | egrep $arg "^$dir$sep|$sep$dir$sep|$sep$dir$"`
	if [ $? -eq 0 ]
	then
		echo "$PATH"
	else
		echo "$PATH$sep$dir"
	fi

	unset dir sep arg foo
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# get the hostname from "hostname" then "uname -n"
#
get_hostname ()
{
	for dir in /bin /usr/bin /usr/sbin /usr/ucb
	do
		if [ -x $dir/hostname ]
		then
			$dir/hostname
			return
		fi
	done

	uname -n
}

# ------------------------------------------------------------------------
# check if a boot entry exists
#
check_boot ()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# check if partial install OK
	#
	PART=no
	if [ x"$1" = x"-p" ]
	then
		PART=yes
		shift
	fi

	# ----------------------------------------------------------------
	# get name to look for
	#
	NAME=$1

	# ----------------------------------------------------------------
	# Check for Linux
	#
	if [ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
	then
		if [ "$PART" = "no" ]
		then
			[ -x /etc/rc.d/rc2.d/S99$NAME -a \
			  -x /etc/rc.d/rc2.d/K01$NAME ]
		else
			[ -x /etc/rc.d/rc2.d/S99$NAME -o \
			  -x /etc/rc.d/rc2.d/K01$NAME -o \
			  -x /etc/rc.d/rc0.d/K01$NAME ]
		fi
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for BSD
	#
	if [ -f /etc/rc.local ]
	then
		grep $NAME /etc/rc.local >/dev/null 2>&1
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for standard System V
	#
	if [ -d /etc/rc2.d ]
	then
		if [ "$PART" = "no" ]
		then
			[ -x /etc/rc2.d/S99$NAME -a \
			  -x /etc/rc2.d/K01$NAME ]
		else
			[ -x /etc/rc2.d/S99$NAME -o \
			  -x /etc/rc2.d/K01$NAME -o \
			  -x /etc/rc0.d/K01$NAME ]
		fi
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for OSF or HP
	#
	if [ -d /sbin/rc2.d ]
	then
		if [ "$PART" = "no" ]
		then
			[ -x /sbin/rc2.d/S99$NAME -a \
			  -x /sbin/rc2.d/K01$NAME ]
		else
			[ -x /sbin/rc2.d/S99$NAME -o \
			  -x /sbin/rc2.d/K01$NAME -o \
			  -x /sbin/rc0.d/K01$NAME ]
		fi
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for old System V
	#
	if [ -d /etc/rc.d/7 ]
	then
		[ -x /etc/rc.d/7/S99$NAME ]
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for AIX
	#
	if [ -f /etc/rc ]
	then
		grep $NAME.sh /etc/rc >/dev/null 2>&1
		return $?
	fi

	# ----------------------------------------------------------------
	# Cannot determine auto-boot location
	#
	return 1
}

# ------------------------------------------------------------------------
# Set a boot entry for auto-start/stop
#
set_boot ()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# get options
	#
	NAME=$1
	OPTION=$2

	SCRIPT="$VSIFAX/lbin/$NAME.sh"

	# ----------------------------------------------------------------
	# Check for Linux
	#
	if [ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
	then
	     rm -f /etc/rc.d/rc5.d/S99$NAME
	     rm -f /etc/rc.d/rc3.d/S99$NAME
	     rm -f /etc/rc.d/rc2.d/S99$NAME
	     rm -f /etc/rc.d/rc6.d/K01$NAME
	     rm -f /etc/rc.d/rc5.d/K01$NAME
	     rm -f /etc/rc.d/rc3.d/K01$NAME
	     rm -f /etc/rc.d/rc2.d/K01$NAME
	     rm -f /etc/rc.d/rc1.d/K01$NAME
	     rm -f /etc/rc.d/rc0.d/K01$NAME	# TEMP hack

	     if [ "$OPTION" = "on" ]
	     then
	      [ -d /etc/rc.d/rc5.d ] && ln -s $SCRIPT /etc/rc.d/rc5.d/S99$NAME
	      [ -d /etc/rc.d/rc3.d ] && ln -s $SCRIPT /etc/rc.d/rc3.d/S99$NAME
	      [ -d /etc/rc.d/rc2.d ] && ln -s $SCRIPT /etc/rc.d/rc2.d/S99$NAME
	      [ -d /etc/rc.d/rc6.d ] && ln -s $SCRIPT /etc/rc.d/rc6.d/K01$NAME
	      [ -d /etc/rc.d/rc5.d ] && ln -s $SCRIPT /etc/rc.d/rc5.d/K01$NAME
	      [ -d /etc/rc.d/rc3.d ] && ln -s $SCRIPT /etc/rc.d/rc3.d/K01$NAME
	      [ -d /etc/rc.d/rc2.d ] && ln -s $SCRIPT /etc/rc.d/rc2.d/K01$NAME
	      [ -d /etc/rc.d/rc1.d ] && ln -s $SCRIPT /etc/rc.d/rc1.d/K01$NAME
	      [ -d /etc/rc.d/rc0.d ] && ln -s $SCRIPT /etc/rc.d/rc0.d/K01$NAME
	     fi

	     return 0
	fi

	# ----------------------------------------------------------------
	# Check for BSD
	#
	if [ -f /etc/rc.local ]
	then
		cp /etc/rc.local /etc/rc.local-
		sed -e "/$NAME/d" /etc/rc.local- > /etc/rc.local
		if [ "$OPTION" = "on" ]
		then
			echo "$SCRIPT start" >> /etc/rc.local
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for standard System V
	#
	if [ -d /etc/rc2.d ]
	then
		rm -f /etc/rc2.d/S99$NAME
		rm -f /etc/rc2.d/K01$NAME
		rm -f /etc/rc0.d/K01$NAME	# TEMP hack

		if [ "$OPTION" = "on" ]
		then
			ln -s $SCRIPT /etc/rc2.d/S99$NAME
			ln -s $SCRIPT /etc/rc2.d/K01$NAME
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for OSF or HP
	#
	if [ -d /sbin/rc2.d ]
	then
		rm -f /sbin/rc2.d/S99$NAME
		rm -f /sbin/rc2.d/K01$NAME
		rm -f /sbin/rc0.d/K01$NAME	# TEMP hack

		if [ "$OPTION" = "on" ]
		then
			ln -s $SCRIPT /sbin/rc2.d/S99$NAME
			ln -s $SCRIPT /sbin/rc2.d/K01$NAME
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for old System V
	#
	if [ -d /etc/rc.d/7 ]
	then
		if [ "$OPTION" = "on" ]
		then
			ln -s $SCRIPT /etc/rc.d/7/S99$NAME
		else
			rm -f /etc/rc.d/7/S99$NAME
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for AIX
	#
	if [ -f /etc/rc ]
	then
		cp /etc/rc /etc/rc-
		sed -e "/$NAME/d" -e '/^exit 0/d' /etc/rc- > /etc/rc
		if [ "$OPTION" = "on" ]
		then
			echo "$SCRIPT start" >> /etc/rc
		fi
		echo "exit 0" >> /etc/rc

		return 0
	fi

	# ----------------------------------------------------------------
	# Cannot determine auto-boot location
	#
	echoerr "Cannot determine auto-start at boot time procedure"
	return 1
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# get Windows directory
#
get_windir ()
{
	{
		if [ "$windir" != "" ]
		then
			echo "$windir"

		elif [ "$WINDIR" != "" ]
		then
			echo "$WINDIR"

		elif [ "$SystemRoot" != "" ]
		then
			echo "$SystemRoot"

		elif [ "$SYSTEMROOT" != "" ]
		then
			echo "$SYSTEMROOT"

		elif [ -d "c:/windows" ]
		then
			echo "c:/windows"

		elif [ -d "c:/winnt" ]
		then
			echo "c:/winnt"

		else
			echo "c:/windows"
		fi
	} | sed -e 's,\\\\*,/,g'
}

# ------------------------------------------------------------------------
# get Windows drive
#
get_windrive()
{
	get_windir | sed -e 's,[/\\].*,,'
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get response dir/file info
#
response_info ()
{
	if in_windows
	then
		RESPONSE_DIR="`get_windrive`/Program Files/Common Files/VSI-FAX Shared"
	else
		RESPONSE_DIR="/var/spool/pkg"
	fi
	RESPONSE_FILE="`prdname`-install.log"

	echo "RESPONSE_DIR=\"$RESPONSE_DIR\""
	echo "SAVE_RESPONSE_FILE=\"$RESPONSE_DIR/$RESPONSE_FILE\""
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -r     read response file"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D     debug install"
	echo "  -x     remove any configurations"
}

# ------------------------------------------------------------------------
# configuration procedure
#
start_config()
{
	QNAME="$1"
	DEVNAME="$2"
	FIMNAME="$3"
	DEFAULT="$4"
	COMMENTS="$5"

	echo ""
	echonl "Configuring modem ."

	if [ "$DEVNAME" != "" ]
	then
		in_windows || chmod 666 $DEVNAME
		DEVNAME="-v $DEVNAME"
	fi

	vfxadmin device -q -a $DEFAULT $DEVNAME \
		-i $FIMNAME -n "$COMMENTS" $QNAME ||
		{ echo ""; return 1; }

	echo ". done."

	return 0
}

# ------------------------------------------------------------------------
# configure any licenses
#
configure_licenses ()
{
	# ----------------------------------------------------------------
	# bail if this is run via the standard install
	#
	[ "$INSTALL_MODE" != "" ] && return 0

	vfxadmin license -s -D ||
	{
		echo ""
		echo "Please enter your Server license information."
		vfxadmin license
	}

	return 0
}

# ------------------------------------------------------------------------
# get configuration parameters
#
auto_config()
{
	# Get the type of fim to use

	DEFAULT_FIM=cx
	FIMNAME=cx

	if [ "$FIMNAME" = "" ]
	then
		num_fims=`vfxadmin listfims -h -d | wc -l`
		[ $num_fims -eq 1 ] && FIMNAME=$DEFAULT_FIM
	fi

	if [ "$FIMNAME" = "" ]
	then
		echo ""
		echo "$PKGNAME currently supports the following modem types:"
		echo ""
		vfxadmin listfims -d

		fim_list=`vfxadmin listfims -h | sed 's/ .*//'`

		while true
		do
			echo ""
			echo "Please select modem type or press q to quit."

			echonl "Modem type [$DEFAULT_FIM] : "
			response=`read_user_input "modem-type-$modem_count"`

			if [ "$response" = "" ]
			then
				response=$DEFAULT_FIM
			fi

			case $response in
			q* | Q*)
				echo "Cancelled. Device not configured."
				return 1
				;;
			esac

			FIM_OK=no
			for i in $fim_list
			do
				if [ "$response" = "$i" ]
				then
					FIM_OK=yes
					break
				fi
			done

			if [ "$FIM_OK" = "no" ]
			then
				echo "Invalid modem type."
				continue
			fi

			echo "Modem type is \"$response\""
			if is_ok "modem-type-$modem_count-ok"
			then
				FIMNAME=$response
				break
			fi
		done
	fi

	# ----------------------------------------------------------------
	# Get the device name to use
	#
	case "$FIMNAME" in
	lb|sm|hold|er|et)	DEVNAME="" ;;
	*)			DEVNAME=/  ;;
	esac

	if [ "$DEVNAME" != "" ]
	then
		echo ""
		echo "Please enter the pathname of the device to which the"
		echo "modem is attached or q to quit."

		while true
		do
			echo ""
			echonl "Enter device pathname: "
			DEVNAME=`read_user_input "device-$modem_count"`

			if [ "$DEVNAME" = "" ]
			then
				continue
			fi

			if [ "$DEVNAME" = "q" -o "$DEVNAME" = "Q" ]
			then
				echo "Cancelled. Device not configured."
				return 1
			fi

			echo "Device is \"$DEVNAME\""
			is_ok "dev-name-$modem_count-ok" || continue

			in_windows ||
			if [ "$FIMNAME" = "c2" -o "$FIMNAME" = "cx" ]
			then
				if [ ! -c $DEVNAME ]
				then
					echo "$DEVNAME is not a valid device pathname."
					echo "The device must be of type \"character special\"."
					continue
				fi
			fi

			break
		done
	fi

	num=`expr $modem_count + 1`
	qname=fax$num
	echo ""
	echo "Enter the destination name you would like to use"
	echo "to refer to this device."

	while true
	do
		echo ""
		echonl "Destination [$qname] : "
		QNAME=`read_user_input "qname-$modem_count"`

		if [ "$QNAME" = "" ]
		then
			QNAME=$qname
		fi

		QNAME="`echo $QNAME | cut -c1-8`"

		echo "Destination is \"$QNAME\""
		is_ok "dest-name-$modem_count-ok" && break
	done

	echo ""
	echo "Enter any comments (e.g. phone number) for this device."

	COMMENTS=""
	while true
	do
		echo ""
		echonl "Comments [$COMMENTS] : "
		comments=`read_user_input "comments-$modem_count"`

		[ "$comments" = "" ] && comments="$COMMENTS"
		COMMENTS="$comments"

		echo "Comments: \"$COMMENTS\""
		is_ok "comments-$modem_count-ok" && break
	done

	DEFAULT=""
	if [ $modem_count -eq 0 ]
	then
		DEFAULT="-d"
	else
		default_name=`vfxadmin default`
		if [ "$default_name" != "" ]
		then
			echo ""
			echo "The current default destination is $default_name."
			if getyesorno -n "default-$QNAME" \
				"Do you want $QNAME to be the default?"
			then
				DEFAULT=-d
			fi
		else
			DEFAULT=-d
		fi
	fi

	start_config "$QNAME" "$DEVNAME" "$FIMNAME" "$DEFAULT" "$COMMENTS"
}

# ------------------------------------------------------------------------
# do device configuration
#
configure_devices ()
{
	# ----------------------------------------------------------------
	# Check if the fxserver license is a demo license
	#
	vfxadmin license -D -s
	if [ $? -ne 0 ]
	then
		return 0
	fi

	# ----------------------------------------------------------------
	# Check to see if devices are available.
	#
	echo ""
	echo "Configuring devices"

	modem_count=0

	modem_count=`vfxadmin list -d | wc -l | sed -e 's/ //g'`
	if [ $modem_count -eq 0 ]
	then
		echo ""
		echo "Note: If you choose not to configure any modems,"
		echo "you can still send faxes to the originating user account"
		echo "via the \"LoopBack\" device (lb)"
	else
		echo ""
		echo "The following devices are already configured:"
		vfxadmin list -d |
		while read line
		do
			eval $line
			echo "$dest using $fim FIM on $device"
		done
	fi

	while true
	do
		if [ $modem_count -eq 0 ]
		then
			msg="Would you like to configure your first modem?"
			default=-y
		else
			msg="Would you like to configure another modem?"
			default=-n
		fi

		echo ""
		getyesorno $default "add-modem-$modem_count" "$msg" || break

		auto_config
		if [ $? -eq 0 ]
		then
			modem_count=`expr $modem_count + 1`
			export modem_count
		fi
	done

	# ----------------------------------------------------------------
	# check for any eicon processing
	#
	if [ -x $VSIFAX/lbin/fim-ei ]
	then
		chans=`fim-ei -X lines=? | \
                awk -F 'channels=' 'BEGIN {sum = 0} {sum+=$2} END { print sum}'`
		if [ $chans -ne 0 ]
		then
			count=`vfxadmin list -d | grep fim=ei | wc -l`
			if [ $chans -ne $count ]
			then
				echo ""
				getyesorno -y "add-eicon" \
			"Do you want to auto-configure your Dialogic Diva boards?" &&
				fim-ei -X "verbose=yes;setup=ch;class=eicon"
			fi
		fi
	fi
}

# ------------------------------------------------------------------------
# configure_location
#
configure_location ()
{
	echo ""
	echo "Setting location parameters"

	OUR_HOST=`get_hostname`

	# ----------------------------------------------------------------
	# get TSI string
	#
	TSI=`vfxadmin config -slq DEVICE TSIstring`
	if [ $? -ne 0 ]
	then
echo ""
echo "The TSI (Transmitting Station Identification) is a character string"
echo "sent to the remote fax device at the start of a transmission."
echo "It should be set to your company name or your return fax number."
echo "Maximum length is 20 characters."

		while true
		do
			echo ""
			echonl "Enter your TSI or <ENTER> for none: "
			TSI=`read_user_input "tsi"`

			TSI="`echo $TSI | cut -c1-20`"
			echo "TSI string is \"$TSI\""
			is_ok "tsi-ok" && break
		done

		vfxadmin config -sa DEVICE TSIstring "$TSI"
	fi

	# ----------------------------------------------------------------
	# get CSI string
	#
	CSI=`vfxadmin config -slq DEVICE CSIstring`
	if [ $? -ne 0 ]
	then
echo ""
echo "The CSI (Called Station Identifier) is a character string"
echo "sent to the remote fax device at the start of receiving."
echo "It is normally the same as your TSI."
echo "Maximum length is 20 characters."

		while true
		do
			echo ""
			echonl "Enter your CSI [$TSI] : "
			CSI=`read_user_input "csi"`

			CSI="`echo $CSI | cut -c1-20`"
			[ "$CSI" = "" ] && CSI="$TSI"

			echo "CSI string is \"$CSI\""
			is_ok "csi-ok" && break
		done

		vfxadmin config -sa DEVICE CSIstring "$CSI"
	fi

	# ----------------------------------------------------------------
	# get SMTP server
	#
	SMTP=`vfxadmin config -slq VMAIL Server`
	if [ $? -ne 0 ]
	then
		echo ""
		echo "You must specify the name of your SMTP (mail) server"
		echo "in order for the SendMail FIM and mail notification"
		echo "to work."

		SMTP="$OUR_HOST"
		while true
		do
			echo ""
			echonl "Enter your SMTP server [$SMTP] : "
			ans=`read_user_input "smtp-server"`

			[ "$ans" = "" ] && ans=$SMTP
			SMTP=$ans

			echo "SMTP server is \"$SMTP\""
			is_ok "smtp-ok" || continue

			[ "$SMTP" != "" ] && break

			echo ""
			echo "If you do not specify a SMTP server, the SendMail"
			echo "FIM will not function and mail notification cannot"
			echo "be done."
			is_ok "no-smtp-ok" && break
		done

		[ "$SMTP" != "" ] && vfxadmin config -sa VMAIL Server "$SMTP"
	fi

	# ----------------------------------------------------------------
	# get domain for mail server if Sender field is not already set
	#
	SENDER=`vfxadmin config -slq VMAIL Sender`
	if [ $? -ne 0 ]
	then
		echo ""
		echo "Enter the domain name for your mail server."

		DOMAIN=""
		while true
		do
			echo ""
			echonl "Domain [$DOMAIN] : "
			ans=`read_user_input "domain-name"`

			[ "$ans" = "" ] && ans=$DOMAIN
			DOMAIN=$ans

			echo "Domain is \"$DOMAIN\""
			is_ok "domain-ok" && break
		done

		SENDER="VsiFaxServer"
		[ "$DOMAIN" != "" ] && SENDER="$SENDER@$DOMAIN"
		vfxadmin config -sa VMAIL Sender "$SENDER"
	fi
}

# ------------------------------------------------------------------------
# process configuration
#
configure_dialstrings ()
{
	echo ""
	echo "Setting dial-string conversion parameters"

	# ----------------------------------------------------------------
	# get dial-string conversion parameters
	#
	country=`vfxadmin config -slq DEVICE CountryCode`
	if [ "$country" = "" ]
	then
		echo ""
		if getyesorno "in-usa" "Are you in the United States or Canada?"
		then
			country=1
			vfxadmin config -sa DEVICE CountryCode    "$country"
			vfxadmin config -sa DEVICE LongDistAccess "1"
		else
			country=""
			while true
			do
				echo ""
				echonl "Enter your country code [$country] : "
				ans=`read_user_input "country-code"`

				[ "$ans" = "" ] && ans="$country"
				[ "$ans" = "" ] && continue

				echo "Country code is \"$ans\""
				is_ok "country-code-ok" && break
			done
			country="$ans"

			vfxadmin config -sa DEVICE CountryCode "$country"
		fi
	fi

	areacode=`vfxadmin config -slq DEVICE AreaCode`
	if [ "$areacode" = "" ]
	then
		while true
		do
			echo ""
			if [ "$country" = "1" ]
			then
				echonl "Enter your area code [$areacode] : "
			else
				echonl "Enter your city code [$areacode] : "
			fi
			ans=`read_user_input "city-code"`

			[ "$ans" = "" ] && ans="$areacode"
			[ "$ans" = "" ] && continue

			if [ "$country" = "1" ]
			then
				echo "Area code is \"$ans\""
			else
				echo "City code is \"$ans\""
			fi
			is_ok "city-code-ok" && break
		done
		areacode="$ans"

		vfxadmin config -sa DEVICE AreaCode "$areacode"
	fi

	if [ "$country" != "1" ]
	then
		intl=`vfxadmin config -slq DEVICE IntlAccess`
		if [ "$intl" = "" ]
		then
			while true
			do
			echo ""
			echonl "Enter your international access code [$intl] : "
			ans=`read_user_input "intl-access"`

				[ "$ans" = "" ] && ans="$intl"
				[ "$ans" = "" ] && continue

				echo "International access code is \"$ans\""
				is_ok "intl-access-ok" && break
			done
			intl="$ans"

			vfxadmin config -sa DEVICE IntlAccess "$intl"
		fi

		ld=`vfxadmin config -slq DEVICE LongDistAccess`
		if [ "$ld" = "" ]
		then
			ld="0"
			while true
			do
				echo ""
			echonl "Enter your long distance access code [$ld] : "
			ans=`read_user_input "ld-access"`

				[ "$ans" = "" ] && ans="$ld"
				[ "$ans" = "" ] && continue

				echo "Long distance access code is \"$ans\""
				is_ok "ld-access-ok" && break
			done
			ld="$ans"

			vfxadmin config -sa DEVICE LongDistAccess "$ld"
		fi

		localnumlen=`vfxadmin config -slq DEVICE LocalNumLen`
		if [ "$localnumlen" = "" ]
		then
			while true
			do
				echo ""
	echonl "Enter number of digits in local number [$localnumlen] : "
			ans=`read_user_input "local-len"`

				[ "$ans" = "" ] && ans="$localnumlen"
				[ "$ans" = "" ] && continue

				echo "Number of digits is \"$ans\""
				is_ok "local-len-ok" && break
			done
			localnumlen="$ans"

			vfxadmin config -sa DEVICE LocalNumLen "$localnumlen"
		fi
	else
		vfxadmin config -sn DEVICE IntlAccess     011
		vfxadmin config -sn DEVICE LongDistAccess 1
		vfxadmin config -sn DEVICE LocalNumLen    7
	fi

	prefix=`vfxadmin config -slq DEVICE DialPrefix`
	if [ $? -ne 0 ]
	then
		while true
		do
			echo ""
			echonl "Enter local dial prefix if needed [$prefix] : "
			ans=`read_user_input "local-prefix"`

			[ "$ans" = "" ] && ans="$prefix"

			echo "Local dial prefix is \"$ans\""
			is_ok "local-prefix-ok" && break
		done
		prefix="$ans"

		vfxadmin config -sa DEVICE DialPrefix "$prefix"

		if [ "$prefix" != "" ]
		then
			ldprefix=`vfxadmin config -slq DEVICE LongDistPrefix`
			if [ $? -ne 0 ]
			then
				ldprefix="$prefix"
				while true
				do
					echo ""
	echonl "Enter long-distance dial prefix if needed [$ldprefix] : "
					ans=`read_user_input "ld-prefix"`

					[ "$ans" = "" ] && ans="$ldprefix"

					echo "Long-distance dial prefix is \"$ans\""
					is_ok "ld-prefix-ok" && break
				done
				ldprefix="$ans"

				vfxadmin config -sa DEVICE LongDistPrefix "$ldprefix"

			fi
		fi
	fi

	vfxadmin config -sn DEVICE DialConvert on
}

# ------------------------------------------------------------------------
# configure VPOPD
#
configure_vpopd ()
{
	# ----------------------------------------------------------------
	# If user specified no VPOPD, skip this section
	#
	RUN_VPOPD=`vfxadmin config -slqb VFXSCHED RunVpopd`
	[ "$RUN_VPOPD" = "0" ] && return 0

	# ----------------------------------------------------------------
	# check if email gateway already setup
	#
	POP_SERVER=`vfxadmin config -slq VPOPD Hostname`
	POP_USERNM=`vfxadmin config -slq VPOPD Username`

	[ "$POP_SERVER" != "" -a "$POP_USERNM" != "" ] && return 0

	# ----------------------------------------------------------------
	# ask about email gateway
	#
	vfxadmin config -sn VFXSCHED RunVpopd off

	echo ""
	getyesorno "setup-email" "Do you want to setup an Email gateway?" ||
		return 0

	# ----------------------------------------------------------------
	# give user chance to bail out
	#
	echo ""
	echo "You must know the address of your POP server and"
	echo "the name & password of the account to be used."
	echo "This account should NOT be used for any other purpose."
	echo ""

	getyesorno "proceed-email" "Do you want to proceed?" || return 0

	# ----------------------------------------------------------------
	# get POP server name
	#
	if [ "$POP_SERVER" = "" ]
	then
		echo ""
		echo "Enter your POP mail server address or Q to quit."

		while true
		do
			echo ""
			echonl "POP server: [$POP_SERVER] : "
			ans=`read_user_input "pop-server"`

			[ "$ans" = "" ] && ans="$POP_SERVER"
			POP_SERVER="$ans"

			if [ "$POP_SERVER" = "" ]
			then
				echo ""
				echo "Empty POP server name."
				continue
			fi

			case "$POP_SERVER" in
			q* | Q*)	return 0;;
			esac

			echo "POP server is \"$POP_SERVER\""
			is_ok "pop-server-ok" && break
		done

		vfxadmin config -sa VPOPD Hostname "$POP_SERVER"
	fi

	# ----------------------------------------------------------------
	# get POP account
	#
	if [ "$POP_USERNM" = "" ]
	then
		POP_USERNM=autofax

		echo ""
		echo "Enter the POP mail account to be used or Q to quit."

		while true
		do
			echo ""
			echonl "POP account [$POP_USERNM] : "
			ans=`read_user_input "pop-account"`

			[ "$ans" = "" ] && ans="$POP_USERNM"
			POP_USERNM="$ans"

			if [ "$POP_USERNM" = "" ]
			then
				echo ""
				echo "Empty POP user name."
				continue
			fi

			case "$POP_USERNM" in
			q | Q)	return 0;;
			esac

			echo "POP account is \"$POP_USERNM\""
			is_ok "pop-account-ok" && break
		done

		vfxadmin config -sa VPOPD Username "$POP_USERNM"
	fi

	# ----------------------------------------------------------------
	# get POP password
	#
	POP_PASSWD=`vfxadmin config -slq VPOPD Password`

	if [ "$POP_PASSWD" = "" ]
	then
		echo ""
		echo "Enter the password for the POP account."
		echo ""

		vfxadmin config -saep VPOPD Password || return 0
	fi

	# ----------------------------------------------------------------
	# store final entries in config file
	#
	vfxadmin config -sa VFXSCHED RunVpopd on
}

# ------------------------------------------------------------------------
# dump info
#
dump_info ()
{
	echo ""
	echo "Company name       : $COMPANY"
	echo "Voice phone number : $VPHONE"
	echo "Fax phone number   : $FPHONE"
	echo "E-mail address     : $EMAIL"
	echo "Address 1          : $ADDR1"
	echo "Address 2          : $ADDR2"
	echo "Address 3          : $ADDR3"
	echo "Country            : $COUNTRY"
	echo ""
}

# ------------------------------------------------------------------------
# confirm info
#
confirm ()
{
	dump_info

	is_ok "company-info-ok"
}

# ------------------------------------------------------------------------
# configure cover info
#
configure_covers ()
{
	COMPANY=`vfxadmin config -slq USERINFO Company`
	VPHONE=` vfxadmin config -slq USERINFO Vphone`
	FPHONE=` vfxadmin config -slq USERINFO Fphone`
	EMAIL=`  vfxadmin config -slq USERINFO Email`
	ADDR1=`  vfxadmin config -slq USERINFO Addr1`
	ADDR2=`  vfxadmin config -slq USERINFO Addr2`
	ADDR3=`  vfxadmin config -slq USERINFO Addr3`
	COUNTRY=`vfxadmin config -slq USERINFO Country`

	FOO="$COMPANY"
	CHANGED=no

	# ----------------------------------------------------------------
	# tell user what we are doing
	#
	if [ "$FOO" = "" ]
	then
		echo "The following information is needed for updating supplied"
		echo "coversheets:"
		echo ""
		echo "Company name"
		echo "Voice phone number"
		echo "Fax phone number"
		echo "E-mail address"
		echo "Address (3 lines)"
		echo "Country"
		echo ""
	else
		echo "The following is your current coversheet information:"
	fi

	# ----------------------------------------------------------------
	# get info from user
	#
	while true
	do
		if [ "$FOO" != "" ]
		then
			confirm && break
		fi
		FOO=foo

		echo ""
		echo "Enter new information:"
		echo ""

		echonl "Company     [$COMPANY] : "
		ans=`read_user_input "company-name"`
		if [ "$ans" != "" ]
		then
			COMPANY="$ans"
			CHANGED=yes
		fi

		echonl "Voice phone [$VPHONE] : "
		ans=`read_user_input "company-voice-phone"`
		if [ "$ans" != "" ]
		then
			VPHONE="$ans"
			CHANGED=yes
		fi

		echonl "Fax phone   [$FPHONE] : "
		ans=`read_user_input "company-fax-phone"`
		if [ "$ans" != "" ]
		then
			FPHONE="$ans"
			CHANGED=yes
		fi

		echonl "E-mail      [$EMAIL] : "
		ans=`read_user_input "company-email"`
		if [ "$ans" != "" ]
		then
			EMAIL="$ans"
			CHANGED=yes
		fi

		echonl "Address 1   [$ADDR1] : "
		ans=`read_user_input "company-addr-1"`
		if [ "$ans" != "" ]
		then
			ADDR1="$ans"
			CHANGED=yes
		fi

		echonl "Address 2   [$ADDR2] : "
		ans=`read_user_input "company-addr-2"`
		if [ "$ans" != "" ]
		then
			ADDR2="$ans"
			CHANGED=yes
		fi

		echonl "Address 3   [$ADDR3] : "
		ans=`read_user_input "company-addr-3"`
		if [ "$ans" != "" ]
		then
			ADDR3="$ans"
			CHANGED=yes
		fi

		echonl "Country     [$COUNTRY] : "
		ans=`read_user_input "company-country"`
		if [ "$ans" != "" ]
		then
			COUNTRY="$ans"
			CHANGED=yes
		fi
	done

	# ----------------------------------------------------------------
	# now apply changes if any made
	#
	if [ "$CHANGED" = "yes" ]
	then
		# --------------------------------------------------------
		# write info back
		#
		vfxadmin config -sa USERINFO Company "$COMPANY"
		vfxadmin config -sa USERINFO Vphone  "$VPHONE"
		vfxadmin config -sa USERINFO Fphone  "$FPHONE"
		vfxadmin config -sa USERINFO Email   "$EMAIL"
		vfxadmin config -sa USERINFO Addr1   "$ADDR1"
		vfxadmin config -sa USERINFO Addr2   "$ADDR2"
		vfxadmin config -sa USERINFO Addr3   "$ADDR3"
		vfxadmin config -sa USERINFO Country "$COUNTRY"
	fi

	# ----------------------------------------------------------------
	# always update covers (for thumbnails)
	#
	echo ""
	echo "Updating installed covers with user information"
	echo ""

	vfxadmin cover -v -f all
}

# ------------------------------------------------------------------------
# check if server set to start at boot time
#
# Note: The old name was "vsifax3", the new name is "vsifax", so we
# automatically change it if we find the old entry.
#
check_boot_server()
{
	# ----------------------------------------------------------------
	# check if old name found
	#
	if check_boot -p vsifax3
	then
		set_boot vsifax3 off
		set_boot vsifax  on
		return 0
	fi

	# ----------------------------------------------------------------
	# now check new name
	#
	check_boot vsifax
}

# ------------------------------------------------------------------------
# check if admin set to start at boot time
#
# Note: The old name was "vsiadmin", the new name is "vsiadm", so we
# automatically change it if we find the old entry.
#
check_boot_admin()
{
	# ----------------------------------------------------------------
	# check if old name found
	#
	if check_boot -p vsiadmin
	then
		set_boot vsiadmin off
		set_boot vsiadm   on
		return 0
	fi

	# ----------------------------------------------------------------
	# now check new name
	#
	check_boot vsiadm
}

# ------------------------------------------------------------------------
# set server auto-start at boot time
#
set_boot_server()
{
	set_boot vsifax $1
}

# ------------------------------------------------------------------------
# set admin auto-start at boot time
#
set_boot_admin()
{
	set_boot vsiadm $1
}

# ------------------------------------------------------------------------
# setup auto-start at boot
#
configure_autostart_server()
{
	if check_boot_server
	then
		:
	else
		echo ""
		echo "Setting the $PKGNAME Fax Server to start" \
			"automatically at system boot time."
		is_ok "auto-start-server-ok" && set_boot_server on
	fi
}

# ------------------------------------------------------------------------
# setup auto-start at boot
#
configure_autostart_admin()
{
	if check_boot_admin
	then
		:
	else
		echo ""
		echo "Setting the $PKGNAME Admin Server to start" \
			"automatically at system boot time."
		is_ok "auto-start-admin-ok" && set_boot_admin on
	fi
}

# ------------------------------------------------------------------------
# setup auto-start at boot
#
configure_autostart()
{
	if [ "$1" = "on" ]
	then
		if in_windows
		then
			:
		else
			configure_autostart_admin
			configure_autostart_server
		fi
	else
		if in_windows
		then
			vfxadmin server remove
		else
			set_boot_server off
			set_boot_admin  off
		fi
	fi
}

# ------------------------------------------------------------------------
# main()
#
PATH=`addpath "$VSIFAX/bin"`
PATH=`addpath "$VSIFAX/lbin"`

PKGNAME=`pkgname`

eval `response_info`

# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

DEBUG=
DELETE=no

case "$1" in
	'-?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | -helpall | --helpall)
		helpall
		exit 0
		;;

	'-V' | -version | --version)
		echo "$PGM: `vfxadmin version`"
		exit 0
		;;
esac

ARGS=`getopt "Drx" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-x
		shift
		;;

	-r)
		LOAD_RESPONSE_FILE="$SAVE_RESPONSE_FILE"
		shift
		;;

	-x)
		DELETE=yes
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

export SAVE_RESPONSE_FILE
export LOAD_RESPONSE_FILE

if [ "$SAVE_RESPONSE_FILE" != "" ]
then
	[ ! -d "$RESPONSE_DIR" ] && mkdir -p "$RESPONSE_DIR"
	touch "$SAVE_RESPONSE_FILE"
	chmod 664 "$SAVE_RESPONSE_FILE"
fi

[ "$DEBUG" != "" ] && set $DEBUG

# ------------------------------------------------------------------------
# now do/undo all configurations
#
if [ "$DELETE" = "no" ]
then
	configure_licenses
	configure_devices
	configure_location
	configure_dialstrings
	configure_vpopd
	configure_covers
	configure_autostart on
else
	configure_autostart off
fi
