#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	front-end to the install scripts
#
#	usage:
#
#	install.sh [-Dvqsnr] [-i platform] [-p <pkg>]
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# output a prompt & get a response from the user
#
get_user_input ()
{
	id="$1"
	dflt="$2"
	msg="$3"

	ok_id="$id-ok"
	[ "$id" = "" ] && ok_id=""

	CONSOLE=`get_dev_tty`

	while true
	do
		if [ x"$dflt" = x"!" ]
		then
			echonl "Enter $msg: "
		else
			echonl "Enter $msg [$dflt]: "
		fi >$CONSOLE

		response=`read_user_input "$id"`
		[ "$response" = "" -a x"$dflt" != x"!" ] && response="$dflt"

		echo "$msg = \"$response\"" >$CONSOLE
		is_ok "$ok_id" >$CONSOLE && { echo "$response"; break; }

		dflt="$response"
	done
}

# ------------------------------------------------------------------------
# look for a program in the path
#
look_for_pgm ()
{
	pgm=$1

	l=`echo $PATH | sed -e 's/:/ /g'`
	for j in $l
	do
		[ -d $j -a -x $j/$pgm ] && return 0
	done

	return 1
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# get Windows directory
#
get_windir ()
{
	{
		if [ "$windir" != "" ]
		then
			echo "$windir"

		elif [ "$WINDIR" != "" ]
		then
			echo "$WINDIR"

		elif [ "$SystemRoot" != "" ]
		then
			echo "$SystemRoot"

		elif [ "$SYSTEMROOT" != "" ]
		then
			echo "$SYSTEMROOT"

		elif [ -d "c:/windows" ]
		then
			echo "c:/windows"

		elif [ -d "c:/winnt" ]
		then
			echo "c:/winnt"

		else
			echo "c:/windows"
		fi
	} | sed -e 's,\\\\*,/,g'
}

# ------------------------------------------------------------------------
# get Windows drive
#
get_windrive()
{
	get_windir | sed -e 's,[/\\].*,,'
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get response dir/file info
#
response_info ()
{
	if in_windows
	then
		RESPONSE_DIR="`get_windrive`/Program Files/Common Files/VSI-FAX Shared"
	else
		RESPONSE_DIR="/var/spool/pkg"
	fi
	RESPONSE_FILE="`prdname`-install.log"

	echo "RESPONSE_DIR=\"$RESPONSE_DIR\""
	echo "SAVE_RESPONSE_FILE=\"$RESPONSE_DIR/$RESPONSE_FILE\""
}

# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# package list
#
# "-n"		display package names only
# "-l"		display package names which need a license
# "-i"		display package names which need "std" install.sh
# "-d name"	display description for <name>
#
pkglist ()
{
	OPT="$1"
	[ "$OPT" = "" ] && OPT="-a"

	# ----------------------------------------------------------------
	# process option
	#
	case "$OPT" in

		-n)
			OUT="cut -d: -f2"
			;;

		-l)
			OUT="grep 'lic' | cut -d: -f2"
			;;

		-i)
			OUT="grep 'ins' | cut -d: -f2"
			;;

		-d)
			OUT="grep $2 | cut -d: -f3"
			;;

		-a)
			OUT=cat
			;;
	esac

	# ----------------------------------------------------------------
	# here is the list
	#
	sed -e '/^#/d' -e 's/		*/:/g' <<-EOF | eval $OUT
	#
	#code	name		description		all	lic	ins
	#----	--------	-----------------------	---	---	---
	s	server		Fax Server		all	lic	ins
	v	vfsxml		Virtual Fax Server	no	lic	ins
	a	faxapi		Fax Client APIs		all	lic	ins
	b	bt1034		Dialogic Brooktrout TR1034-SR140 drivers	all	no	ins
	EOF
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -i platform   install for <platform>"
	echo "  -p package    install <package> (\"all\" for all)"
	echo "  -q            quick install"
	echo "  -s            silent install"
	echo "  -n            no delete of install files"
	echo "  -r            read response file"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D            debug install"
	echo "  -v            verbose mode"
}

# ------------------------------------------------------------------------
# platform list
#
# fields are:
#
# 1.  platform description
# 2.  platform name
#
# NOTE: the order of this list is important to make sure platform tests work
#
platform_list ()
{
	sed -e '/^#/d' -e 's/		*/:/g' <<-EOF
	#
	#description		platform
	#------------------	--------
	HP PA/risc (HP/Ux)	hp0
	IBM PowerPC (AIX)	aix
	Intel (OpenServer)	os5
	Intel (UnixWare)	uw7
	Intel (Red Hat Enterprise)	el3
	Intel (Solaris)		6sl
	Sun Sparc (Solaris)	sol
	EOF
}

# ------------------------------------------------------------------------
# platform-specific test routines
#
test_for_aix ()
{
	[ -d /lpp ] || return 1

	return 0
}

test_for_6sl ()
{
	[ -d /boot/solaris ] || return 1

	return 0
}

test_for_dux ()
{
	[ `uname -s` = "OSF1" ] || return 1

	return 0
}

test_for_hp0 ()
{
	[ `uname -s` = "HP-UX" ] || return 1

	return 0
}

test_for_el3 ()
{
# TODO Make sure we are on Enterprise/Advanced 3.0 or 4.0
# egrep -q "(Advanced)|(Enterprise)" /etc/redhat-release


	if [ -f /etc/redhat-release ]
	then
		version_str=`cat /etc/redhat-release | sed  -e 's/.* \(.*\) \((.*)\)$/version=\1 code=\2/' | sed  -e 's/[()]//g'`
		eval $version_str
		case $code in
			Taroon | Nahant | Tikanga )
				return 0
				;;
			* )
				return 1
				;;
		esac
	fi
	return 1
}

test_for_lnx ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ] || return 1

	return 0
}

test_for_sol ()
{
	[ -f /ufsboot  ] && return 0
	[ -d /platform ] && return 0

	return 1
}

test_for_sun ()
{
	[ -f /vmunix ] || return 1

	return 0
}

test_for_os5 ()
{
	[ -d /tcb        -a -d /sbin         ] && return 0

	return 1
}

test_for_uw7 ()
{
	[ -f /stand/unix -a -f /stand/resmgr ] && return 0

	return 1
}

test_for_fsd ()
{
	[ -f /kernel ] || return 1

	return 0
}

test_for_bsd ()
{
	[ -f /netbsd ] || return 1

	return 0
}

test_for_osd ()
{
	[ -f /bsd ] || return 1

	return 0
}

# ------------------------------------------------------------------------
# platform table
#
platform_table ()
{
	i=1
	platform_list |
	while read line
	do
		echo "$i:$line"
		i=`expr $i + 1`
	done
}

# ------------------------------------------------------------------------
# get line from table
#
table_line ()
{
	choice=$1

	grep -i "^$choice:" 2>/dev/null
}

# ------------------------------------------------------------------------
# get platform fields
#
platform_fields ()
{
	SAVE="$IFS"
	IFS=:
	set $1
	IFS="$SAVE"

	echo \
		"PLAT_CODE=\"$1\"" \
		"PLAT_DESC=\"$2\"" \
		"PLAT_NAME=\"$3\""
}

# ------------------------------------------------------------------------
# get package table
#
package_table ()
{
	pkglist |
	while read line
	do
		eval `package_fields "$line"`
		[ -f $PLAT_DIR/$PKG_NAME.tgz -o -f $PLAT_ALL/$PKG_NAME.tgz ] &&
			echo "$line"
	done
}

# ------------------------------------------------------------------------
# get package fields
#
package_fields ()
{
	SAVE="$IFS"
	IFS=:
	set $1
	IFS="$SAVE"

	echo \
		"PKG_CODE=\"$1\"" \
		"PKG_NAME=\"$2\"" \
		"PKG_DESC=\"$3\"" \
		"PKG_IALL=\"$4\""
}

# ------------------------------------------------------------------------
# Get list of all packages available
#
package_all ()
{
	pkglist |
	while read line
	do
		eval `package_fields "$line"`
		if [ "$PKG_IALL" = "all" ]
		then
			[ -f $PLAT_DIR/$PKG_NAME.tgz -o \
			  -f $PLAT_ALL/$PKG_NAME.tgz ] &&
				echo "$PKG_NAME"
		fi
	done
}

# ------------------------------------------------------------------------
# put up platform menu
#
platform_menu ()
{
	echo ""

	echo "Available platforms are:"
	echo ""

	platform_table |
	while read line
	do
		eval `platform_fields "$line"`

		echo "$PLAT_CODE)  $PLAT_DESC"
	done

	echo "Q)  quit"
}

# ------------------------------------------------------------------------
# get default platform
#
get_platform_default ()
{
	found=no

	platform_table |
	while read line
	do
		eval `platform_fields "$line"`

		[ "$found" = "yes" ] && continue

		if test_for_${PLAT_NAME}
		then
			echo \
				"PLAT_CODE=\"$PLAT_CODE\"" \
				"PLAT_DESC=\"$PLAT_DESC\"" \
				"PLAT_NAME=\"$PLAT_NAME\""
			found=yes
		fi
	done
}

# ------------------------------------------------------------------------
# get platform
#
get_platform ()
{
	while true
	do
		$SLEEP

		platform_menu

		CHOICE=`get_user_input "install-platform" "$DEFAULT" \
			"platform to install"`

		case "$CHOICE" in
		q* | Q*)
			abort_install
			;;
		esac

		line="`platform_table | table_line $CHOICE`"
		if [ "$line" = "" ]
		then
			echo "Invalid choice \"$CHOICE\".  Try again."
			continue
		fi

		eval `platform_fields "$line"`

		echo ""
		echo "Installing on \"$PLAT_DESC\" platform."
		is_ok "install-on-plat" || continue

		if [ "$DEFAULT" != "" -a "$CHOICE" != "$DEFAULT" ]
		then
			echo ""
			getyesorno -n "not-native" \
				"Not native platform. Continue?" ||
				continue
		fi

		break
	done
}

# ------------------------------------------------------------------------
# put up package menu
#
package_menu ()
{
	echo ""

	echo "Available packages are:"
	echo ""

	package_table |
	while read line
	do
		eval `package_fields "$line"`

		echo "$PKG_CODE)  $PKG_DESC"
	done

	echo "*)  All packages"
	echo "Q)  quit"
}

# ------------------------------------------------------------------------
# get package
#
get_package ()
{
	DEFAULT=s
	while true
	do
		$SLEEP

		package_menu

		CHOICE=`get_user_input "install-package" "$DEFAULT" \
			"package to install"`

		case "$CHOICE" in

		'*')
			PKG_NAME=all
			echo ""
			echo "Installing all packages."
			is_ok "install-all-packages" || continue
			break
			;;

		q* | Q*)
			abort_install
			;;
		esac

		line="`package_table | table_line $CHOICE`"
		if [ "$line" = "" ]
		then
			echo "Invalid choice \"$CHOICE\". Try again."
			continue
		fi

		eval `package_fields "$line"`

		echo ""
		echo "Installing \"$PKG_DESC\" package."
		is_ok "install-pkg-$PKG_DESC" || continue

		break
	done
}

# ------------------------------------------------------------------------
# extract the install files
#
extract_package ()
{
	PKG=$1

	CUR_DIR=`pwd`
	(
		cd $INST_DIR

		if [ -f "$CUR_DIR/$PLAT_DIR/$PKG.tgz" ]
		then
			FILE=$CUR_DIR/$PLAT_DIR/$PKG.tgz
		elif [ -f "$CUR_DIR/$PLAT_ALL/$PKG.tgz" ]
		then
			FILE=$CUR_DIR/$PLAT_ALL/$PKG.tgz
		else
			echoerr "Package \"$PKG\" not found."
			return
		fi

		echo ""
		echo "Extracting \"$PKG\" install files. Please wait."

		$UNCOMPRESS -c < "$FILE" |
		tar xvf - 2>&1 |
		while read line
		do
			echonl "."
		done
		echo ""
	)
}

# ========================================================================
# main()
#
CMD_SH=/bin/sh

PKGNAME=`pkgname`

eval `response_info`

# ------------------------------------------------------------------------
# hack to get around very weird AIX 4.2 bug
#
unset SLEEP
[ -d /lpp ] && SLEEP="sleep 1"

# ------------------------------------------------------------------------
# collect all options
#
PGM=`basename $0`
PKG_NAME=
INST_PLAT=
PLAT_BASE=server/unix
OPTS=

if [ "$1" = "-?" -o "$1" = "-help" ]
then
	usage
	exit 0
fi

if [ "$1" = "-??" -o "$1" = "-helpall" ]
then
	helpall
	exit 0
fi

ARGS=`getopt "Dvi:p:nqsr" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		OPTS="$OPTS -D"
		CMD_SH="$CMD_SH -x"
		shift
		;;

	-n)
		OPTS="$OPTS -n"
		shift
		;;

	-i)
		INST_PLAT="$2"
		shift 2
		;;

	-p)
		PKG_NAME="$2"
		shift 2
		;;

	-v)
		OPTS="$OPTS -v"
		shift
		;;

	-q)
		OPTS="$OPTS -q"
		shift
		break
		;;

	-s)
		OPTS="$OPTS -s"
		shift
		break
		;;

	-r)
		LOAD_RESPONSE_FILE="$SAVE_RESPONSE_FILE"
		OPTS="$OPTS -r"
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "$PGM: invalid argument $1"
		usage >&2
		abort_install
		;;
	esac
done

export LOAD_RESPONSE_FILE
export SAVE_RESPONSE_FILE

if [ "$SAVE_RESPONSE_FILE" != "" ]
then
	[ ! -d "$RESPONSE_DIR" ] && mkdir -p "$RESPONSE_DIR"
	touch "$SAVE_RESPONSE_FILE"
	chmod 664 "$SAVE_RESPONSE_FILE"
fi

# ------------------------------------------------------------------------
# check that user is root
#
check_root

# ------------------------------------------------------------------------
# get uncompress program to use
#
if look_for_pgm gunzip
then
	UNCOMPRESS="gunzip"

elif look_for_pgm gzip
then
	UNCOMPRESS="gzip -d"

elif look_for_pgm uncompress
then
	UNCOMPRESS="uncompress"

else
	echoerr "$PGM: Cannot determine uncompress program to use."
	abort_install
fi

# ------------------------------------------------------------------------
# start display to user
#
echo ""
echo "$PKGNAME Fax System Installation"
echo ""

# ------------------------------------------------------------------------
# get default platform to install for
#
if [ "$INST_PLAT" = "" ]
then
	eval `get_platform_default`
else
	PLAT_CODE=0
	PLAT_DESC=$INST_PLAT
	PLAT_NAME=$INST_PLAT

	if [ ! -d $PLAT_BASE/$INST_PLAT ]
	then
		echoerr "Platform \"$INST_PLAT\" not found."
		abort_install
	fi
fi

# ------------------------------------------------------------------------
# if platform found, verify it with user
#
DEFAULT=$PLAT_CODE
if [ "$PLAT_CODE" != "" -a "$PLAT_CODE" != "0" ]
then
	echo "Installing $PKGNAME for \"$PLAT_DESC\"."
	is_ok "install-for-$PLAT_CODE"|| PLAT_CODE=""
fi

# ------------------------------------------------------------------------
# put up menu and ask for choice if either user didn't want default
# or default not found
#
[ "$PLAT_CODE" = "" ] && get_platform

# ------------------------------------------------------------------------
# check platform install directory
#
PLAT_DIR=$PLAT_BASE/$PLAT_NAME
PLAT_ALL=$PLAT_BASE/all

if [ ! -d $PLAT_DIR ]
then
	echoerr "Directory $PLAT_DIR for \"$PLAT_DESC\" not found."
	abort_install
fi

# ------------------------------------------------------------------------
# request package if not specified
#
if [ "$PKG_NAME" != "" ]
then
	if [ "$PKG_NAME" != "all" ]
	then
		line=`package_table | grep "^[^:]*:$PKG_NAME:"`
		if [ "$line" = "" ]
		then
			echoerr "Package \"$PKG_NAME\"" \
				" for platform \"$PLAT_DESC\" not found."
			abort_install
		fi
		eval `package_fields "$line"`
	fi
else
	get_package
fi

# ------------------------------------------------------------------------
# get directory for extraction
#
echo ""
echo "Enter the name of the directory to extract the install files into."
echo "This directory must have at least 60 MB free."

INST_DIR=`get_user_input "extract-dir" "/tmp" "extract directory"`
INST_DIR=$INST_DIR/vsi-inst

rm -rf $INST_DIR
mkdir -p $INST_DIR
if [ $? -ne 0 ]
then
	echoerr "Cannot create temp directory $INST_DIR"
	abort_install
fi

# ------------------------------------------------------------------------
# extract the install files
#
if [ "$PKG_NAME" = "all" ]
then
	PKG_NAME=fxserver

	PKGS=`package_all`
	for pkg in $PKGS
	do
		rm -f $INST_DIR/install.sh
		extract_package `basename $pkg .tgz`
	done
else
	extract_package $PKG_NAME
fi

if [ ! -x $INST_DIR/install.sh ]
then
	echoerr ""
	echoerr "Extraction failed."
	rm -rf $INST_DIR
	abort_install
fi

# ------------------------------------------------------------------------
# do the install
#
(
	cd $INST_DIR
	$CMD_SH ./install.sh $OPTS
)

# ------------------------------------------------------------------------
# remove the extract directory if empty
#
[ ! -f $INST_DIR/install.sh ] && rm -rf $INST_DIR
