#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	setup processing for fxserver installation
#
#	this script does the following:
#
#		1.	creates any databases that don't exist
#		2.	adds the admin entry to the user database if it
#			doesn't exist
#		3.	adds the vsidemo entry to the user database if it
#			doesn't exist
#		4.	creates the LoopBack FIM if it doesn't exist
#		5.	creates the SendMail FIM if it doesn't exist
#		6.	adds some entries to the system .ini files
#		7.	adds vsinet entries to /etc/services if necessary
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# add a directory to the path
#
# Note that the directory is ALWAYS added to the end.
#
addpath ()
{
	dir="$1"

	sep=":"
	arg=""
	if in_windows
	then
		sep=";"
		arg="-i"
	fi

	foo=`echo "$PATH" | egrep $arg "^$dir$sep|$sep$dir$sep|$sep$dir$"`
	if [ $? -eq 0 ]
	then
		echo "$PATH"
	else
		echo "$PATH$sep$dir"
	fi

	unset dir sep arg foo
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# echo a line only if $SILENT_INSTALL != "yes"
#
echons ()
{
	[ "$SILENT_INSTALL" = "yes" ] || echo "$*"
}

# ------------------------------------------------------------------------
# get Windows directory
#
get_windir ()
{
	{
		if [ "$windir" != "" ]
		then
			echo "$windir"

		elif [ "$WINDIR" != "" ]
		then
			echo "$WINDIR"

		elif [ "$SystemRoot" != "" ]
		then
			echo "$SystemRoot"

		elif [ "$SYSTEMROOT" != "" ]
		then
			echo "$SYSTEMROOT"

		elif [ -d "c:/windows" ]
		then
			echo "c:/windows"

		elif [ -d "c:/winnt" ]
		then
			echo "c:/winnt"

		else
			echo "c:/windows"
		fi
	} | sed -e 's,\\\\*,/,g'
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# setup services entries
#
setup_services ()
{
	PKGNAME=`pkgname`

	# ----------------------------------------------------------------
	# get name of services file
	#
	if [ -d /etc ]
	then
		SERVICES=/etc/services
	else
		SERVICES=`get_windir`/system32/drivers/etc/services
	fi

	[ -f $SERVICES ] || return

	# ----------------------------------------------------------------
	# Check if all entries are present
	#
	HAVE_VSINET=`grep -c '^vsinet'   $SERVICES`
	HAVE_VSIXML=`grep -c '^vsixml'   $SERVICES`
	HAVE_VSIADM=`grep -c '^vsiadmin' $SERVICES`

	# ----------------------------------------------------------------
	# If any entries are missing, add them
	#
	if [ $HAVE_VSINET -eq 0 -o $HAVE_VSIXML -eq 0 -o $HAVE_VSIADM -eq 0 ]
	then
		# --------------------------------------------------------
		# Official port numbers from IANA
		#
		NET_PORT=996
		XML_PORT=2996
		ADM_PORT=2539

		NET_NAME=vsinet
		XML_NAME=vsixml
		ADM_NAME=vsiadmin

		# --------------------------------------------------------
		# create backup file
		#
		chmod u+w $SERVICES
		cp $SERVICES $SERVICES.bak

		# --------------------------------------------------------
		# now add any missing entries
		#
		if [ $HAVE_VSINET -eq 0 ]
		then
			(
		echo "$NET_NAME	$NET_PORT/tcp	# $PKGNAME network daemon"
		echo "$NET_NAME	$NET_PORT/udp	# $PKGNAME network daemon"
			) >>$SERVICES
		fi

		if [ $HAVE_VSIXML -eq 0 ]
		then
			(
		echo "$XML_NAME	$XML_PORT/tcp	# $PKGNAME xmlnet daemon"
		echo "$XML_NAME	$XML_PORT/udp	# $PKGNAME xmlnet daemon"
			) >>$SERVICES
		fi

		if [ $HAVE_VSIADM -eq 0 ]
		then
			(
		echo "$ADM_NAME	$ADM_PORT/tcp	# $PKGNAME admin daemon"
		echo "$ADM_NAME	$ADM_PORT/udp	# $PKGNAME admin daemon"
			) >>$SERVICES
		fi
	fi
}

# ------------------------------------------------------------------------
# Return the user name to use
#
usrname ()
{
	echo "${VSI_USRNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# Return the group name to use
#
grpname ()
{
	echo "${VSI_GRPNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# Return the legacy product name to use
#
legname ()
{
	echo "vsifax3"
}

# ------------------------------------------------------------------------
# Set owner & group of a file
#
set_usrgrp ()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# get args
	#
	USRNAME="$1"
	GRPNAME="$2"
	FILE="$3"

	# ----------------------------------------------------------------
	# set the owner & group
	#
	chgrp $GRPNAME "$FILE"
	chown $USRNAME "$FILE"
}

# ------------------------------------------------------------------------
# create env files & update system files to source them
#
add_env_files ()
{
	# ----------------------------------------------------------------
	# validate that /etc exists
	#
	[ -d /etc ] || return

	# ----------------------------------------------------------------
	# get package info
	#
	USRNAME=`usrname`
	GRPNAME=`grpname`
	PKGNAME=`pkgname`
	PRDNAME=`prdname`
	LEGNAME=`legname`

	# ----------------------------------------------------------------
	# Create env files
	#
	SH_ENV_FILE=/etc/$PRDNAME.sh
	(
		echo "# setup $PKGNAME environment"
		echo "VSIFAX=\"$VSIFAX\""
		echo "PATH=\"\$PATH:\$VSIFAX/bin\""
		echo 'id=`id | sed -e "s/[^=]*=//" -e "s/(.*//"`'
		echo "[ \$id -eq 0 -o \"\$LOGNAME\" = \"$USRNAME\" ] &&"
		echo "	PATH=\"\$PATH:\$VSIFAX/lbin\""
		echo "unset id"
		echo "[ -d \"\$VSIFAX/obin\" ] && PATH=\"\$VSIFAX/obin:\$PATH\""
		echo "export VSIFAX PATH"
	) >$SH_ENV_FILE

	chmod 775 $SH_ENV_FILE
	set_usrgrp $USRNAME $GRPNAME $SH_ENV_FILE

	SH_LEG_FILE=/etc/$LEGNAME.sh
	rm -f $SH_LEG_FILE
	ln -s `basename $SH_ENV_FILE` $SH_LEG_FILE

	CSH_ENV_FILE=/etc/$PRDNAME.csh
	(
		echo "# setup $PKGNAME environment"
		echo "setenv VSIFAX \"$VSIFAX\""
		echo "set path=(\$path \"\$VSIFAX/bin\")"
		echo 'set id=`id | sed -e "s/[^=]*=//" -e "s/(.*//"`'
		echo "if ( \$id == 0 || \"$LOGNAME\" == \"$USRNAME\" ) \\"
		echo "	set path=(\$path \"\$VSIFAX/lbin\")"
		echo "unset id"
		echo "if (-d \"\$VSIFAX/obin\") set path=(\"\$VSIFAX/obin\" \$path)"
	) >$CSH_ENV_FILE

	chmod 775 $CSH_ENV_FILE
	set_usrgrp $USRNAME $GRPNAME $CSH_ENV_FILE

	CSH_LEG_FILE=/etc/$LEGNAME.csh
	rm -f $CSH_LEG_FILE
	ln -s `basename $CSH_ENV_FILE` $CSH_LEG_FILE

	# ----------------------------------------------------------------
	# Check if /etc/profile or /etc/cshrc need to be added
	#
	DO_ADD=no
	FILES=""
	if [ -f /etc/profile ]
	then
		grep $SH_ENV_FILE /etc/profile >/dev/null 2>&1
		if [ $? -ne 0 ]
		then
			DO_ADD=yes
			FILES="file /etc/profile"
		fi
	fi

	if [ -f /etc/cshrc ]
	then
		grep $CSH_ENV_FILE /etc/cshrc >/dev/null 2>&1
		if [ $? -ne 0 ]
		then
			DO_ADD=yes
			FILES="files /etc/profile and /etc/cshrc"
		fi
	fi

	# ----------------------------------------------------------------
	# Ask if we should update system files
	#
	if false # [ "$DO_ADD" = "yes" -a "$QUICK_INSTALL" != "yes" ]
	then
		echo ""
		echo "The $PKGNAME system requires users to set a few"
		echo "environment variables prior to use."
		echo ""
		echo "The file $SH_ENV_FILE will set these variables for"
		echo "Bourne and Korn shell users and the file $CSH_ENV_FILE"
		echo "will set these variables for C-shell users."
		echo ""
		echo "The $FILES can be updated"
		echo "by this installation procedure to automatically"
		echo "source these files."
		echo ""
		echo "Updating $FILES."

		is_ok "set-env-files-ok" || DO_ADD=no
	fi

	# ----------------------------------------------------------------
	# Check if entry exists in /etc/profile to use the sh env file
	#
	if [ "$DO_ADD" = "yes" ]
	then
		if [ -f /etc/profile ]
		then
			grep $SH_ENV_FILE /etc/profile >/dev/null 2>&1
			if [ $? -ne 0 ]
			then
			(
			echo ""
			echo "# Setup the $PKGNAME environment"
			echo "[ -f $SH_ENV_FILE ] && . $SH_ENV_FILE"
			) >>/etc/profile
			fi
		fi
	fi

	# ----------------------------------------------------------------
	# Check if entry exists in /etc/cshrc to use the csh env file
	#
	if [ "$DO_ADD" = "yes" ]
	then
		if [ -f /etc/cshrc ]
		then
			grep $CSH_ENV_FILE /etc/cshrc >/dev/null 2>&1
			if [ $? -ne 0 ]
			then
			(
			echo ""
			echo "# Setup the $PKGNAME environment"
			echo "if (-f $CSH_ENV_FILE) source $CSH_ENV_FILE"
			) >>/etc/cshrc
			fi
		fi
	fi
}

# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# get name of the NULL device
#
get_dev_null ()
{
	if in_windows
	then
		echo "nul:"
	else
		echo "/dev/null"
	fi
}

# ------------------------------------------------------------------------
# get the hostname from "hostname" then "uname -n"
#
get_hostname ()
{
	for dir in /bin /usr/bin /usr/sbin /usr/ucb
	do
		if [ -x $dir/hostname ]
		then
			$dir/hostname
			return
		fi
	done

	uname -n
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -q     quick install"
	echo "  -s     silent install"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D     debug install"
}

# ------------------------------------------------------------------------
# un-setup services entries
#
# We no longer use inetd for admin functionality.
# We now run vadmind as a true daemon.
#
# At some point, we can remove this sub-script.
#
unsetup_inetd ()
{
	# ----------------------------------------------------------------
	# update file inetd.conf or xinetd.conf
	#
	if [ -f /etc/xinetd.d/vsiadmin ]
	then
		INETD_CONF=/etc/xinetd.conf
		rm -f /etc/xinetd.d/vsiadmin

	elif [ -f /etc/inetd.conf ]
	then
		if [ "`grep vsiadmin /etc/inetd.conf`" != "" ]
		then
			INETD_CONF=/etc/inetd.conf

			chmod u+w $INETD_CONF
			mv -f $INETD_CONF $INETD_CONF.bak
			sed -e '/vsiadmin/d' -e '/VSI-FAX/d' $INETD_CONF.bak \
				>$INETD_CONF
		else
			return
		fi

	else
		return
	fi

	# ----------------------------------------------------------------
	# find PID for inetd or xinetd
	#
	if [ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
	then
		# --------------------------------------------------------
		# Linux
		#
		if [ -f /var/run/inetd.pid -a \
			"$INETD_CONF" = "/etc/inetd.conf" ]
		then
			PID=`cat /var/run/inetd.pid`

		elif [ -f /var/run/xinetd.pid -a \
			"$INETD_CONF" = "/etc/xinetd.conf" ]
		then
			PID=`cat /var/run/xinetd.pid`
		else
			PID=0
		fi

	elif [ -x /vmunix ]
	then
		# --------------------------------------------------------
		# BSD
		#
		if [ "$INETD_CONF" = "/etc/inetd.conf" ]
		then
			PID=`ps -ef |
				grep inetd |
				grep -v grep |
				head -1 |
				sed -e 's/  */	/g' |
				cut -f2'`
		elif [ "$INETD_CONF" = "/etc/xinetd.conf" ]
		then
			PID=`ps -ef |
				grep xinetd |
				grep -v grep |
				head -1 |
				sed -e 's/  */	/g' |
				cut -f2'`
		fi

	else
		# --------------------------------------------------------
		# System V
		#
		if [ "$INETD_CONF" = "/etc/inetd.conf" ]
		then
			PID=`ps -e |
				grep inetd |
				grep -v grep |
				head -1 |
				sed -e 's/^  *//' -e 's/ .*//'`
		elif [ "$INETD_CONF" = "/etc/xinetd.conf" ]
		then
			PID=`ps -e |
				grep xinetd |
				grep -v grep |
				head -1 |
				sed -e 's/^  *//' -e 's/ .*//'`
		fi
	fi

	# ----------------------------------------------------------------
	# now notify the appropriate daemon that its configuration has changed
	#
	if [ "$PID" != "" -a "$PID" -ne 0 ]
	then
		kill -HUP "$PID"
	fi
}

# ------------------------------------------------------------------------
# set defaults in ini files for easier changing
#
setup_config ()
{
	FODCCLID="`get_hostname`_`date +%Y%m%d_%H%M%S`"

	vfxadmin config -sn COMMON   DateFormat		"%m/%d/%y"
	vfxadmin config -sn COMMON   TimeFormat		"%I:%M %p %Z"
	vfxadmin config -sn COMMON   log-max-files	5
	vfxadmin config -sn COMMON   log-max-lines	0
	vfxadmin config -sn COMMON   log-max-size	0
	vfxadmin config -sn COMMON   log-line-length	80
	vfxadmin config -sn COMMON   UseDefaults	on

	vfxadmin config -sn VADMIND  Debug		off
	vfxadmin config -sn VADMIND  DebugCron		off

	vfxadmin config -sn VFXSCHED Debug		off
	vfxadmin config -sn VFXSCHED DebugDiskspace	off
	vfxadmin config -sn VFXSCHED DebugNotify	off
	vfxadmin config -sn VFXSCHED StartAdmin		on
	vfxadmin config -sn VFXSCHED ResendFromStart	off
	vfxadmin config -sn VFXSCHED UseResendCover	on
	vfxadmin config -sn VFXSCHED DiskSpaceMin	50k
	vfxadmin config -sn VFXSCHED AccelNotfaxExpire	on
	vfxadmin config -sn VFXSCHED MaxRespawnTime	240
	vfxadmin config -sn VFXSCHED MaxRespawnCount	3
	vfxadmin config -sn VFXSCHED MaxNodials		3
	vfxadmin config -sn VFXSCHED MaxNoRsps		3
	vfxadmin config -sn VFXSCHED MaxModems		128
	vfxadmin config -sn VFXSCHED WaitStartupSecs	30
	vfxadmin config -sn VFXSCHED WaitShutdownSecs	3
	vfxadmin config -sn VFXSCHED MaxFimIdle		700
	vfxadmin config -sn VFXSCHED MaxSpawnCount	0
	vfxadmin config -sn VFXSCHED RunVgsched		on
	vfxadmin config -sn VFXSCHED RunVsmtpd		off
	vfxadmin config -sn VFXSCHED RunVmaild		on
	vfxadmin config -sn VFXSCHED CheckDupNum	on
	vfxadmin config -sn VFXSCHED CallBatching	off
	vfxadmin config -sn VFXSCHED GracePeriod	0
	vfxadmin config -sn VFXSCHED GroupLNP		off
	vfxadmin config -sn VFXSCHED GroupTNP		off
	vfxadmin config -sn VFXSCHED ShouldRestart	on
	vfxadmin config -sn VFXSCHED LoadBalance	off
	vfxadmin config -sn VFXSCHED LoadBias		5
	vfxadmin config -sn VFXSCHED LoadDiffBias	5

	vfxadmin config -sn VGSCHED  DebugStats		off
	vfxadmin config -sn VGSCHED  MaxMembersQueued	16
	vfxadmin config -sn VGSCHED  MaxIdleMins	0

	vfxadmin config -sn VSINET   RemoteRegOK	on
	vfxadmin config -sn VSINET   AutoReg		off
	vfxadmin config -sn VSINET   AutosendDelay	60

	vfxadmin config -sn VPOPD    Debug		off
	vfxadmin config -sn VPOPD    AuthID		licensed
	vfxadmin config -sn VPOPD    ExtChoice		txt
	vfxadmin config -sn VPOPD    Hostname		""
	vfxadmin config -sn VPOPD    Username		""
	vfxadmin config -sn VPOPD    Password		""

	vfxadmin config -sn DEVICE   PageDelay		10
	vfxadmin config -sn DEVICE   SendDelay		10
	vfxadmin config -sn DEVICE   EnableDID		off
	vfxadmin config -sn DEVICE   AnswerRings	1
	vfxadmin config -sn DEVICE   PagehdrString	"%3 %4  %1 %2"
	vfxadmin config -sn DEVICE   PagehdrEnd		"Page %5 of %6 #%7 %8"
	vfxadmin config -sn DEVICE   EnableHeaders	on
	vfxadmin config -sn DEVICE   PagehdrLine	on
	vfxadmin config -sn DEVICE   AdString		"via $PKGNAME"
	vfxadmin config -sn DEVICE   dial-debug		on
	vfxadmin config -sn DEVICE   dis-debug		off
	vfxadmin config -sn DEVICE   event-debug	on
	vfxadmin config -sn DEVICE   fifo-debug		off
	vfxadmin config -sn DEVICE   mtype-debug	on
	vfxadmin config -sn DEVICE   nsf-debug		off
	vfxadmin config -sn DEVICE   query-debug	on
	vfxadmin config -sn DEVICE   raster-debug	off
	vfxadmin config -sn DEVICE   readline-debug	off
	vfxadmin config -sn DEVICE   reset-debug	off
	vfxadmin config -sn DEVICE   t30-debug		on
	vfxadmin config -sn DEVICE   termio-debug	on
	vfxadmin config -sn DEVICE   waitfor-debug	on
	vfxadmin config -sn DEVICE   xstate-debug	off

	vfxadmin config -sn -i FoDC DEVICE debug		off
	vfxadmin config -sn -i FoDC DEVICE failover		on
	vfxadmin config -sn -i FoDC DEVICE keep-files		off
	vfxadmin config -sn -i FoDC DEVICE dial-convert 	off
	vfxadmin config -sn -i FoDC DEVICE request-timeout 	300
	vfxadmin config -sn -i FoDC DEVICE tcp-dbglevel    	0
	vfxadmin config -sn -i FoDC DEVICE secured		on
	vfxadmin config -sn -i FoDC DEVICE hostname		na.ondemand.esker.com
	vfxadmin config -sn -i FoDC DEVICE clientid		"$FODCCLID"

	vfxadmin config -sn -i FoDC2 DEVICE debug		off
	vfxadmin config -sn -i FoDC2 DEVICE keep-files		off
	vfxadmin config -sn -i FoDC2 DEVICE dial-convert 	off
	vfxadmin config -sn -i FoDC2 DEVICE request-timeout 	300
	vfxadmin config -sn -i FoDC2 DEVICE tcp-dbglevel    	0
	vfxadmin config -sn -i FoDC2 DEVICE secured		on
	vfxadmin config -sn -i FoDC2 DEVICE hostname		as1.ondemand.esker.com
	vfxadmin config -sn -i FoDC2 DEVICE clientid		"$FODCCLID"

	vfxadmin config -sn VNOTIFY  Debug		off

	vfxadmin config -sn VMAIL    Debug		off

	vfxadmin config -cn LOGIN    Hostname		"localhost"
	vfxadmin config -cn COMMON   UseDefaults	on
}

# ------------------------------------------------------------------------
# setup any users needed
#
setup_users ()
{
	# ----------------------------------------------------------------
	# create master-profile if needed
	#
	vfxadmin user -M

	# ----------------------------------------------------------------
	# create admin entry in faxq if needed
	#
	ADMIN=vsifax
	vfxadmin user -d -n "VSIFAX Administrator" -t umd=usr-admin $ADMIN

	# ----------------------------------------------------------------
	# set admin entry as profile inheriting from master-profile
	#
	vfxadmin user -t isp=yes -t pid="-master-" $ADMIN

	# ----------------------------------------------------------------
	# create demo entry in faxq if needed
	#
	DEMO_LOGIN=vsidemo
	vfxadmin user -n "$PKGNAME Demo Login" $DEMO_LOGIN

	# ----------------------------------------------------------------
	# create lcr entry in faxq if needed
	#
	LCR_LOGIN=vsilcr
	vfxadmin user -d -n "$PKGNAME LCR name" $LCR_LOGIN
}

# ------------------------------------------------------------------------
# setup any devices needed
#
setup_devices ()
{
	# ----------------------------------------------------------------
	# setup loopback FIM if not already done
	#
	# We also make it the default if no default dest is yet setup.
	#
	vfxadmin device -ql lb ||
	{
		DEFAULT=
		vfxadmin default -q || DEFAULT=-d
		vfxadmin device -q -a $DEFAULT -i lb -n "LoopBack" lb
	}

	# ----------------------------------------------------------------
	# setup sendmail FIM if not already done
	#
	vfxadmin device -ql sm ||
	{
		vfxadmin device -q -a -i sm -n "SendMail" sm
	}

	# ----------------------------------------------------------------
	# setup LCR FIM if not already done
	#
	vfxadmin device -ql lcr ||
	{
		vfxadmin device -q -a -i hold -T 240 -n "LCR hold queue" lcr
	}

	# ----------------------------------------------------------------
	# setup FoDC FIM if not already done
	#
	vfxadmin device -ql FoDC ||
	{
		vfxadmin device -q -a -i et -E off -X fodcpgm -n "Fax on Demand" FoDC
	}

	# ----------------------------------------------------------------
	# setup FoDC2 FIM if not already done
	#
	vfxadmin device -ql FoDC2 ||
	{
		vfxadmin device -q -a -i et -E off -X fodcpgm -n "FoDC Failover" FoDC2
	}
}

# ------------------------------------------------------------------------
# setup any classes needed
#
setup_classes ()
{
	# ----------------------------------------------------------------
	# setup FoDC Class if not already done
	#
	vfxadmin class -ql FoDCClass ||
	{
		vfxadmin class -q -a FoDC FoDCClass
		vfxadmin class -q -a FoDC2 FoDCClass
		vfxadmin class -q -u -n "Fax on Demand Class" -A on -S on FoDCClass
	}
}

# ------------------------------------------------------------------------
# setup any config files needed
#
process_cfg_file ()
{
	file=$1

	# ----------------------------------------------------------------
	# bail if no new version of this file
	#
	[ -f cfg-new/$file ] || return

	# ----------------------------------------------------------------
	# copy the new version of this file if needed
	#
	if [ ! -f $file ]
	then
		cp -p cfg-new/$file .

	elif [ -f cfg-old/$file ]
	then
		if diff $file cfg-old >$DEVNULL 2>&1
		then
			cp -p cfg-new/$file .
		else
			diff cfg-new/$file cfg-old >$DEVNULL ||
			echons "WARNING: file changed: $file"
		fi
	fi

	# ----------------------------------------------------------------
	# cache the new config file for next update install
	#
	cp -p cfg-new/$file cfg-old
}

setup_cfg_files ()
{
	(
		cd "$VSIFAX/lib"

		# --------------------------------------------------------
		# delete all "*.new" files (this can be removed at some point)
		#
		rm -f *.new

		# --------------------------------------------------------
		# process all config files
		#
		process_cfg_file autotypes.lst
		process_cfg_file daemons.lst
		process_cfg_file dialcode.lst
		process_cfg_file dialcvt.lst
		process_cfg_file rejects.lst
		process_cfg_file retrys.lst
		process_cfg_file printers.lst
		process_cfg_file lcrhosts.lst
		process_cfg_file lcrrules.lst
		process_cfg_file devrules.lst
		process_cfg_file formatters.lst
		process_cfg_file crontab.lst
		process_cfg_file cluster.lst
		process_cfg_file notify.lst
		process_cfg_file certauth.lst
		process_cfg_file c2modems.ini
	)
}

# ------------------------------------------------------------------------
# setup all databases
#
setup_databases ()
{
	VFLAG=
	if [ "$SILENT_INSTALL" != "yes" ]
	then
		echo ""
		echo "Checking databases:"
		VFLAG="-v"
	fi

	vfxdb $VFLAG
}

# ------------------------------------------------------------------------
# remove or change any old files no longer used
#
check_file ()
{
	old_file="$VSIFAX/lbin/$1"
	new_file="$VSIFAX/lbin/$2"

	if [ -f "$new_file" ]
	then
		rm -f "$old_file"
	else
		[ -f "$old_file" ] && mv -f "$old_file" "$new_file"
	fi
}

fix_old_files ()
{
	for i in bm c2 ex lb sm
	do
		check_file $i-fim fim-$i
	done
}

fix_old_modem_class ()
{
myList=`vfxadmin list | grep 'fim=c2' | awk '{split($1, ar,"="); print ar[2] }'`
	if [ "$myList" != "" ]
	then
		echons "Updating modem fim type from c2 to cx"
	fi

	for i in $myList
	do
        	vfxadmin device -i cx -u  $i
	done

}

setup_cronjobs()
{
	if [ `vfxadmin cron -l | grep 'fax-purge' | grep '\-o' | grep '\-x' | wc -l` -eq "0" ]
	then
		echo ""
		echo "Configure automatic purge of archived jobs..."
		echo "0 0 * * 1 fax-purge -x -o 30" | vfxadmin cron -a
		echo "done"
	fi
	if [ `vfxadmin cron -l | grep 'restorefodc.tcl'| wc -l` -eq "0" ]
	then
		echo ""
		echo "Configure FoDC Failover cron..."
		echo "0,5,10,15,20,25,30,35,40,45,50,55 * * * * exec-online ./faxtcl \"../lib/enp/restorefodc.tcl\"" | vfxadmin cron -a
		echo "done"
	fi
}

# ------------------------------------------------------------------------
# main()
#
PATH=`addpath "$VSIFAX/bin"`
PATH=`addpath "$VSIFAX/lbin"`

USRNAME=`usrname`
GRPNAME=`grpname`
PKGNAME=`pkgname`

DEVNULL=`get_dev_null`

# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

DEBUG=

case "$1" in
	'-?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | -helpall | --helpall)
		helpall
		exit 0
		;;

	'-V' | -version | --version)
		echo "$PGM: `vfxadmin version`"
		exit 0
		;;
esac

ARGS=`getopt "Dqs" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-x
		shift
		;;

	-q)
		QUICK_INSTALL=yes	export QUICK_INSTALL
		shift
		;;

	-s)
		SILENT_INSTALL=yes	export SILENT_INSTALL
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

[ "$DEBUG" != "" ] && set $DEBUG

# ------------------------------------------------------------------------
# check that user is root
#
check_root

# ------------------------------------------------------------------------
# check if any license stuff needs to be done
#
vfxadmin license -is

vfxadmin license -s -D ||
{
	if [ "$VSI_SERVER_LIC" != "" ]
	then
		vfxadmin license -s -L "$VSI_SERVER_LIC" &&
			echons "Server license added."
	fi
}

# ------------------------------------------------------------------------
# setup setuid/setgid programs
#
in_windows ||
{
	SETUID_ROOT_BIN="vfxsched"
	SETUID_ROOT_LBIN=""
	SETUID_VSI_BIN="vfxadmin vfxpurge"
	SETUID_VSI_LBIN="vadmind"

	(
		cd "$VSIFAX/bin"

		if [ "$SETUID_ROOT_BIN" != "" ]
		then
			for i in $SETUID_ROOT_BIN
			do
				if [ -x $i ]
				then
					set_usrgrp root $GRPNAME $i
					chmod 6755 $i
				fi
			done
		fi

		if [ "$SETUID_VSI_BIN" != "" ]
		then
			for i in $SETUID_VSI_BIN
			do
				if [ -x $i ]
				then
					chmod 6755 $i
				fi
			done
		fi
	)

	(
		cd "$VSIFAX/lbin"

		if [ "$SETUID_ROOT_LBIN" != "" ]
		then
			for i in $SETUID_ROOT_LBIN
			do
				if [ -x $i ]
				then
					set_usrgrp root $GRPNAME $i
					chmod 6755 $i
				fi
			done
		fi

		if [ "$SETUID_VSI_LBIN" != "" ]
		then
			for i in $SETUID_VSI_LBIN
			do
				if [ -x $i ]
				then
					chmod 6755 $i
				fi
			done
		fi
	)
}

# ------------------------------------------------------------------------
# Now do any setups
#
add_env_files
setup_cfg_files
setup_databases
setup_users
setup_devices
setup_classes
setup_config
setup_services
setup_cronjobs
unsetup_inetd
fix_old_files
fix_old_modem_class
# ------------------------------------------------------------------------
# if in windows, setup services & registry
#
in_windows && vfxadmin server install -s "$VSIFAX"

exit 0
