// +++FHDR+++
//
//
//   Class Name : TransportManager
//   Super Class: n/a
//
//   Description:
// 	The TransportManager class manages the transports that are available
//	for sending XML-F requests to the network fax server. Transports can
//	be added at runtime by binding a transport protocol name to a
//	transporting object. All transports must be sub-classed from the
//	DocumentTransport class. Currently the HTTP transport is implemented
//	and automatically bound when the TransportManager class is initialized. 
//
// 	Additionally, the TransportManager class manages XML document decoders.
//	These document decoders take an XML document and map it into an object
//	which represents that document. Document decoders must be sub-classed
//	from the DocumentDecoder class. Document decoders are provided for the
//	three XML-F response documents (fax-submit-response,
//	fax-status-response, and fax-cancel-response documents).
//
//
//   Notes	: none
//
//
//   Method - Description
//   ------------------------------------------------------------------------
//
//
// ---FHDR---

#include <string.h>

#include "TransportManager.h"
#include "Url.h"

// +++PHDR+++
//
//  Method     : TransportManager::TransportManager()
//
//  Visibility : public
//
//  Description: Default constructor for Transportmanager class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

TransportManager::TransportManager ()
{
	_default_protocol = NULL;
	_err_msg.clear();
	_decoders = new Registry();
	_transports = new Registry();
}


// +++PHDR+++
//
//  Method     : TransportManager::~TransportManager()
//
//  Visibility : public
//
//  Description: Default destructor for Transportmanager class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

TransportManager::~TransportManager ()
{
	if (_default_protocol != NULL) {
		delete _default_protocol;
		_default_protocol = NULL;
	}

	delete _decoders;
	_decoders = NULL;

	delete _transports;
	_transports = NULL;
}


// +++PHDR+++
//
//  Method     : TransportManager::getLastError()
//
//  Visibility : public
//
//  Description: Returns the last reported error message.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:    char * with last reported error message
//
//
//  Notes:     none
//
// ---PHDR---

char * TransportManager::getLastError ()
{
	return (_err_msg);
}


// +++PHDR+++
//
//  Method     : TransportManager::getDefaultTransport()
//
//  Visibility : public
//
//  Description: Returns the default transport protocol.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:    char * containing the default transport protocol or NULL if no
//		default protocol has been set.
//
//
//  Notes:     none
//
// ---PHDR---

char * TransportManager::getDefaultTransport ()
{
	return (_default_protocol);
}


// +++PHDR+++
//
//  Method     : TransportManager::setDefaultTransport()
//
//  Visibility : public
//
//  Description: Sets the default transport protocol. The default transport
//               protocol is used whenever no protocol is present in a URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  protocol        char *                  the new default protocol
//
//
//  Returns:    n/a
//
//
//  Notes:     none
//
// ---PHDR---

void TransportManager::setDefaultTransport (char * protocol)
{
	if (_default_protocol != NULL) {
		delete _default_protocol;
		
	}

	if (protocol == NULL) {
		_default_protocol = NULL;
		return;
	}
	
	_default_protocol = new char[strlen(protocol) + 1];
	strcpy(_default_protocol, protocol);
}


// +++PHDR+++
//
//  Method     : TransportManager::bindDecoder()
//
//  Visibility : public
//
//  Description: Binds (maps) a document type to a document decoder.
//		 A document type may be bound to only one decoder at a time.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  doc_type        char *                  name of the document type to bind
//  decoder         DocumentDecoder *       pointer to the document decoder
//
//
//  Returns:   TRUE on success, FALSE on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int TransportManager::bindDecoder (char * doc_type, DocumentDecoder * decoder)
{
	return (_decoders->bind(doc_type, decoder));
}


// +++PHDR+++
//
//  Method     : TransportManager::bindTransport()
//
//  Visibility : public
//
//  Description: Binds (maps) a protocol name to a transport. A protocol
//               name may be bound to only one transport at a time.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  protocol        char *                  name of transport protocol to bind
//  transport       DocumentTransport *     pointer to transport for protocol
//
//
//  Returns:   TRUE on success, FALSE on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int TransportManager::bindTransport (char * protocol, 
	DocumentTransport * transport)
{
	return (_transports->bind(protocol, transport));
}


// +++PHDR+++
//
//  Method     : TransportManager::listDecoders()
//
//  Visibility : public
//
//  Description: Returns an enumeration of the currently bound
//               document decoders.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   An array of char *, one for each bound decoder.
//
//
//  Notes:     none
//
// ---PHDR---

char ** TransportManager::listDecoders ()
{
	return (_decoders->list());
}


// +++PHDR+++
//
//  Method     : TransportManager::listTransports()
//
//  Visibility : public
//
//  Description: Returns an enumeration of the currently bound transports.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   An array of char *, one for each bound transport.
//
//
//  Notes:     none
//
// ---PHDR---

char ** TransportManager::listTransports ()
{
	return (_transports->list());
}


// +++PHDR+++
//
//  Method     : TransportManager::lookupDecoder()
//
//  Visibility : public
//
//  Description: Returns the DocumentDecodert object for the specified
//               document type. If the document type is not bound NULL
//               will be returned.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  doc_type        char *                  document type to lookup
//
//
//  Returns:   A pointer to the DocumentDecoder mapped to the specified
//             document type.
//
//
//  Notes:     none
//
// ---PHDR---

DocumentDecoder * TransportManager::lookupDecoder (char * doc_type)
{
	return ((DocumentDecoder *)_decoders->lookup(doc_type));
}

// +++PHDR+++
//
//  Method     : TransportManager::lookupTransport()
//
//  Visibility : public
//
//  Description: Returns the DocumentTransport object for the specified
//               protocol. If the protocol name is not bound null will
//               be returned.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  protocol        char *                  name of the transport protocol
//
//
//  Returns:   A pointer to the DocumentTransport mapped to the specified
//             protocol name.
//
//
//  Notes:     none
//
// ---PHDR---

DocumentTransport * TransportManager::lookupTransport (char * protocol)
{
	return ((DocumentTransport *)_transports->lookup(protocol));
}


// +++PHDR+++
//
//  Method     : TransportManager::send()
//
//  Visibility : public
//
//  Description: Sends the XML-F data (document) contained by xml_data to
//               the fax server specified by url. Returns the data returned
//               from the fax server.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  url             char *                  URL to send the XMLF data to
//  xml_data        char *                  the XML data to send
//
//
//  Returns:   A char * containing the response from the fax server.
//
//
//  Notes:     none
//
// ---PHDR---

char * TransportManager::send (char * url_spec, char * xml_data)
{
	Url url(url_spec); 
	char * protocol = url.getProtocol();
	if (protocol == NULL) {
		protocol = getDefaultTransport();
		if (protocol == NULL) {
			_err_msg << "URL spec must contain a protocol" << ends;
			return (NULL);
		}
	
		url.setProtocol(protocol);
	}

	DocumentTransport * transport = lookupTransport(protocol);
	if (transport == NULL) {
		_err_msg << "invalid protocol: " << protocol << ends;
		return (NULL);
	}

	char * response = transport->send(url.toString(), xml_data);
	if (response == NULL) {
		_err_msg << transport->getLastError() << ends;
	}

	return (response);
}


// +++PHDR+++
//
//  Method     : TransportManager::send()
//
//  Visibility : public
//
//  Description: Sends the XML-F data (document) contained by xml_stream to
//               the fax server specified by url. Returns the data returned
//               from the fax server.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  url             char *                  URL to send the XMLF data to
//  xml_stream      istream *               the stream to read XML data from
//
//
//  Returns:   A char * containing the response from the fax server.
//
//
//  Notes:     none
//
// ---PHDR---

char * TransportManager::send (char * url_spec, istream * xml_stream)
{
	Url url(url_spec); 
	char * protocol = url.getProtocol();
	if (protocol == NULL) {
		protocol = getDefaultTransport();
		if (protocol == NULL) {
			_err_msg << "URL spec must contain a protocol" << ends;
			return (NULL);
		}
	
		url.setProtocol(protocol);
	}

	DocumentTransport * transport = lookupTransport(protocol);
	if (transport == NULL) {
		_err_msg << "invalid protocol: " << protocol << ends;
		return (NULL);
	}

	char * response = transport->send(url.toString(), xml_stream);
	if (response == NULL) {
		_err_msg << transport->getLastError() << ends;
	}

	return (response);
}


// +++PHDR+++
//
//  Method     : TransportManager::send()
//
//  Visibility : public
//
//  Description: Sends the XML-F document contained by the XmlfObject to the
//               fax server specified by url. Returns the data decoded into
//               A Response *.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  url             char *                  URL to send the XMLF data to
//  obj             XmlfObject              pointer to XmlfObject to send
//
//
//  Returns:   A Response * containing the response from the fax server.
//
//
//  Notes:     none
//
// ---PHDR---

Response * TransportManager::send (char * url, XmlfObject * obj)
{
	return (NULL);
}


// +++PHDR+++
//
//  Method     : TransportManager::decode()
//
//  Visibility : public
//
//  Description: Decodes an XMLF document into an XmlfObject *.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  xml_data        char *                  XMLF document data
//
//
//  Returns:   A Response *.
//
//
//  Notes:     none
//
// ---PHDR---

Response * TransportManager::decode (char * xml_data)
{
	return (NULL);
}



// +++PHDR+++
//
//  Method     : TransportManager::unbindDecoder()
//
//  Visibility : public
//
//  Description: Removes the specified document type from the bindings.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  doc_type	    char *                  name of the document decoder
//
//  Returns:   TRUE on success, FALSE on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int TransportManager::unbindDecoder (char * doc_type)
{
	return (_decoders->unbind(doc_type));
}


// +++PHDR+++
//
//  Method     : TransportManager::unbindTransport()
//
//  Visibility : public
//
//  Description: Removes the specified protocol name from the bindings. 
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  protocol	    char *                  name of the transport to unbind
//
//
//  Returns:   TRUE on success, FALSE on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int TransportManager::unbindTransport (char * protocol)
{
	return (_transports->unbind(protocol));
}

