// +++FHDR+++
//
//
//   Class Name : CmdLine
//   Super Class: n/a
//
//   Description: This class encapsulates the command line passed to
//		  the main() function of an executable. It processes
//		  the arguments in the same way as Unixes getopt().
//
//
//   Notes      : none
//
//
//   Method - Description
//   ------------------------------------------------------------------------
//
//
// ---FHDR---


#include "CmdLine.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>


#define CMDLINE_NUM_CHR	 '#'
#define CMDLINE_ARG_XPTD ':'

// +++PHDR+++
//
//  Method     : CmdLine::CmdLine
//
//  Visibility : public
//
//  Description: Alternate constructor.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  optstr          const char *            option specifier string
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

CmdLine::CmdLine (const char * optstr)
{
	_optstr   = optstr;
	_optarg   = NULL;
	_optind   = 0;
	_optopt   = 0;
	_optsw    = 0;
	_charpos  = 0;
	_switches = "-";
	*_msgbuf  = '\0';
}


// +++PHDR+++
//
//  Method     : CmdLine::~CmdLine
//
//  Visibility : public
//
//  Description: Default destructor.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

CmdLine::~CmdLine ()
{
}


// +++PHDR+++
//
//  Method     : CmdLine::getOpt
//
//  Visibility : public
//
//  Description: process cmd line arguments (getopt substitute)
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  argc	    int			    number of arguments
//  argv	    char **		    argument vector
//
//
//  Returns:   option found, CMDLINE_ERROR if error or CMDLINE_EOF if at end
//
//
//  Notes:     	This routine is a replacement for the UNIX routine getopt
//		(which is not universally available).
//
// ---PHDR---

int
CmdLine::getOpt (int argc, char **argv, const char *optstr)
{
	const char *op;
	const char *lcl_optstr;
	
	if (optstr != NULL)
		lcl_optstr = optstr;
	else
		lcl_optstr = _optstr;

	_optarg = (char *)NULL;

	/* check if we should start over */
	if (_optind == 0)
	{
		_optind = 1;
		_charpos = 0;
	}

	/* if past all arguments, done */

	if (_optind >= argc)
	{
		_charpos = 0;
		return (CMDLINE_EOF);
	}

	/* check if new argument */

	if (_charpos == 0)
	{
		/* check if option & longer than 1 char */

		if (strchr(_switches, argv[_optind][0]) && argv[_optind][1])
		{
			_charpos = 1;
			_optsw = argv[_optind][0];
		}
		else
		{
			return (CMDLINE_EOF);
		}
	}

	/* get next option & bump ptr */

	_optopt = argv[_optind][_charpos++];

	/* check for double option chars */

	if (_optopt == _optsw)
	{
		_charpos = 0;
		_optind++;
		return (CMDLINE_EOF);
	}

	/* check for number */

	if (isdigit(_optopt))
	{
		for (op = lcl_optstr; *op; op++)
		{
			if (*op == CMDLINE_NUM_CHR)
			{
				_optopt = CMDLINE_NUM_CHR;
				_optarg = &argv[_optind][_charpos - 1];
				_charpos = 0;
				_optind++;
				return (_optopt);
			}
		}
	}

	/* check if at end of string of options */

	if (argv[_optind][_charpos] == '\0')
	{
		_charpos = 0;
		_optind++;
	}

	/* search for next option */

	for (op = lcl_optstr; *op; op++)
	{
		/* skip over colons */
		if (*op == CMDLINE_ARG_XPTD)
			continue;

		if (_optopt != *op)
			continue;

		/* got a matching letter in string */

		/* check if no arg expected */
		if (*(op+1) != CMDLINE_ARG_XPTD)
		{
			return (_optopt);
		}

		/* process option argument */

		if (_optind == argc)
		{
			/* at end of list */

			sprintf(_msgbuf,
				"option requires an argument -- %c", _optopt);
			return (CMDLINE_ERROR);
		}

		_optarg  = &argv[_optind][_charpos];
		_charpos = 0;
		_optind++;
		return (_optopt);
	}

	sprintf(_msgbuf, "invalid option -- %c", _optopt);
	return (CMDLINE_ERROR);
}


// +++PHDR+++
//
//  Method     : CmdLine::getOptArg
//
//  Visibility : public
//
//  Description: Returns the argument for the last parsed option.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- --------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

char *
CmdLine::getOptArg ()
{
	return (_optarg);
}



// +++PHDR+++
//
//  Method     : CmdLine::getOptIndex
//
//  Visibility : public
//
//  Description: Returns the current option index.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- --------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

int
CmdLine::getOptIndex ()
{
	return (_optind);
}


// +++PHDR+++
//
//  Method     : CmdLine::containsArg
//
//  Visibility : public
//
//  Description: Check if an arg is present
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  argc	    int			    argc from cmd line
//  argv	    char **		    argv from cmd line
//  arg		    int			    arg to look for
//
//
//  Returns:   1 if found, 0 if not
//
//
//  Notes:     none
//
// ---PHDR---

int
CmdLine::containsArg (int argc, char **argv, int arg)
{
	int c;
	int found = 0;
	char opts[2];

	opts[0] = arg;
	opts[1] = 0;

	for (;;)
	{
		if (_optind >= argc)
			break;

		c = getOpt(argc, argv, opts);
		if (c == arg)
		{
			found = 1;
			break;
		}
		else if (c == CMDLINE_EOF)
		{
			_optind++;
		}
	}

	_optind = 0;		/* reset for real pass */

	return (found);
}


// +++PHDR+++
//
//  Method     : CmdLine::getStringArg
//
//  Visibility : public
//
//  Description: Check if an arg is present
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  argc	    int			    argc from cmd line
//  argv	    char **		    argv from cmd line
//  arg		    int			    arg to look for
//
//
//  Returns:   pointer to arg if present, NULL if not
//
//
//  Notes:     none
//
// ---PHDR--- 

char *
CmdLine::getStringArg (int argc, char **argv, int arg)
{
	int c;
	char *p = (char *)NULL;
	char opts[3];

	opts[0] = arg;
	opts[1] = ':';
	opts[2] = 0;

	while (1)
	{
		if (_optind >= argc)
			break;

		c = getOpt(argc, argv, opts);
		if (c == arg)
		{
			p = _optarg;
			break;
		}
		else if (c == CMDLINE_EOF)
		{
			_optind++;
		}
	}

	_optind = 0;		/* reset for real pass */

	return (p);
}


// +++PHDR+++
//
//  Method     : CmdLine::getErrorMsg
//
//  Visibility : public
//
//  Description:
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   pointer to error message
//
//
//  Notes:     none
//
// ---PHDR---

char *
CmdLine::getErrorMsg ()
{
	return (_msgbuf);
}


// +++PHDR+++
//
//  Method     : CmdLine::setSwitches
//
//  Visibility : public
//
//  Description: Set option switches to look for
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  str		    char *		    string of switch characters
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void
CmdLine::setSwitches (const char *str)
{
	_switches = str;
}
