/*------------------------------------------------------------------------
 * Program to list server resources
 */
#include <stdio.h>
#include <stdlib.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"WU:H:Z:?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] restype\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
	fprintf(fp, "restype         Resource type wanted\n");
}

/*------------------------------------------------------------------------
 * display a resource list
 */
static void
display_resource (V_TAG *tags)
{
	int n = VSI_Tag_Count(tags, 0);
	int i;

	for (i=0; i<n; i++)
	{
		const char *name = VSI_Tag_Get_Name_By_Index(tags, i, 0, 0);
		const char *data = VSI_Tag_Get_Data_By_Index(tags, i, 0, 0);

		printf("%-16s  %s\n", name, data);
	}
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	const char *	restype;
	V_TAG *		tag_list	= 0;
	const char *	pgmname		= argv[0];
	int		opt_index;
	int		c;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * already processed
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * get restype argument
	 */
	opt_index = VSI_Opt_Index();
	if ((argc - opt_index) != 1)
	{
		fprintf(stderr, "%s: No resource type given\n", pgmname);
		usage(pgmname, stderr);
		return (EXIT_FAILURE);
	}

	restype = argv[opt_index++];

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * get list of resource tags
	 */
	tag_list = VSI_Server_Resource(vptr, restype, msgbuf);

	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	/*----------------------------------------------------------------
	 * check if any errors
	 */
	if (*msgbuf != 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * display resource list
	 */
	display_resource(tag_list);

	/*----------------------------------------------------------------
	 * free the tag list
	 */
	tag_list = VSI_Tag_Free(tag_list, 0);

	return (EXIT_SUCCESS);
}
