/*------------------------------------------------------------------------
 * Program to submit a fax request
 */
#include <stdio.h>
#include <stdlib.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"un:C:t:c:o:WU:H:Z:?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] file ...\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -u            Use user ini file\n");
	fprintf(fp, "  -n number     Phone-number\n");
	fprintf(fp, "  -C cover      Coverpage to use\n");
	fprintf(fp, "  -t tag=val    Tag to add\n");
	fprintf(fp, "  -c tagfile    Tag file to process\n");
	fprintf(fp, "  -o file       Image to file (use \"-\" for stdout)\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
	fprintf(fp, "file ...        File(s) to fax (use \"-\" for stdin)\n");
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	char		reqid[V_TAG_DATALEN];
	V_TAG *		tag_list	= 0;
	const char *	imgpath		= 0;
	const char *	pgmname		= argv[0];
	int		opt_index;
	int		c;
	int		rc;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * initialize tag list with entries from system ini file
	 */
	tag_list = VSI_Tag_Add_Ini_Sys(tag_list, msgbuf);
	if (*msgbuf != 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		const char *opt	= VSI_Opt_Argstr();

		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'u':
			/*------------------------------------------------
			 * process user ini file
			 */
			tag_list = VSI_Tag_Add_Ini_Usr(tag_list, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;
			
		case 'n':
			/*------------------------------------------------
			 * process phone-number
			 */
			tag_list = VSI_Tag_Add(tag_list, V_TAG_ENV_TOFNUM,
				opt, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 'C':
			/*------------------------------------------------
			 * process coverpage
			 */
			tag_list = VSI_Tag_Add(tag_list, V_TAG_ENV_COVERPAGE,
				opt, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 'c':
			/*------------------------------------------------
			 * process tag file
			 */
#if 1
			tag_list = VSI_Tag_Load(tag_list, opt, msgbuf);

#else	/* alternate method (defers error msgs) */

			tag_list = VSI_Tag_Add(tag_list, V_TAG_ENV_TAGFILE,
				opt, msgbuf);
#endif
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 't':
			/*------------------------------------------------
			 * process tag in form "tag=value"
			 */
			tag_list = VSI_Tag_Add_Str(tag_list, opt, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 'o':
			/*------------------------------------------------
			 * image to file
			 */
			imgpath = opt;
			break;

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * processed already
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * collect all file arguments
	 */
	opt_index = VSI_Opt_Index();
	while (opt_index < argc)
	{
		tag_list = VSI_Tag_Add(tag_list, V_TAG_ENV_FILELCL,
			argv[opt_index++], msgbuf);
		if (*msgbuf != 0)
		{
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * submit the tag list to be faxed or imaged
	 */
	if (imgpath == 0)
		rc = VSI_Server_Submit(vptr, reqid,   tag_list, msgbuf);
	else
		rc = VSI_Server_Image (vptr, imgpath, tag_list, msgbuf);

	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	/*----------------------------------------------------------------
	 * free the tag list
	 */
	tag_list = VSI_Tag_Free(tag_list, 0);

	/*----------------------------------------------------------------
	 * check if any errors
	 */
	if (rc)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * output the request-id to the user if faxing
	 */
	if (imgpath == 0)
		printf("%s\n", reqid);

	return (EXIT_SUCCESS);
}
