#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	processing for vfs installation
#
#---FHDR---

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# add a directory to the path
#
# Note that the directory is ALWAYS added to the end.
#
addpath ()
{
	dir="$1"

	sep=":"
	arg=""
	if in_windows
	then
		sep=";"
		arg="-i"
	fi

	foo=`echo "$PATH" | egrep $arg "^$dir$sep|$sep$dir$sep|$sep$dir$"`
	if [ $? -eq 0 ]
	then
		echo "$PATH"
	else
		echo "$PATH$sep$dir"
	fi

	unset dir sep arg foo
}

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# Return the Brooktrout SDK version ifor TR1034 located in a file
#
bt1034sdkver ()
{
 file_version=$1
 text=$2

 if [ "$text" = "-t" ]
 then
   sed  -e '/^#/d' -n -e's/VERSION\=\(.*\)/\1/p'  $file_version
 else
   eval `sed  -e '/^#/d' -n -e \
  's/VERSION\=\(.*\)\.\(.*\)\.\(.*\)/expr \1 \\\* 10000 + \2 \\\* 100 + \3/p' \
  $file_version`
 fi
}

# ------------------------------------------------------------------------
# main()
#
PATH=`addpath "$VSIFAX/bin"`
PATH=`addpath "$VSIFAX/lbin"`

# ------------------------------------------------------------------------
# verify that the server is installed
#
if [ ! -x "$VSIFAX/bin/vfxsched" ]
then
	echoerr ""
	echoerr "The `pkgname` server must be installed prior to this package."
	exit 1
fi

# ------------------------------------------------------------------------
# verify Brooktrout SDK used
#

SDK_INSTALLED=0
SDK_TO_INSTALL=0
SDK_INSTALLED_TXT="of the previous version"
SDK_TO_INSTALL_TXT="of the previous version"

#Check the originaly installed file
sdk_file=$VSIFAX/lib/boston/sdk_version.txt
if [ -f $sdk_file ]
then
 SDK_INSTALLED=`bt1034sdkver $sdk_file`
 SDK_INSTALLED_TXT=`bt1034sdkver $sdk_file -t`
fi

#Check the really installed one
sdk_file=/usr/sys/boston/sdk_version.txt
if [ -f $sdk_file ]
then
 SDK_INSTALLED=`bt1034sdkver $sdk_file`
 SDK_INSTALLED_TXT=`bt1034sdkver $sdk_file -t`
fi

#Check the one to be installed
sdk_file=lib/boston/sdk_version.txt
if [ -f $sdk_file ]
then
 SDK_TO_INSTALL=`bt1034sdkver $sdk_file`
 SDK_TO_INSTALL_TXT=`bt1034sdkver $sdk_file -t`
fi

#----------------------------------------
# if equal SDK  to install is newer, UPGRADE
if [ $SDK_TO_INSTALL -ne  $SDK_INSTALLED ]
then
#Do the uninstall
 if  [ -d /usr/sys/boston  -a  -x $VSIFAX/lbin/vsi-boston-uninstall.sh ]
 then
	if [ $SDK_INSTALLED -lt 40100 ]
	then
       	echo "This setup has to use Dialogic Brooktrout driver $SDK_TO_INSTALL_TXT."
	echo "The setup has to uninstall the Dialogic Brooktrout driver $SDK_INSTALLED_TXT."

       		if getyesorno -y "uninstall-bt1034" \
          		"Do you want to continue installing Dialogic Brooktrout driver \"$SDK_TO_INSTALL_TXT\"?"
       		then
	#--------------------------------------------------------------
	# Backup old files
	#
	 		BACKUP_STR=`date +%Y%m%d-%H%M%S`
	 		[ -f /usr/sys/boston/btcall.cfg ] && cp /usr/sys/boston/btcall.cfg lib/boston/btcall.cfg.$BACKUP_STR

	 		[ -f /usr/sys/boston/callctrl.cfg ] && cp /usr/sys/boston/callctrl.cfg lib/boston/callctrl.cfg.$BACKUP_STR

	 		[ -f /usr/sys/boston/ecc.cfg ] && cp /usr/sys/boston/ecc.cfg lib/boston/ecc.cfg.$BACKUP_STR
	#---------------------------------------------------------------
	# Do the uninstallation
	#
         		$VSIFAX/lbin/vsi-boston-uninstall.sh -f
       		else
        		abort_install
       		fi
	else
       		echo "The setup has to update the Dialogic Brooktrout driver \"$SDK_INSTALLED_TXT\""
		echo "with version \"$SDK_TO_INSTALL_TXT\"."

       		getyesorno -y "update-bt1034" \
          		"Do you want to continue installing Dialogic Brooktrout driver \"$SDK_TO_INSTALL_TXT\"?" || abort_install
		rm $VSIFAX/lib/boston/fw/flashupd.bin 2> /dev/null
		rm $VSIFAX/lib/boston/lib/7/osilib_mt.so 2> /dev/null
		rm $VSIFAX/lib/boston/lib/3.4/osilib_mt.so 2> /dev/null
	fi
  fi
fi

exit 0
