#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#        master install script for Brooktrout TR1034 boards
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# get name of the NULL device
#
get_dev_null ()
{
	if in_windows
	then
		echo "nul:"
	else
		echo "/dev/null"
	fi
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# output a prompt & get a response from the user
#
get_user_input ()
{
	id="$1"
	dflt="$2"
	msg="$3"

	ok_id="$id-ok"
	[ "$id" = "" ] && ok_id=""

	CONSOLE=`get_dev_tty`

	while true
	do
		if [ x"$dflt" = x"!" ]
		then
			echonl "Enter $msg: "
		else
			echonl "Enter $msg [$dflt]: "
		fi >$CONSOLE

		response=`read_user_input "$id"`
		[ "$response" = "" -a x"$dflt" != x"!" ] && response="$dflt"

		echo "$msg = \"$response\"" >$CONSOLE
		is_ok "$ok_id" >$CONSOLE && { echo "$response"; break; }

		dflt="$response"
	done
}

# ------------------------------------------------------------------------
# get Windows directory
#
get_windir ()
{
	{
		if [ "$windir" != "" ]
		then
			echo "$windir"

		elif [ "$WINDIR" != "" ]
		then
			echo "$WINDIR"

		elif [ "$SystemRoot" != "" ]
		then
			echo "$SystemRoot"

		elif [ "$SYSTEMROOT" != "" ]
		then
			echo "$SYSTEMROOT"

		elif [ -d "c:/windows" ]
		then
			echo "c:/windows"

		elif [ -d "c:/winnt" ]
		then
			echo "c:/winnt"

		else
			echo "c:/windows"
		fi
	} | sed -e 's,\\\\*,/,g'
}

# ------------------------------------------------------------------------
# get Windows drive
#
get_windrive()
{
	get_windir | sed -e 's,[/\\].*,,'
}

#-------------------------------------------------------------------------
# install the board driver (Linux intel version)
#
install_driver ()
{
	# ----------------------------------------------------------------
	# Install parameters:
	#
	# Num ISA modules	0
	# ISA intr line		-1
	# Max PCI modules	16
	# Physical buffer size	32768
	# App buffer size	10240
	# Reserved	 	200
	# Machine ID		1
	# Reset option		1
	# History enable	0
	# History size		1024000
	# Num physical hists	0
	# Num app hists		0
	# Restrict phys chan	1
	# Num I2O frames	0
	# Flow control interval	0
	# Mem alloc min		0
	# Mem alloc quanta	0
	# Reserved		0
	# Test param 1		0
	# Test param 2		0
	# Test param 3		0
	# Reserved mod info	0
	# Reserved		0
	#
	INST_OPTS="-c 0 -1 16 32768 10240 0 1 1 0 1024000 0 0 1 0 0 0 0 0 0 0 0 0 0"

	./dinstall $INST_OPTS

	if [ $? -eq 2 ]
	then
		echo ""
echoerr "By default, the support for your current kernel version is not included. You"
echoerr "need to recompile the driver so it can operate with the kernel currently"
echoerr "running on your system."
		echo ""
echoerr "Please follow the instructions provided in the topic 'Recompiling on Linux"
echoerr "Platforms' in the Fax Boards and Fax Drivers Installation guide. The source"
echoerr "files that will let you recompile the driver have been installed"
echoerr "under /usr/sys/boston/kernel. Once the driver has been successfully"
echoerr "recompiled, please resume the setup procedure by relaunching"
echoerr "$VSIFAX/lbin/vsi-boston-config.sh"
		echo ""
		abort_install
	fi
	. /etc/startbost

}

#-------------------------------------------------------------------------
# display usage
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -r       Read response file"
	echo "  -f       No board/firmware load"
	echo "  -i       No driver install"
	echo "  -c       No configuration needed"
	echo "  -x       Remove any configurations"
	echo "  -d       Dialogic Brooktrout cmd debug"
}

#-------------------------------------------------------------------------
# helpall usage
#
helpall ()
{
	usage

	echo "  -B dir   Specify location of Boston directory"
	echo "  -C code  Specify country code"
	echo "  -D       Debug install script"
}



#-------------------------------------------------------------------------
# get board ip addresses for SR140
#
select_board_sr140_ip_interface()
{
	echo ""
	echo "The following information is needed to configure the IP network interface of the SR140 virtual board."
	echo ""
	echo "You can type your network interface name (eth0, eth1 ...) or you can type 'default' to use your default network interface"
	NET_INTERFACE=`get_user_input bt_interface_name "default" "Network interface name"`
	echo ""
	echo ""
	return 0
}

#-------------------------------------------------------------------------
# get board ip addresses
#
select_board_ip_add()
{
	echo ""
	echo "The following information is needed to configure the IP interface of the board #$MODULE:"
	echo ""
	echo "board ip address"
	echo "board ip mask"
	echo "board ip gateway"
	echo "board ip broadcast"
	echo ""
	while true
	do
		NET_IP=`get_user_input bt_ip "0.0.0.0" "board ip address"`
		NET_MASK=`get_user_input bt_mask "0.0.0.0" "board ip mask"`
		NET_GATEWAY=`get_user_input bt_gateway "0.0.0.0" "board ip gateway"`
		NET_BROADCAST=`get_user_input bt_broadcast "0.0.0.0" "board ip broadcast"`
		echo ""
		echo "board ip address = $NET_IP"
		echo "board ip mask     = $NET_MASK"
		echo "board ip gateway  = $NET_GATEWAY"
		echo "board ip broadcast= $NET_BROADCAST"
		echo ""
		is_ok "confirm-ip-$MODULE-ok" && break
	done
	echo ""
	return 0
}

#-------------------------------------------------------------------------
# get  H323 parameters
#
select_param_H323()
{
	echo ""
	echo "The following information is needed to configure the H323 stack:"
	echo ""
	echo "H323 default gateway"
	echo "H323 gatekeeper ip address"
	echo ""
	while true
	do
	H323_DFLT_GATEWAY=`get_user_input bt_h323_gateway "" "H323 default gateway"`
	H323_GATEKEEPER_IP=`get_user_input bt_h323_gatekeeper_ip "" "H323 gatekeeper ip address"`
	echo ""
	echo "H323 default gateway        = $H323_DFLT_GATEWAY"
	echo "H323 gatekeeper ip address = $H323_GATEKEEPER_IP"
	echo ""
	is_ok "confirm-H323-ok" && break
	done
	echo ""
	return 0
}

#-------------------------------------------------------------------------
# get  SIP parameters
#
select_param_SIP()
{
	echo ""
	echo "The following information is needed to configure the SIP stack:"
	echo ""
	echo "SIP default gateway"
	echo ""
	SIP_DFLT_GATEWAY=`get_user_input bt_sip_gateway "" "SIP default gateway"`
	echo ""
        echo ""

	return 0
}
#-------------------------------------------------------------------------
# get T38 parameters for H323 gateway
#
select_t38_H323 ()
{
	return 0
}

#-------------------------------------------------------------------------
# get T38 parameters for SIP gateway
#
select_t38_SIP ()
{
	return 0
}

#-------------------------------------------------------------------------
# get country code
#
select_country_code ()
{
	if [ "$COUNTRY_CODE" = "" ]
	then
		echo ""
		if getyesorno "in_usa" "Are you in the USA or Canada?"
		then
			COUNTRY_CODE=1
		else
			echo ""
			echo "Enter your country code."
			COUNTRY_CODE=`get_user_input country_code "!" \
				"country code"`
		fi
	fi

	COUNTRY_CODE="${COUNTRY_CODE}0"
}

# -----------------------------------------------------------------------
# select SIP or H323
#
select_ipstack_type ()
{
	while true
	do
		echo ""
		echo "Select IP stack for SR140 virtual board."
		echo ""
		echo "1 SIP"
		echo "2 H323"
		echo "q Quit"
		echo ""

		CIRCUIT=`get_user_input bt_ip_stack $1 \
			"ip stack"`

		case $CIRCUIT in
			1)	PORT_CONFIG="SR140_SIP";	break	;;
			2)	PORT_CONFIG="SR140_H323";break	;;
			q|Q)	return 1 ;;

			*)	echoerr "Invalid choice" ;;
		esac
	done

	return 0
}

# -----------------------------------------------------------------------
# select RBS or PRI
#
select_circuit_type ()
{
	board_type=$2
	while true
	do
		echo ""
		echo "Select circuit type for this ${board_type}1 board."
		echo ""
		[ "$board_type" = "T" ] && echo "0 Robbed-bit (RBS)"
		echo "1 ISDN       (PRI)"
		echo "2 SIP         (IP)"
		echo "3 H323        (IP)"
		echo "q Quit"
		echo ""

		CIRCUIT=`get_user_input bt_circuit_type $1 \
			"circuit type"`

		case $CIRCUIT in
			0)
			if [ $board_type = "T" ]
			then
				PORT_CONFIG="T1_ROBBED_BIT"
				break
			else
				echoerr "Invalid choice"
			fi
			;;

			1)	PORT_CONFIG="${board_type}1_ISDN";	break	;;
			2)	PORT_CONFIG="SIP";		break	;;
			3)	PORT_CONFIG="H323";		break	;;

			q|Q)	return 1 ;;

			*)	echoerr "Invalid choice" ;;
		esac
	done

	return 0
}



list_license_fields()
{
        cat $1 |
        sed -e :a -e '$!N; s/\n/ /; ta'  |
        sed -e :a -e 's/ \\//; ta' -e 's/\t//g '  |
	sed -e :a -e 's/ \([^ ="]*=\)"\([^"]*\)"/\n\1"\2"/ ;ta' |
	sed -e :a -e 's/ \([^ =]*=\)\([^ ]*\)/\n\1\2/ ;ta'
}
get_licensed_field_HOSTID()
{
	my_file=$1
	list_license_fields $my_file |
        awk -F '=' '{  if  ( $1 ~ /HOSTID/ ) {print "NODETYPE=" $2 "  " "NODEID=\"" $3 "\""; } }'
}

get_licensed_field_ISSUED()
{
	my_file=$1
	list_license_fields $my_file |
        awk -F '=' '{  if  ( $1 ~ /ISSUED/ ) {print "ISSUED=\"" $2 "\""; } }'
}
get_licensed_field_VENDOR()
{
	my_file=$1
	list_license_fields $my_file |
	awk -F '=' '{  if  ( $1 ~ /VENDOR_STRING/ ) {line=$2; for (i=3; i<=NF; i++) { line= line "=" $i;}; print line; } }'  |
        sed -n -e 's/:/ /gp'

}

#-------------------------------------------------------------------------
#  Get SR140 license file and number of licensed channels
#
get_sr140_channels()
{
	CHANNELS=0
	while true
	do
		unset current_node
		current_node=`$BOSTON_DIR/listnodeid|sed -e 's/\([^:]*: *\)\(.*\)/\2/'`
		echo ""
		echo "Did you already generate SR140 Fax Software license for this computer with NodeId:"
		echo "$current_node"
		if  is_ok "is-sr140-license-generated"
		then
			echo ""
			echo "Enter the source path of the SR140 Fax Software license file:"
			echo ""
			SR140_LICENSE=`get_user_input bt_sr140_license_file "" "SR140 license file"`
			if [ -f $SR140_LICENSE -a -s $SR140_LICENSE ]
			then
				echo ""
				echo "Information from $SR140_LICENSE"
				my_test=`get_licensed_field_ISSUED $SR140_LICENSE`
				unset ISSUED
				eval $my_test

				my_test=`get_licensed_field_HOSTID $SR140_LICENSE`
				unset $NODEID
				unset $NODETYPE
				eval $my_test

				my_test=`get_licensed_field_VENDOR $SR140_LICENSE`
				unset T38Enab
				eval $my_test
				echo "License generated: $ISSUED"
				echo "License for Nodeid: $NODEID"
				echo "License node type: $NODETYPE"
				echo "Licensed channels: $T38Enab"
#
# Check license information
#
        			if [ "$T38Enab" = "" ]
        			then
                			echo "Cannot read number of licensed channels from $SR140_LICENSE."
                			echo "May be SR140 licensed file is corrupted."
					echo "Quit the setup now and retry later"
					is_ok "is-sr140-license-generated-quit" && return 1
					continue
				fi
				if [ ! "$current_node" = "$NODEID" ]
				then
					echo ""
					echo "SR140 license file was generated for '$NODEID' "
					echo "and the NodeId for this computer is '$current_node'"
					echo ""
					echo "Generate first your SR140 Fax Software license from Dialogic web site. "
					echo "When you are asked for Node Id enter this value:'$current_node'."
					echo ""
					echo "You can use this tool '$BOSTON_DIR/listnodeid' to display at any time the NodeId of this computer"
					echo  ""
					echo "Quit the setup now and retry later"
					is_ok "is-sr140-license-generated-quit" && return 1
					continue
				fi
				my_file=`echo $SR140_LICENSE | sed -e 's/\(.*\/\)//g'`
				cp -f $SR140_LICENSE $BOSTON_DIR/$my_file
				CHANNELS=$T38Enab
				break
			else
				echo ""
				echo "SR140 license file does not exist or corrupted."
				echo "Quit the setup now and retry later"
				is_ok "is-sr140-license-generated-quit" && return 1
				continue
			fi
			echo ""
		else
			echo ""
			echo "IMPORTANT:"
			echo "You have first to generate your SR140 Fax Software license from Dialogic web site and get the license file."
			echo "You have also to know that the NodeId for this computer needed when generating the license file is:"
			echo "$current_node"
			echo "At any time, you can use this tool '$BOSTON_DIR/listnodeid' to display the NodeId of this computer."
			echo  ""
			echo "Quit the setup now and retry later"
			is_ok "is-sr140-license-generated-quit" && return 1
			continue
		fi
	done

	echo  ""
	while true
	do
		CHANNELS=`get_user_input bt_board_channels_65 $CHANNELS \
			"number of channels to use with the SR140 virtual board"`
		if [ "$CHANNELS" = "" -o $CHANNELS -lt 0 -o $CHANNELS -gt $T38Enab ]
		then
			CHANNELS=$T38Enab
			 continue
		fi
		break;
	done
}

#-------------------------------------------------------------------------
# get board type
#
select_board_type ()
{
	unset TYPE
	while true
	do
		echo ""
		echo "Select the Dialogic Brooktrout board or SR140 Fax Software to configure:"
		echo " 1) TR1034 -   T1   - 24 channels"
		echo " 2) TR1034 -   T1   - 16 channels"
		echo " 3) TR1034 -   T1   -  8 channels"
		echo " 4) TR1034 -   T1   -  4 channels"

		echo " 5) TR1034 -   E1   - 30 channels"
		echo " 6) TR1034 -   E1   - 20 channels"
		echo " 7) TR1034 -   E1   - 16 channels"
		echo " 8) TR1034 -   E1   - 10 channels"
		echo " 9) TR1034 -   E1   -  8 channels"
		echo "10) TR1034 -   E1   -  4 channels"

		echo "11) TR1034 - Analog -  8 channels"
		echo "12) TR1034 - Analog -  4 channels"
		echo "13) TR1034 - Analog -  2 channels"

		echo "14) TruFax - Analog -  2 channels"
		echo "15) TruFax - Analog -  1 channel"

		echo "16) TR1034 -   BRI  -  4 channels"
		echo "17) TR1034 -   BRI  -  2 channels"

		echo "18) TruFax -   BRI  -  4 channels"
		echo "19) TruFax -   BRI  -  2 channels"

		echo "20) SR140  -   IP   -  The number of channels depends on the license"

		echo " q) Quit"
		echo ""

		CHOICE=`get_user_input bt_board_type_$MODULE_COUNT $1 \
			"your choice"`

		case "$CHOICE" in
			1)	TYPE=T; CHANNELS=24	;;
			2)	TYPE=T; CHANNELS=16	;;
			3)	TYPE=T; CHANNELS=8	;;
			4)	TYPE=T; CHANNELS=4	;;

			5)	TYPE=E; CHANNELS=30	;;
			6)	TYPE=E; CHANNELS=20	;;
			7)	TYPE=E; CHANNELS=16	;;
			8)	TYPE=E; CHANNELS=10	;;
			9)	TYPE=E; CHANNELS=8	;;
			10)	TYPE=E; CHANNELS=4	;;

			11)	TYPE=A; CHANNELS=8	;;
			12)	TYPE=A; CHANNELS=4	;;
			13)	TYPE=A; CHANNELS=2	;;

			14)	TYPE=A; CHANNELS=2	;;
			15)	TYPE=A; CHANNELS=1	;;

			16)	TYPE=B; CHANNELS=4	;;
			17)	TYPE=B; CHANNELS=2	;;

			18)	TYPE=B; CHANNELS=4	;;
			19)	TYPE=B; CHANNELS=2	;;

			20)	TYPE=V; get_sr140_channels || return 1
			;;

			q|Q)	return 1 ;;

			*)	echoerr "Invalid choice"
				continue ;;
		esac

		case "$TYPE" in
			V)
			select_ipstack_type 1 $TYPE || return 1
				;;
			T)
			select_circuit_type 0 $TYPE || return 1
			[ "$PORT_CONFIG" = "T1_ISDN" -a $CHANNELS -eq 24 ] && CHANNELS=23
			;;

			E) 	select_circuit_type 1 $TYPE || return 1
				;;

			A)	PORT_CONFIG="analog"
				;;

			B)	PORT_CONFIG="BRI"
				;;
		esac

		break
	done

	BOARD_TYPE=$CHOICE
	return 0
}

#-------------------------------------------------------------------------
# get board module number
#
select_module ()
{
	module=$1

	while true
	do
		if [ "$TYPE" != "V" ]
		then
			echo ""
			echo "Select the module number [2-15] for this board."
			echo ""

			CHOICE=`get_user_input bt_module_$MODULE_COUNT $module \
				"module number"`

			case "$CHOICE" in
				2)	MODULE=2 ;;
				3)	MODULE=3 ;;
				4)	MODULE=4 ;;
				5)	MODULE=5 ;;
				6)	MODULE=6 ;;
				7)	MODULE=7 ;;
				8)	MODULE=8 ;;
				9)	MODULE=9 ;;
				10|A|a)	MODULE=A ;;
				11|B|b)	MODULE=B ;;
				12|C|c)	MODULE=C ;;
				13|D|d)	MODULE=D ;;
				14|E|e)	MODULE=E ;;
				15|F|f)	MODULE=F ;;

				q|Q)	return 1 ;;

				*)	echoerr "Invalid choice"
					continue ;;
			esac
		else
			MODULE=41
		fi

		if is_module_used $MODULE
		then
			echoerr ""
			echoerr "Module $MODULE already defined."
			continue
		fi

		set_module_used $MODULE
		break
	done

	return 0
}

#-------------------------------------------------------------------------
# create the btcall configuration file
#
write_btcall_file ()
{
	{
		echo "# $BTCALL_CFG"
		echo "agc                1"
		echo "badline_behavior   0"
		echo "bt_cparm           $BOSTON_DIR/config/BT_CPARM.CFG"
		echo "busy_dt_ct         1"
		echo "call_control       $BOSTON_DIR/$CALLCT_CFG"
		echo "ced_timeout        4000"
		echo "country_code       $COUNTRY_CODE"
		echo "debug_control      9 1"
		echo "debug_control      e b"
		echo "dtmf_thresh        0"
		echo "ecm_enable         1"
		echo "eff_pt_caps        0"
		echo "error_enable       1"
		echo "error_mult         40"
		echo "error_mult_rtp     200"
		echo "error_thresh       3"
		echo "fax_rtp_enable     1"
		echo "font_file          ibmpcps.fz8"
		echo "id_string          20"
		echo "line_compression   5"
		echo "line_encoding      0"
		echo "max_pagelist       30"
		echo "max_timeout        600"
		echo "max_width          0"
		echo "min_length         0"
		echo "restrict_res       1"
		echo "silcompr_start     500"
		echo "silcompr_middle    1000"
		echo "subpwdsep          1"
		echo "v_play_gain        0"
		echo "v_timeout          60"
		echo "v34_2400_baud_ctrl 1"
		echo "v34_ci_enable      1"
		echo "v34_enable         1"
		echo "width_res_behavior 1"
	} > $BTCALL_CFG
}

# ----------------------------------------------------------------------
# init the callctrl configuration file
#
init_callctrl_file ()
{
	{
		echo "# $CALLCT_CFG"
		echo "#"
		echo
		echo "   l3l4_trace=none"
		echo "   l4l3_trace=none"
		echo "   api_trace=none"
		echo "   internal_trace=none"
		echo "   host_module_trace=none"
		echo "   ip_stack_trace=none"
		echo "   trace_file=$BOSTON_DIR/ecc.log"
		echo "   max_trace_files_size=10"
		echo "   max_trace_files=1"
		echo
	} > $CALLCT_CFG
}

# ----------------------------------------------------------------------
# write the module entry for T1 RBS modules
#
write_callctrl_module_T1_ROBBED_BIT ()
{
        module=$1
	echo "[module.$module]"
	echo "    set_api=ECC"
	echo "    auto_connect=TRUE"
        echo "    pcm_law=mulaw"

        echo "[module.$module/clock_config]"
        echo "    clock_mode=master"
        echo "    clock_source=TrunkA"

        echo "[module.$module/port.1]"
        echo "    emulation=CPE"
        echo "    port_config=T1_ROBBED_BIT"
        echo "    protocol_file=$BOSTON_DIR/config/winkstart.lec"
        echo "    line_build_out=0_133"
        echo "    line_coding=B8ZS"
        echo "    line_type=ESF"
        echo "    max_did_digits=4"
        echo "    did_timeout=10"
        echo "    reject_incomplete_did=false"

        echo "[module.$module/port.2]"
        echo "    emulation=CPE"
        echo "    port_config=T1_ROBBED_BIT"
        echo "    protocol_file=$BOSTON_DIR/config/winkstart.lec"
        echo "    line_build_out=0_133"
        echo "    line_coding=B8ZS"
        echo "    line_type=ESF"
        echo "    max_did_digits=4"
        echo "    did_timeout=10"
        echo "    reject_incomplete_did=false"
	echo
}

#-----------------------------------------------------------------------
# write t38 for SIP
#
write_callctrl_t38_SIP()
{
	echo "[host_module.$sipdone/t38parameters]"
	echo "    t38_fax_rate_management=transferredTCF"
	echo "    t38_fax_udp_ec=t38UDPRedundancy"
	echo "    t38_max_bit_rate=14400"
	echo "    media_renegotiate_delay_inbound=1000"
	echo "    media_renegotiate_delay_outbound=2000"
	echo "    t38_fax_fill_bit_removal=false"
	echo "    t38_fax_transcoding_jbig=false"
	echo "    t38_fax_transcoding_mmr=false"
	echo "    t38_t30_fastnotify=true"
	echo "    t38_UDPTL_redundancy_depth_control=5"
	echo "    t38_UDPTL_redundancy_depth_image=2"
}

#-----------------------------------------------------------------------
# write the SIP ip stack
#
write_callctrl_host_SIP()
{
  	echo "[host_module.$sipdone]"
  	echo "   module_library=/usr/lib/brktsip_mt.so"
  	echo "   enabled=true"
  	echo "[host_module.$sipdone/parameters]"
	echo "    sip_Contact="
	echo "    sip_default_gateway=$SIP_DFLT_GATEWAY"
	echo "    sip_description_URI="
	echo "    sip_email="
	echo "    sip_From=Anonymous <sip:no_from_info@anonymous.invalid>"
	echo "    sip_Max-Forwards=70"
	echo "    sip_max_sessions=256"
	echo "    sip_phone="
	echo "    sip_proxy_server1="
	echo "    sip_proxy_server2="
	echo "    sip_proxy_server3="
	echo "    sip_proxy_server4="
	echo "    sip_registration_interval=60"
	echo "    sip_registration_server1="
	echo "    sip_registration_server1_aor="
	echo "    sip_registration_server1_expires=3600"
	echo "    sip_registration_server1_password="
	echo "    sip_registration_server1_username="
	echo "    sip_registration_server2="
	echo "    sip_registration_server2_aor="
	echo "    sip_registration_server2_expires=3600"
	echo "    sip_registration_server2_password="
	echo "    sip_registration_server2_username="
	echo "    sip_registration_server3="
	echo "    sip_registration_server3_aor="
	echo "    sip_registration_server3_expires=3600"
	echo "    sip_registration_server3_password="
	echo "    sip_registration_server3_username="
	echo "    sip_registration_server4="
	echo "    sip_registration_server4_aor="
	echo "    sip_registration_server4_expires=3600"
	echo "    sip_registration_server4_password="
	echo "    sip_registration_server4_username="
	echo "    sip_Route="
	echo "    sip_session_description="
	echo "    sip_session_name=no_session_name"
	echo "    sip_username=-"
}
#-----------------------------------------------------------------------
# write the SR140 module SIP entry
#
write_callctrl_module_SR140_SIP ()
{
	module=$1
	if [ "$sipwritten" = "" ]
	then
		write_callctrl_host_SIP
		write_callctrl_t38_SIP
		sipwritten="yes"
		echo
	fi
	echo "[module.$module]"
  	echo "    vb_firm=$BOSTON_DIR/fw/bostvb.so"
  	echo "    channels=$CHANNELS"
  	echo "[module.$module/ethernet.1]"
	if [ "$NET_INTERFACE" = "default" ]
	then
  		echo "    ip_interface="
	else
  		echo "    ip_interface=$NET_INTERFACE"
	fi
  	echo "    media_port_min=56000"
  	echo "    media_port_max=57000"
  	echo "[module.$module/host_cc.1]"
  	echo "    host_module=$sipdone"
  	echo "    number_of_channels=$CHANNELS"
	echo
}
#-----------------------------------------------------------------------
# write the module entry for SIP modules
#
write_callctrl_module_SIP ()
{
	module=$1
	if [ "$sipwritten" = "" ]
	then
		write_callctrl_host_SIP
		write_callctrl_t38_SIP
		sipwritten="yes"
		echo
	fi
	echo "[module.$module]"
  	echo "    set_api=bfv"
  	echo "    auto_connect=true"
  	echo "    pcm_law=mulaw"
  	echo "    static_ring_detect_enable=true"
  	echo "[module.$module/port.1]"
  	echo "    port_config=inactive"
  	echo "[module.$module/clock_config]"
  	echo "    clock_mode=master"
  	echo "    clock_source=internal"
  	echo "    clock_compatibility=none"
  	echo "    bus_speed=2"
  	echo "    master_ref_fallback=disabled"
  	echo "    master_drive=clock_a"
  	echo "[module.$module/ethernet.1]"
  	echo "    dhcp=disabled"
  	echo "    ip_address=$NET_IP"
  	echo "    ip_netmask=$NET_MASK"
  	echo "    ip_gateway=$NET_GATEWAY"
  	echo "    ip_broadcast=$NET_BROADCAST"
  	echo "    media_port_min=56000"
  	echo "    media_port_max=57000"
  	echo "    ethernet_speed=auto"
  	echo "    ethernet_duplex=half"
  	echo "    ethernet_flow_control=auto"
  	echo "    ip_arp_timeout=10"
  	echo "[module.$module/host_cc.1]"
  	echo "    host_module=$sipdone"
  	echo "    number_of_channels=$CHANNELS"
	echo
}

#-----------------------------------------------------------------------
# write t38 for H323
#
write_callctrl_t38_H323()
{
  	echo "[host_module.$h323done/t38parameters]"
  	echo "    t38_fax_rate_management=transferredTCF"
  	echo "    t38_fax_udp_ec=t38UDPRedundancy"
  	echo "    t38_max_bit_rate=14400"
  	echo "    media_renegotiate_delay_inbound=1000"
  	echo "    media_renegotiate_delay_outbound=2000"
  	echo "    t38_fax_fill_bit_removal=false"
  	echo "    t38_fax_transcoding_jbig=false"
  	echo "    t38_fax_transcoding_mmr=false"
  	echo "    t38_t30_fastnotify=false"
  	echo "    t38_UDPTL_redundancy_depth_control=5"
  	echo "    t38_UDPTL_redundancy_depth_image=2"
}

#-----------------------------------------------------------------------
# write the H323 ip stack
#
write_callctrl_host_H323()
{
  	echo "[host_module.$h323done]"
  	echo "    module_library=/usr/lib/brkth323_mt.so"
  	echo "    enabled=true"
  	echo "[host_module.$h323done/parameters]"
  	echo "    h323_default_gateway=$H323_DFLT_GATEWAY"
  	echo "    h323_e164alias="
  	echo "    h323_gatekeeper_id="
  	echo "    h323_gatekeeper_ip_address=$H323_GATEKEEPER_IP"
  	echo "    h323_gatekeeper_ttl=0"
  	echo "    h323_h323IDalias="
  	echo "    h323_local_ip_address="
  	echo "    h323_max_sessions=256"
  	echo "    h323_register=0"
  	echo "    h323_support_alternate_gk=0"
}
#-----------------------------------------------------------------------
# write the SR140 module H323 entry
#
write_callctrl_module_SR140_H323 ()
{
	module=$1
	if [ "$h323written" = "" ]
	then
		write_callctrl_host_H323
		write_callctrl_t38_H323
		h323written="yes"
		echo
	fi
	echo "[module.$module]"
  	echo "    vb_firm=$BOSTON_DIR/fw/bostvb.so"
  	echo "    channels=$CHANNELS"
  	echo "[module.$module/ethernet.1]"
	if [ "$NET_INTERFACE" = "default" ]
	then
  		echo "    ip_interface="
	else
  		echo "    ip_interface=$NET_INTERFACE"
	fi
  	echo "    media_port_min=56000"
  	echo "    media_port_max=57000"
  	echo "[module.$module/host_cc.1]"
  	echo "    host_module=$h323done"
  	echo "    number_of_channels=$CHANNELS"
	echo
}

#-----------------------------------------------------------------------
# write the module entry for H323 modules
#
write_callctrl_module_H323 ()
{
	module=$1
	if [ "$h323written" = "" ]
	then
		write_callctrl_host_H323
		write_callctrl_t38_H323
		h323written="yes"
		echo
	fi
	echo "[module.$module]"
  	echo "    set_api=bfv"
  	echo "    auto_connect=true"
  	echo "    pcm_law=mulaw"
  	echo "    static_ring_detect_enable=true"
  	echo "[module.$module/port.1]"
  	echo "    port_config=inactive"
  	echo "[module.$module/clock_config]"
  	echo "    clock_mode=master"
  	echo "    clock_source=internal"
  	echo "    clock_compatibility=none"
  	echo "    bus_speed=2"
  	echo "    master_ref_fallback=disabled"
  	echo "    master_drive=clock_a"
  	echo "[module.$module/ethernet.1]"
  	echo "    dhcp=disabled"
  	echo "    ip_address=$NET_IP"
  	echo "    ip_netmask=$NET_MASK"
  	echo "    ip_gateway=$NET_GATEWAY"
  	echo "    ip_broadcast=$NET_BROADCAST"
  	echo "    media_port_min=56000"
  	echo "    media_port_max=57000"
  	echo "    ethernet_speed=auto"
  	echo "    ethernet_duplex=half"
  	echo "    ethernet_flow_control=auto"
  	echo "    ip_arp_timeout=10"
  	echo "[module.$module/host_cc.1]"
  	echo "    host_module=$h323done"
  	echo "    number_of_channels=$CHANNELS"
	echo
}
# ----------------------------------------------------------------------
# write the module entry for T1 PRI modules
#
write_callctrl_module_T1_ISDN ()
{
        module=$1
	echo "[module.$module]"
	echo "    set_api=ECC"
	echo "    auto_connect=TRUE"
        echo "    pcm_law=mulaw"

        echo "[module.$module/clock_config]"
        echo "    clock_mode=master"
        echo "    clock_source=TrunkA"

        echo "[module.$module/port.1]"
        echo "    emulation=CPE"
        echo "    port_config=T1_ISDN"
        echo "    call_type=auto"
        echo "    line_build_out=0_133"
        echo "    line_coding=B8ZS"
        echo "    protocol=ATT"
        echo "    max_did_digits=4"
        echo "    did_timeout=10"
        echo "    reject_incomplete_did=false"
        echo "    switch_type=UNKNOWN"

        echo "[module.$module/port.2]"
        echo "    emulation=CPE"
        echo "    port_config=T1_ISDN"
        echo "    call_type=auto"
        echo "    line_build_out=0_133"
        echo "    line_coding=B8ZS"
        echo "    protocol=ATT"
        echo "    max_did_digits=4"
        echo "    did_timeout=10"
        echo "    reject_incomplete_did=false"
        echo "    switch_type=UNKNOWN"
	echo
}

# ----------------------------------------------------------------------
# write the module entry for E1 ISDN modules
#
write_callctrl_module_E1_ISDN ()
{
        module=$1
	echo "[module.$module]"
	echo "    set_api=ECC"
	echo "    auto_connect=TRUE"
	echo "    pcm_law=alaw"

	echo "[module.$module/clock_config]"
	echo "    clock_mode=master"
	echo "    clock_source=TrunkA"

	echo "[module.$module/port.1]"
	echo "    emulation=CPE"
	echo "    port_config=E1_ISDN"
	echo "    protocol=EURO"
	echo "    line_coding=HDB3"
	echo "    line_impedance=120"
	echo "    call_type=auto"
	echo "    max_did_digits=4"
	echo "    did_timeout=10"
	echo "    reject_incomplete_did=false"
	echo "    switch_type=UNKNOWN"

	echo "[module.$module/port.2]"
	echo "    emulation=CPE"
	echo "    port_config=E1_ISDN"
	echo "    protocol=EURO"
	echo "    line_coding=HDB3"
	echo "    line_impedance=120"
	echo "    call_type=auto"
	echo "    max_did_digits=4"
	echo "    did_timeout=10"
	echo "    reject_incomplete_did=false"
	echo "    switch_type=UNKNOWN"
	echo
}

# ----------------------------------------------------------------------
# write the module entry for BRI modules
#
write_callctrl_module_BRI ()
{
        module=$1
	echo "[module.$module]"
	echo "    set_api=ECC"
	echo "    auto_connect=TRUE"
	echo "    pcm_law=alaw"

	echo "[module.$module/port.1]"
	echo "    emulation=CPE"
	echo "    port_config=BRI"
	echo "    call_type=auto"
	echo "    datalink=auto"
	echo "    did_timeout=10"
	echo "    max_did_digits=4"
	echo "    reject_incomplete_did=false"

	echo "[module.$module/port.2]"
	echo "    emulation=CPE"
	echo "    port_config=BRI"
	echo "    call_type=auto"
	echo "    datalink=auto"
	echo "    did_timeout=10"
	echo "    max_did_digits=4"
	echo "    reject_incomplete_did=false"
	echo
}

# ----------------------------------------------------------------------
# write the module entry for analog modules
#
write_callctrl_module_analog ()
{
        module=$1
	echo "[module.$module]"
	echo "    set_api=ECC"
	echo "    auto_connect=TRUE"
	echo "    pcm_law=mulaw"

	i=1
	while [ $i -le $CHANNELS ]
	do

	echo "[module.$module/port.$i]"
	echo "    port_config=analog"
	if [ "$COUNTRY_CODE" = "10" ]
	then
	echo "    protocol_file=$BOSTON_DIR/config/analog_loopstart_us.lec"
	echo "    country=$BOSTON_DIR/config/us600.qslac"
	else
	echo "    protocol_file=$BOSTON_DIR/config/analog_loopstart_europe.lec"
	echo "    country=$BOSTON_DIR/config/ctr21.qslac"
	fi
	echo "    callerid=enabled"
	echo "    max_did_digits=4"
	echo "    num_rings=2"
	echo "    input_gain=0"
	echo "    output_gain=0"

	i=`expr $i + 1`
	done
	echo
}


select_ip_info()
{
	if [  "$PORT_CONFIG" = "SR140_SIP" -o "$PORT_CONFIG" = "SR140_H323" ]
	then
		select_board_sr140_ip_interface
		if [  "$PORT_CONFIG" = "SR140_SIP" -a "$sipdone" = "" ]
		then
			select_t38_SIP
			select_param_SIP
			nbipstack=`expr $nbipstack + 1`
			sipdone=$nbipstack
		fi

		if [ "$PORT_CONFIG" = "SR140_H323" -a  "$h323done" = "" ]
		then
			select_t38_H323
			select_param_H323
			nbipstack=`expr $nbipstack + 1`
                	h323done=$nbipstack
		fi
	fi
	if [  "$PORT_CONFIG" = "SIP" -o "$PORT_CONFIG" = "H323" ]
	then
		select_board_ip_add
		if [  "$PORT_CONFIG" = "SIP" -a "$sipdone" = "" ]
		then
			select_t38_SIP
			select_param_SIP
			nbipstack=`expr $nbipstack + 1`
			sipdone=$nbipstack
		fi

		if [ "$PORT_CONFIG" = "H323" -a  "$h323done" = "" ]
		then
			select_t38_H323
			select_param_H323
			nbipstack=`expr $nbipstack + 1`
                	h323done=$nbipstack
		fi
	fi
}

# ----------------------------------------------------------------------
# write the module entry in the callctrl configuration file
#
write_callctrl_module ()
{
	{
        	module=$1
		write_callctrl_module_$PORT_CONFIG $module
	} >> $CALLCT_CFG
}

# ------------------------------------------------------------------------
# flash the board to latest firmware
#
flash_board ()
{
	echo "Flashing board firmware..."
	./firm $BT_DEBUG -m $MODULE -t 2 fw/flashupd.bin
}

# ------------------------------------------------------------------------
# load firmware
#
load_firmware ()
{
	echo "Loading firmware..."
	./Bostsrv -k
	i="1"
	while [ $i -lt 10 -a -a /var/run/Bostsrv.pid ]
	do
		i=`expr $i + 1`
		sleep 2
	done
	./Bostsrv -c $BOSTON_DIR/btcall.cfg  -f $BOSTON_DIR/fw -D
	sleep 5
}

# ------------------------------------------------------------------------
# process a module
#
process_module ()
{
        module=$1
	# --------------------------------------------------------
	# write config info for all modules
	#
	write_callctrl_module $module

	# ----------------------------------------------------------------
	# bump our count of installed modules
	#
	MODULE_COUNT=`expr $MODULE_COUNT + 1`

	return 0
}

# ------------------------------------------------------------------------
# remove the boston devices & driver
#
remove_boston ()
{
	# ----------------------------------------------------------------
	# remove the driver
	#
	(
		cd install
		if in_windows
		then
			./install  -r
		else
			./dinstall -r
		fi
	)
}

# ------------------------------------------------------------------------
# set a module number as used
#
set_module_used ()
{
	MODULES_USED="$MODULES_USED$1"
}

# ------------------------------------------------------------------------
# check if a module number has used already
#
is_module_used ()
{
	echo "$MODULES_USED" | grep $1 >$DEV_NULL
}

# ------------------------------------------------------------------------
# get next available module number
#
next_module ()
{
	for i in 2 3 4 5 6 7 8 9 A B C D E F
	do
		is_module_used $i || break
	done

	echo $i
}

# ========================================================================
# main ()
#
PGM=`basename $0`
DEV_NULL=`get_dev_null`

# ------------------------------------------------------------------------
# check for help/usage requests
#
case "$1" in
	'-?' | '--?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | '--??' | -helpall | --helpall)
		helpall
		exit 0
		;;
esac

# ------------------------------------------------------------------------
# setup save-response file
#
if in_windows
then
	RESPONSE_DIR="`get_windrive`/Program Files/Common Files/Boston"
	RESPONSE_FILE="install.log"
else
	RESPONSE_DIR="/var/spool/pkg"
	RESPONSE_FILE="boston-install.log"
fi

SAVE_RESPONSE_FILE="$RESPONSE_DIR/$RESPONSE_FILE"

if [ -d "$RESPONSE_DIR" ]
then
	[ -w "$RESPONSE_DIR" ] || SAVE_RESPONSE_FILE=""
else
	mkdir -p "$RESPONSE_DIR" 2>$DEV_NULL || SAVE_RESPONSE_FILE=""
fi

if [ "$SAVE_RESPONSE_FILE" != "" ]
then
	touch "$SAVE_RESPONSE_FILE" 2>$DEV_NULL || SAVE_RESPONSE_FILE=""
	[ "$SAVE_RESPONSE_FILE" != "" ] && chmod 664 "$SAVE_RESPONSE_FILE"
fi

# ------------------------------------------------------------------------
# process options
#
ARGS=`getopt "rcfixdB:C:D" $*` || { usage >&2; exit 1; }
set -- $ARGS

COUNTRY_CODE=
DO_CONFIG=yes
DO_FIRMLOAD=yes
DO_INSTALL=yes
DO_REMOVE=no
NEED_ADMIN=yes
BT_DEBUG=
DEBUG=no
BOSTON_DIR=

while [ "$1" != "" ]
do
	case "$1" in

	-B)
		BOSTON_DIR=$2
		shift 2
		;;

	-C)
		COUNTRY_CODE=$2
		shift 2
		;;

	-D)
		DEBUG=yes
		shift
		;;

	-d)
		BT_DEBUG=-D
		shift
		;;

	-r)
		LOAD_RESPONSE_FILE="$SAVE_RESPONSE_FILE"
		shift
		;;

	-f)
		DO_FIRMLOAD=no
		DO_INSTALL=no
		NEED_ADMIN=no
		shift
		;;

	-i)
		DO_INSTALL=no
		shift
		;;

	-c)
		DO_CONFIG=no
		shift
		;;

	-x)
		DO_REMOVE=yes
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

[ "$DEBUG" = "yes" ] && set -x

# ------------------------------------------------------------------------
# Check if we have admin privilege if needed.
#
[ "$NEED_ADMIN" = "yes" ] && check_root

# ------------------------------------------------------------------------
# setup Brooktrout directory & config-file names
#
if [ "$BOSTON_DIR" = "" ]
then
	if in_windows
	then
		BOSTON_DIR=`get_windir`/Boston
	else
		BOSTON_DIR=/usr/sys/boston
	fi
fi

BTCALL_CFG=btcall.cfg
CALLCT_CFG=callctrl.cfg

cd $BOSTON_DIR

# ------------------------------------------------------------------------
# Check if removal wanted
#
if [ "$DO_REMOVE" = "yes" ]
then
	remove_boston
	exit $?
fi

# ------------------------------------------------------------------------
# install the driver if needed
#
if [ "$DO_INSTALL" = "yes" ]
then
	(
		echo ""
		echo "Installing the Boston driver..."

		cd install
		install_driver || abort_install
	) || exit 1
fi

# ------------------------------------------------------------------------
# Check if all config files exist, and, if so, query if the user
# wants to make any changes.
#
if [ "$DO_CONFIG" = "yes" -a \
     -f $BTCALL_CFG       -a \
     -f $CALLCT_CFG ]
then
	echo ""
	getyesorno "use_config" \
		"Do you want to keep your existing configuration?" &&
	DO_CONFIG=no
fi

# ------------------------------------------------------------------------
# do configuration if requested
#
if [ "$DO_CONFIG" = "yes" ]
then
	#----------------------------------------------------------------
	# Do a backup
	#
	BACKUP_STR=`date +%Y%m%d-%H%M%S`
	[ -f $BTCALL_CFG ] && cp $BTCALL_CFG $BTCALL_CFG.$BACKUP_STR
	[ -f $CALLCT_CFG ] && cp $CALLCT_CFG $CALLCT_CFG.$BACKUP_STR
	# ----------------------------------------------------------------
	# get country code
	#
	select_country_code

	# ----------------------------------------------------------------
	# We can now write the (generic) btcall file &
	# initialize the (module-specific) callctrl file.
	#
	write_btcall_file
	init_callctrl_file

	# ----------------------------------------------------------------
	# now install all boards
	#
	MODULE_COUNT=0
	BOARD_TYPE=1
	sipdone=""
	h323done=""
	nbipstack=0

	while true
	do
		# --------------------------------------------------------
		# get board type (this will also set the port type)
		#
		select_board_type $BOARD_TYPE || break

		# --------------------------------------------------------
		# get module number
		#
		select_module `next_module` || break

		#---------------------------------------------------------
		# For FoIP get board ip address
		#
		select_ip_info || break

		# --------------------------------------------------------
		# process this module
		#
		process_module $MODULE

		# --------------------------------------------------------
		# ask if any more boards to do
		#
		echo ""
		getyesorno -n another_board_$MODULE_COUNT \
			"Do you want to configure another board?" || break
	done

	# ----------------------------------------------------------------
	# flash the board & load the firmware
	#
	if [ "$DO_FIRMLOAD" = "yes" ]
	then
		echo ""
		load_firmware || return 1
	fi

	echo ""
	[ $MODULE_COUNT -ne 1 ] && PLURAL=s
	echo "Done: $MODULE_COUNT board$PLURAL installed."
fi

exit 0
