#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	Uninstall the boston driver
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get name of the NULL device
#
get_dev_null ()
{
	if in_windows
	then
		echo "nul:"
	else
		echo "/dev/null"
	fi
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# output a prompt & get a response from the user
#
get_user_input ()
{
	id="$1"
	dflt="$2"
	msg="$3"

	ok_id="$id-ok"
	[ "$id" = "" ] && ok_id=""

	CONSOLE=`get_dev_tty`

	while true
	do
		if [ x"$dflt" = x"!" ]
		then
			echonl "Enter $msg: "
		else
			echonl "Enter $msg [$dflt]: "
		fi >$CONSOLE

		response=`read_user_input "$id"`
		[ "$response" = "" -a x"$dflt" != x"!" ] && response="$dflt"

		echo "$msg = \"$response\"" >$CONSOLE
		is_ok "$ok_id" >$CONSOLE && { echo "$response"; break; }

		dflt="$response"
	done
}

all_libs_file()
{
	mylibs=$1
	for i in $mylibs
	do
		[ -a /usr/lib/$i -o -a /lib/$i ] || return 2
	done
	return 0
}


#-----------------------------------------------
# Brooktrout shared libraries
#
setup_rm_boston()
{
	rm -f  "/usr/lib/bostlib_mt.so"  2> /dev/null
	rm -f  "/usr/lib/osilib_mt.so"   2> /dev/null
	rm -f  "/usr/lib/brktsip_mt.so"  2> /dev/null
	rm -f  "/usr/lib/brkth323_mt.so" 2> /dev/null
	rm -f  "/usr/lib/liblicUtilSR140.so" 2> /dev/null
}

# ------------------------------------------------------------------------
# copy all boston files to proper place
#
setup_cp_boston ()
{

	[ -x $VSIFAX/lbin/vsibost.sh ] && $VSIFAX/lbin/vsibost.sh stop

	[ -d $BOSTON_DIR ] || mkdir -p $BOSTON_DIR

	#
	# Remove previous lib links
	#
	rm -f /usr/lib/bostlib_mt.so   2> /dev/null
	rm -f /usr/lib/osilib_mt.so   2> /dev/null
	rm -f /usr/lib/brktsip_mt.so   2> /dev/null
	rm -f /usr/lib/brkth323_mt.so   2> /dev/null
	rm -f /usr/lib/liblicUtilSR140.so   2> /dev/null
	rm -f $BOSTON_DIR/fw/bostvb.so   2> /dev/null

	#
	# Remove previous binary links
	#
	rm -f $BOSTON_DIR/Bostsrv   2> /dev/null
	rm -f $BOSTON_DIR/listnodeid   2> /dev/null
	rm -f $BOSTON_DIR/fw/flashupd.bin   2> /dev/null


	#
	# Copy Brooktrout tree to /usr/sys/boston
	#
	cp -r $VSIFAX/lib/boston/* $BOSTON_DIR


   	if grep "[AE]S release 3" /etc/redhat-release > /dev/null
   	then
                ln -s $BOSTON_DIR/lib/7/bostlib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/7/brkth323_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/7/brktsip_mt.so /usr/lib
                #ln -s $BOSTON_DIR/lib/7/osilib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/7/liblicUtilSR140.so /usr/lib

		ln -s $BOSTON_DIR/lib/7/bostvb.so $BOSTON_DIR/fw/bostvb.so

		ln -s $BOSTON_DIR/bin/7/Bostsrv $BOSTON_DIR/Bostsrv
		ln -s $BOSTON_DIR/bin/7/listnodeid $BOSTON_DIR/listnodeid
	fi
	if grep "[AE]S release 4" /etc/redhat-release > /dev/null
   	then
                ln -s $BOSTON_DIR/lib/3.4/bostlib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/3.4/brkth323_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/3.4/brktsip_mt.so /usr/lib
                #ln -s $BOSTON_DIR/lib/3.4/osilib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/3.4/liblicUtilSR140.so /usr/lib

		ln -s $BOSTON_DIR/lib/3.4/bostvb.so $BOSTON_DIR/fw/bostvb.so

		ln -s $BOSTON_DIR/bin/3.4/Bostsrv $BOSTON_DIR/Bostsrv
		ln -s $BOSTON_DIR/bin/3.4/listnodeid $BOSTON_DIR/listnodeid
	fi
	if grep "Server release 5" /etc/redhat-release > /dev/null
   	then
                ln -s $BOSTON_DIR/lib/4.1/bostlib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/4.1/brkth323_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/4.1/brktsip_mt.so /usr/lib
                #ln -s $BOSTON_DIR/lib/4.1/osilib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/4.1/liblicUtilSR140.so /usr/lib

		ln -s $BOSTON_DIR/lib/4.1/bostvb.so $BOSTON_DIR/fw/bostvb.so

		ln -s $BOSTON_DIR/bin/4.1/Bostsrv $BOSTON_DIR/Bostsrv
		ln -s $BOSTON_DIR/bin/4.1/listnodeid $BOSTON_DIR/listnodeid
		#
		# Check if SELINUX is enabled
		#
		selinuxenabled 2>/dev/null
		if [ $? -eq 0 ]
		then
			ENFORCE=`getenforce 2> /dev/null | tr [:upper:] [:lower:] `
			if [ "$ENFORCE" = "enforcing" ]
			then
				TEXTREL_SHLIB_T=`ls -Z $BOSTON_DIR/lib/4.1/bostlib_mt.so $BOSTON_DIR/lib/4.1/brkth323_mt.so $BOSTON_DIR/lib/4.1/brktsip_mt.so $BOSTON_DIR/lib/4.1/bostvb.so | grep textrel_shlib_t | wc -l`
				if [ $TEXTREL_SHLIB_T -ne 4 ]
				then
					echo ""
					echo "SELinux is enabled and set to 'Enforcing' on this server. It will"
					echo "prevent bostlib_mt.so and other Dialogic-supplied shared libraries"
					echo "from loading because they require text relocation, which is very "
					echo "common for third-party shared libraries."
					echo ""
					echo "The setup has to run these commands to allow loading the Dialogic daemon:"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostlib_mt.so"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brkth323_mt.so"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brktsip_mt.so"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostvb.so"
					echo""
					if getyesorno "allow-setup-enable-selinux" "Can the setup run the above commands?"
					then
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostlib_mt.so
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brkth323_mt.so
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brktsip_mt.so
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostvb.so
					else
				TEXTREL_SHLIB_T=`ls -Z $BOSTON_DIR/lib/4.1/bostlib_mt.so $BOSTON_DIR/lib/4.1/brkth323_mt.so $BOSTON_DIR/lib/4.1/brktsip_mt.so $BOSTON_DIR/lib/4.1/bostvb.so | grep textrel_shlib_t | wc -l`
				if [ $TEXTREL_SHLIB_T -ne 4 ]
				then
					echoerr "The setup will abort because the listed librairies have not text relocation attributes."
					return 0
				fi
					fi
				fi
			fi
		fi

	fi

	in_windows ||
	{

	# Needed by SR140
		ldconfig
		chgrp -R bin $BOSTON_DIR
		chown -R bin $BOSTON_DIR
	}
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -f     force uninstall (no prompt)"
	echo "  -o     only remove non-installed files & directories"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D     debug uninstall"
}

# ------------------------------------------------------------------------
# main()
#
PRDNAME=`prdname`

[ -f /etc/$PRDNAME.sh ] && . /etc/$PRDNAME.sh

if [ x"$VSIFAX" = x"" ]
then
	echo "$PRDNAME is not installed. Nothing to remove."
	exit 0
fi


# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

DEBUG=
FORCE=no
ONLY=no

case "$1" in
	'-?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | -helpall | --helpall)
		helpall
		exit 0
		;;

	'-V' | -version | --version)
		echo "$PGM: `vfxadmin version`"
		exit 0
		;;
esac

ARGS=`getopt "Dugfo" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-x
		shift
		;;

	-u)
		shift
		;;

	-g)
		shift
		;;

	-f)
		FORCE=yes
		shift
		;;

	-o)
		ONLY=yes
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

[ "$DEBUG" != "" ] && set $DEBUG

# ------------------------------------------------------------------------
# check if we should check if OK
#
if [ "$FORCE" = "no" ]
then
	echo ""
	echo "This will completely uninstall the `pkgname` Dialogic Brooktrout drivers from $VSIFAX."
	getyesorno -n "" "Do you want to proceed?" || exit 0
fi

# ------------------------------------------------------------------------
# check that user is root
#
check_root

# ------------------------------------------------------------------------
# now undo all configurations
#
vsi-boston-config.sh -x

# ------------------------------------------------------------------------
# now remove all installed files & directories
#
if [ "$ONLY" != "yes" ]
then
	rm -rf "$VSIFAX/lib/boston"
	rm -f  "$VSIFAX/lbin/vsi-boston-config.sh"
	rm -f  "$VSIFAX/lbin/vsi-boston-setup.sh"
	rm -f  "$VSIFAX/lbin/vsi-boston-uninstall.sh"

	setup_rm_boston
fi

exit 0
