#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	configuration for the Brooktrout package
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# add a directory to the path
#
# Note that the directory is ALWAYS added to the end.
#
addpath ()
{
	dir="$1"

	sep=":"
	arg=""
	if in_windows
	then
		sep=";"
		arg="-i"
	fi

	foo=`echo "$PATH" | egrep $arg "^$dir$sep|$sep$dir$sep|$sep$dir$"`
	if [ $? -eq 0 ]
	then
		echo "$PATH"
	else
		echo "$PATH$sep$dir"
	fi

	unset dir sep arg foo
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# output a prompt & get a response from the user
#
get_user_input ()
{
	id="$1"
	dflt="$2"
	msg="$3"

	ok_id="$id-ok"
	[ "$id" = "" ] && ok_id=""

	CONSOLE=`get_dev_tty`

	while true
	do
		if [ x"$dflt" = x"!" ]
		then
			echonl "Enter $msg: "
		else
			echonl "Enter $msg [$dflt]: "
		fi >$CONSOLE

		response=`read_user_input "$id"`
		[ "$response" = "" -a x"$dflt" != x"!" ] && response="$dflt"

		echo "$msg = \"$response\"" >$CONSOLE
		is_ok "$ok_id" >$CONSOLE && { echo "$response"; break; }

		dflt="$response"
	done
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# get Windows directory
#
get_windir ()
{
	{
		if [ "$windir" != "" ]
		then
			echo "$windir"

		elif [ "$WINDIR" != "" ]
		then
			echo "$WINDIR"

		elif [ "$SystemRoot" != "" ]
		then
			echo "$SystemRoot"

		elif [ "$SYSTEMROOT" != "" ]
		then
			echo "$SYSTEMROOT"

		elif [ -d "c:/windows" ]
		then
			echo "c:/windows"

		elif [ -d "c:/winnt" ]
		then
			echo "c:/winnt"

		else
			echo "c:/windows"
		fi
	} | sed -e 's,\\\\*,/,g'
}

# ------------------------------------------------------------------------
# get Windows drive
#
get_windrive()
{
	get_windir | sed -e 's,[/\\].*,,'
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get response dir/file info
#
response_info ()
{
	if in_windows
	then
		RESPONSE_DIR="`get_windrive`/Program Files/Common Files/VSI-FAX Shared"
	else
		RESPONSE_DIR="/var/spool/pkg"
	fi
	RESPONSE_FILE="`prdname`-install.log"

	echo "RESPONSE_DIR=\"$RESPONSE_DIR\""
	echo "SAVE_RESPONSE_FILE=\"$RESPONSE_DIR/$RESPONSE_FILE\""
}

# ------------------------------------------------------------------------
# get name of the NULL device
#
get_dev_null ()
{
	if in_windows
	then
		echo "nul:"
	else
		echo "/dev/null"
	fi
}

# ------------------------------------------------------------------------
# check that user is root
#
is_root ()
{
	in_windows && return 0

	unset uid
	unset euid
	eval `id | awk '
 BEGIN { uids="" }
 {
   for (i=1 ; i <= NF; i++ )
   {
      if ( $i ~ /^(uid|euid)=/ ) { split($i,ar,"("); uids= uids " " ar[1]}
    }
 }
 END { print uids }'`

	[ "$euid" = "" ] && euid=$uid

	[ $uid -eq 0 -o $euid -eq 0 ]
}

# ------------------------------------------------------------------------
# get name of root
#
rootname ()
{
	if in_windows
	then
		echo "Administrator"
	else
		grep '^[^:]*:[^:]*:0:' /etc/passwd | head -1 | cut -f1 -d:
	fi
}

# ------------------------------------------------------------------------
# check that user is root & abort if not
#
check_root ()
{
	is_root ||
	{
		root=`rootname`
		echoerr ""
		echoerr "Error: you are not running as \"$root\"."
		echoerr "You must be running as \"$root\" to proceed."
		abort_install
	}
}

# ------------------------------------------------------------------------
# Return the Brooktrout SDK version ifor TR1034 located in a file
#
bt1034sdkver ()
{
 file_version=$1
 text=$2

 if [ "$text" = "-t" ]
 then
   sed  -e '/^#/d' -n -e's/VERSION\=\(.*\)/\1/p'  $file_version
 else
   eval `sed  -e '/^#/d' -n -e \
  's/VERSION\=\(.*\)\.\(.*\)\.\(.*\)/expr \1 \\\* 10000 + \2 \\\* 100 + \3/p' \
  $file_version`
 fi
}

all_libs_file()
{
	mylibs=$1
	for i in $mylibs
	do
		[ -a /usr/lib/$i -o -a /lib/$i ] || return 2
	done
	return 0
}


#-----------------------------------------------
# Brooktrout shared libraries
#
setup_rm_boston()
{
	rm -f  "/usr/lib/bostlib_mt.so"  2> /dev/null
	rm -f  "/usr/lib/osilib_mt.so"   2> /dev/null
	rm -f  "/usr/lib/brktsip_mt.so"  2> /dev/null
	rm -f  "/usr/lib/brkth323_mt.so" 2> /dev/null
	rm -f  "/usr/lib/liblicUtilSR140.so" 2> /dev/null
}

# ------------------------------------------------------------------------
# copy all boston files to proper place
#
setup_cp_boston ()
{

	[ -x $VSIFAX/lbin/vsibost.sh ] && $VSIFAX/lbin/vsibost.sh stop

	[ -d $BOSTON_DIR ] || mkdir -p $BOSTON_DIR

	#
	# Remove previous lib links
	#
	rm -f /usr/lib/bostlib_mt.so   2> /dev/null
	rm -f /usr/lib/osilib_mt.so   2> /dev/null
	rm -f /usr/lib/brktsip_mt.so   2> /dev/null
	rm -f /usr/lib/brkth323_mt.so   2> /dev/null
	rm -f /usr/lib/liblicUtilSR140.so   2> /dev/null
	rm -f $BOSTON_DIR/fw/bostvb.so   2> /dev/null

	#
	# Remove previous binary links
	#
	rm -f $BOSTON_DIR/Bostsrv   2> /dev/null
	rm -f $BOSTON_DIR/listnodeid   2> /dev/null
	rm -f $BOSTON_DIR/fw/flashupd.bin   2> /dev/null


	#
	# Copy Brooktrout tree to /usr/sys/boston
	#
	cp -r $VSIFAX/lib/boston/* $BOSTON_DIR


   	if grep "[AE]S release 3" /etc/redhat-release > /dev/null
   	then
                ln -s $BOSTON_DIR/lib/7/bostlib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/7/brkth323_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/7/brktsip_mt.so /usr/lib
                #ln -s $BOSTON_DIR/lib/7/osilib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/7/liblicUtilSR140.so /usr/lib

		ln -s $BOSTON_DIR/lib/7/bostvb.so $BOSTON_DIR/fw/bostvb.so

		ln -s $BOSTON_DIR/bin/7/Bostsrv $BOSTON_DIR/Bostsrv
		ln -s $BOSTON_DIR/bin/7/listnodeid $BOSTON_DIR/listnodeid
	fi
	if grep "[AE]S release 4" /etc/redhat-release > /dev/null
   	then
                ln -s $BOSTON_DIR/lib/3.4/bostlib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/3.4/brkth323_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/3.4/brktsip_mt.so /usr/lib
                #ln -s $BOSTON_DIR/lib/3.4/osilib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/3.4/liblicUtilSR140.so /usr/lib

		ln -s $BOSTON_DIR/lib/3.4/bostvb.so $BOSTON_DIR/fw/bostvb.so

		ln -s $BOSTON_DIR/bin/3.4/Bostsrv $BOSTON_DIR/Bostsrv
		ln -s $BOSTON_DIR/bin/3.4/listnodeid $BOSTON_DIR/listnodeid
	fi
	if grep "Server release 5" /etc/redhat-release > /dev/null
   	then
                ln -s $BOSTON_DIR/lib/4.1/bostlib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/4.1/brkth323_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/4.1/brktsip_mt.so /usr/lib
                #ln -s $BOSTON_DIR/lib/4.1/osilib_mt.so /usr/lib
                ln -s $BOSTON_DIR/lib/4.1/liblicUtilSR140.so /usr/lib

		ln -s $BOSTON_DIR/lib/4.1/bostvb.so $BOSTON_DIR/fw/bostvb.so

		ln -s $BOSTON_DIR/bin/4.1/Bostsrv $BOSTON_DIR/Bostsrv
		ln -s $BOSTON_DIR/bin/4.1/listnodeid $BOSTON_DIR/listnodeid
		#
		# Check if SELINUX is enabled
		#
		selinuxenabled 2>/dev/null
		if [ $? -eq 0 ]
		then
			ENFORCE=`getenforce 2> /dev/null | tr [:upper:] [:lower:] `
			if [ "$ENFORCE" = "enforcing" ]
			then
				TEXTREL_SHLIB_T=`ls -Z $BOSTON_DIR/lib/4.1/bostlib_mt.so $BOSTON_DIR/lib/4.1/brkth323_mt.so $BOSTON_DIR/lib/4.1/brktsip_mt.so $BOSTON_DIR/lib/4.1/bostvb.so | grep textrel_shlib_t | wc -l`
				if [ $TEXTREL_SHLIB_T -ne 4 ]
				then
					echo ""
					echo "SELinux is enabled and set to 'Enforcing' on this server. It will"
					echo "prevent bostlib_mt.so and other Dialogic-supplied shared libraries"
					echo "from loading because they require text relocation, which is very "
					echo "common for third-party shared libraries."
					echo ""
					echo "The setup has to run these commands to allow loading the Dialogic daemon:"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostlib_mt.so"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brkth323_mt.so"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brktsip_mt.so"
					echo " chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostvb.so"
					echo""
					if getyesorno "allow-setup-enable-selinux" "Can the setup run the above commands?"
					then
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostlib_mt.so
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brkth323_mt.so
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/brktsip_mt.so
						chcon -t textrel_shlib_t $BOSTON_DIR/lib/4.1/bostvb.so
					else
				TEXTREL_SHLIB_T=`ls -Z $BOSTON_DIR/lib/4.1/bostlib_mt.so $BOSTON_DIR/lib/4.1/brkth323_mt.so $BOSTON_DIR/lib/4.1/brktsip_mt.so $BOSTON_DIR/lib/4.1/bostvb.so | grep textrel_shlib_t | wc -l`
				if [ $TEXTREL_SHLIB_T -ne 4 ]
				then
					echoerr "The setup will abort because the listed librairies have not text relocation attributes."
					return 0
				fi
					fi
				fi
			fi
		fi

	fi

	in_windows ||
	{

	# Needed by SR140
		ldconfig
		chgrp -R bin $BOSTON_DIR
		chown -R bin $BOSTON_DIR
	}
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -r     read response file"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D     debug install"
	echo "  -x     remove any configurations"
}

# ------------------------------------------------------------------------
# do all boston configuration
#
setup_boston ()
{
	COUNTRY_CODE=`vfxadmin config -slq DEVICE     CountryCode`
	[ "$COUNTRY_CODE"  = "" ] &&
	COUNTRY_CODE=`vfxadmin config -slq DIALSTRING CountryCode`
	[ "$COUNTRY_CODE" != "" ] && COUNTRY_CODE="-C $COUNTRY_CODE"

	(
		cd $BOSTON_DIR
		./btinstall.sh $RESPONSE_OPT $DEBUG $COUNTRY_CODE || exit 1
	) || exit 1
}

# ------------------------------------------------------------------------
# setup brooktrout configuration entries
#
setup_config ()
{
	# ----------------------------------------------------------------
	# setup device ini entries
	#
	vfxadmin config -sn BROOKTROUT config-dir		$BOSTON_DIR
	vfxadmin config -sn BROOKTROUT config-file		$BTCALL_CFG
	vfxadmin config -sn BROOKTROUT dsp_ldr			$BOSTON_DIR/fw
	vfxadmin config -sn BROOKTROUT add-dialtone-wait	on
	vfxadmin config -sn BROOKTROUT reset-wait		5
	vfxadmin config -sn BROOKTROUT reset-attempts		5
	vfxadmin config -sn BROOKTROUT add-dialtone-wait	on
	vfxadmin config -sn BROOKTROUT debug-reset		off
	vfxadmin config -sn BROOKTROUT btapi-debug		off
	vfxadmin config -sn BROOKTROUT btdebug-enable		on
	vfxadmin config -sn BROOKTROUT btdebug-inhibit		off
	vfxadmin config -sn BROOKTROUT btdebug-min		on
	vfxadmin config -sn BROOKTROUT btdebug-max		off
	vfxadmin config -sn BROOKTROUT btdedug-PRINT_CMD	off
	vfxadmin config -sn BROOKTROUT btdebug-PRINT_INTR	off
	vfxadmin config -sn BROOKTROUT btdebug-MON		off
	vfxadmin config -sn BROOKTROUT btdebug-ERR		off
	vfxadmin config -sn BROOKTROUT btdebug-DEBUG		off

	# ----------------------------------------------------------------
	# remove any existing module entries
	# 65 (0x41) for virtual board
	#
	for module in 2 3 4 5 6 7 8 9 A B C D E F 10 11 12 13 14 15 41 65
	do
		vfxadmin config -sx BROOKTROUT module$module
	done

	# ----------------------------------------------------------------
	# add bmldrd entry to daemons.lst file, removing any old entries
	#
	ex "$VSIFAX/lib/daemons.lst" <<-EOF >$DEV_NULL
	:g/bmldrd/d
	:wq
	EOF
	echo "bmldrd;-1;false;true" >>$VSIFAX/lib/daemons.lst
}

# ------------------------------------------------------------------------
# setup brooktrout devices for a given module
#
setup_module ()
{
	module=$1
	channels=$2
	unit=$3
	MS=FALSE
	case "$module" in
		10|A|a)	Hexamodule=A ;;
		11|B|b)	Hexamodule=B ;;
		12|C|c)	Hexamodule=C ;;
		13|D|d)	Hexamodule=D ;;
		14|E|e)	Hexamodule=E ;;
		15|F|f)	Hexamodule=F ;;
		65)     Hexamodule=41;;
		*)	Hexamodule=$module
	esac

	# ----------------------------------------------------------------
	# check if RBS or PRI
	#
	port_config=`sed -n -e "/module.$Hexamodule\\/port/, /\\[/ { /port_config/ {p} }"  $BOSTON_DIR/$CALLCT_CFG`
	eval $port_config
	case "$port_config" in
		T1_ROBBED_BIT)	type=t1_rbs
				;;
		T1_ISDN)	type=t1_isdn
				;;
		E1_ISDN)	type=e1_isdn
				;;
		analog)		type=analog
				;;
		BRI)		type=bri
				;;
		*)		type=ip
				;;
	esac
	[ "$type" = "t1_isdn" -a $channels -eq 24 ] && channels=23
	MS=TRUE

	# ----------------------------------------------------------------
	# write module info into ini file
	#
	vfxadmin config -sa BROOKTROUT module$Hexamodule "$type,$channels"


	# ----------------------------------------------------------------
	# create any devices needed
	#
	if [ $unit -lt 10 ]
	then
		base=ch0$unit
	else
		base=ch$unit
	fi
	if [ $module -lt 15 ]
	then
		product_name=`$BOSTON_DIR/feature -m $module -q | grep "Product = " | head -1 | sed -e 's/.*=//' | sed -e 's/.* //'`
		if [ "$product_name"  = "" ]
		then
			product_name="TR1034"
		fi
	else
		product_name="SR140"
	fi
	#If we have to use master/slave architecture
	[ "$USE_MS" = "FALSE" ] && MS=FALSE

	ms_opt="-M"
	i=0
	while [ $i -lt $channels ]
	do
		if [ $unit -lt 10 ]
		then
			name=ch0$unit
		else
			name=ch$unit
		fi

		phy_unit=`expr $i + 2`
		device_name="ch"$module"-"$phy_unit

		OPT=
                [ "$MS" = "TRUE" ] && OPT="$ms_opt"
		if vfxadmin device -ql $name
                then
		   if [ "$MS" = "TRUE" ]
                   then
		   	vfxadmin device -q -u $OPT $name
		   fi
		else
			vfxadmin device -q -a -i bm -v $device_name $OPT \
			   -n "$product_name mod $module ch $phy_unit" $name
		fi

		i=`   expr $i    + 1`
		unit=`expr $unit + 1`
		ms_opt="-Y $base"
	done
}

# ------------------------------------------------------------------------
# setup brooktrout devices
#
setup_devices ()
{
	sleep 40
	$BOSTON_DIR/modinfo -e |
	while read line
	do
		eval $line
		if [ $module -lt 66 ]
		then
			setup_module $module $channels $unit
		fi
	done
}

# ------------------------------------------------------------------------
# remove the boston devices & driver
#
remove_boston ()
{
	# ----------------------------------------------------------------
	# remove all Brooktrout devices
	#
	vfxadmin device -l -F pipe -f dst -i bm |
	while read dest
	do
		vfxadmin device -q -x -O $dest
	done

	# ----------------------------------------------------------------
	# remove all Brooktrout entries from ini file
	#
	vfxadmin config -sx BROOKTROUT

	# ----------------------------------------------------------------
	# remove bmldrd entry from daemons.lst file
	#
	ex "$VSIFAX/lib/daemons.lst" <<-EOF >$DEV_NULL
	:g/bmldrd/d
	:wq
	EOF

	# ----------------------------------------------------------------
	# remove the driver if present
	#
	if [ -d $BOSTON_DIR ]
	then
		(
			cd $BOSTON_DIR
			./btinstall.sh -x
		)
	fi

	# ----------------------------------------------------------------
	# now delete the entire boston directory
	#
	rm -rf $BOSTON_DIR
}

# ------------------------------------------------------------------------
# check if boston set to start at boot time
#
check_boot_boston()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# get name to look for
	#
	NAME="vsibost"

	# ----------------------------------------------------------------
	# Check for Linux
	#
	if [ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
	then
	      if [ -x /etc/rc.d/rc5.d/S98$NAME -a \
	       -x /etc/rc.d/rc3.d/S98$NAME -a \
	       -x /etc/rc.d/rc2.d/S98$NAME -a \
	       -x /etc/rc.d/rc6.d/K02$NAME -a \
	       -x /etc/rc.d/rc5.d/K02$NAME -a \
	       -x /etc/rc.d/rc3.d/K02$NAME -a \
	       -x /etc/rc.d/rc2.d/K02$NAME -a \
	       -x /etc/rc.d/rc1.d/K02$NAME -a \
	       -x /etc/rc.d/rc0.d/K02$NAME ]
	      then
		 return 0
              else
		 return 1
              fi

	fi
	return 0
}

# ------------------------------------------------------------------------
# set boston auto-start at boot time
#
set_boot_boston()
{
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# get options
	#
	NAME="vsibost"
	OPTION=$1

	SCRIPT="$VSIFAX/lbin/$NAME.sh"

	# ----------------------------------------------------------------
	# Check for Linux
	#
	if [ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
	then
	     rm -f /etc/rc.d/rc5.d/S98$NAME
	     rm -f /etc/rc.d/rc3.d/S98$NAME
	     rm -f /etc/rc.d/rc2.d/S98$NAME
	     rm -f /etc/rc.d/rc6.d/K02$NAME
	     rm -f /etc/rc.d/rc5.d/K02$NAME
	     rm -f /etc/rc.d/rc3.d/K02$NAME
	     rm -f /etc/rc.d/rc2.d/K02$NAME
	     rm -f /etc/rc.d/rc1.d/K02$NAME
	     rm -f /etc/rc.d/rc0.d/K02$NAME	# TEMP hack

	     if [ "$OPTION" = "on" ]
	     then
	      [ -d /etc/rc.d/rc5.d ] && ln -s $SCRIPT /etc/rc.d/rc5.d/S98$NAME
	      [ -d /etc/rc.d/rc3.d ] && ln -s $SCRIPT /etc/rc.d/rc3.d/S98$NAME
	      [ -d /etc/rc.d/rc2.d ] && ln -s $SCRIPT /etc/rc.d/rc2.d/S98$NAME
	      [ -d /etc/rc.d/rc6.d ] && ln -s $SCRIPT /etc/rc.d/rc6.d/K02$NAME
	      [ -d /etc/rc.d/rc5.d ] && ln -s $SCRIPT /etc/rc.d/rc5.d/K02$NAME
	      [ -d /etc/rc.d/rc3.d ] && ln -s $SCRIPT /etc/rc.d/rc3.d/K02$NAME
	      [ -d /etc/rc.d/rc2.d ] && ln -s $SCRIPT /etc/rc.d/rc2.d/K02$NAME
	      [ -d /etc/rc.d/rc1.d ] && ln -s $SCRIPT /etc/rc.d/rc1.d/K02$NAME
	      [ -d /etc/rc.d/rc0.d ] && ln -s $SCRIPT /etc/rc.d/rc0.d/K02$NAME
	     fi

	     return 0
	fi
	return 0
}


# ------------------------------------------------------------------------
# setup auto-start at boot
#
configure_autostart_boston()
{
	if check_boot_boston
	then
		:
	else
		echo ""
		echo "Setting the $PKGNAME Dialogic Brooktroot TR1034-SR140 drivers to start" \
			"automatically at system boot time."
		is_ok "auto-start-boston-ok" && set_boot_boston on
	fi
}

# ------------------------------------------------------------------------
# setup auto-start at boot
#
configure_autostart()
{
	if [ "$1" = "on" ]
	then
		if in_windows
		then
			:
		else
			configure_autostart_boston
		fi
	else
		if in_windows
		then
			:
		else
			set_boot_boston off
		fi
	fi
}

# ------------------------------------------------------------------------
# main()
#
PATH=`addpath "$VSIFAX/bin"`
PATH=`addpath "$VSIFAX/lbin"`

PKGNAME=`pkgname`
DEV_NULL=`get_dev_null`

eval `response_info`

# ------------------------------------------------------------------------
# setup Brooktrout directory & config-file names
#
if in_windows
then
	BOSTON_DIR=`get_windir`/Boston
else
	BOSTON_DIR=/usr/sys/boston
fi

BTCALL_CFG=btcall.cfg
CALLCT_CFG=callctrl.cfg


# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

DEBUG=
DELETE=no

case "$1" in
	'-?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | -helpall | --helpall)
		helpall
		exit 0
		;;

	'-V' | -version | --version)
		echo "$PGM: `vfxadmin version`"
		exit 0
		;;
esac

RESPONSE_OPT=

ARGS=`getopt "Drx" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-D
		shift
		;;

	-r)
		LOAD_RESPONSE_FILE="$SAVE_RESPONSE_FILE"
		RESPONSE_OPT=-r
		shift
		;;

	-x)
		DELETE=yes
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

export SAVE_RESPONSE_FILE
export LOAD_RESPONSE_FILE

if [ "$SAVE_RESPONSE_FILE" != "" ]
then
	[ ! -d "$RESPONSE_DIR" ] && mkdir -p "$RESPONSE_DIR"
	touch "$SAVE_RESPONSE_FILE"
	chmod 664 "$SAVE_RESPONSE_FILE"
fi

[ "$DEBUG" != "" ] && set -x

# ------------------------------------------------------------------------
# now do/undo all configurations
#
SDK_VER=0
[ -f $VSIFAX/lib/boston/sdk_version.txt ] &&  \
  SDK_VER=`bt1034sdkver $VSIFAX/lib/boston/sdk_version.txt`

[ -f $BOSTON_DIR/sdk_version.txt ] &&  \
  SDK_VER=`bt1034sdkver $BOSTON_DIR/sdk_version.txt`

# ------------------------------------------------------------------------
# check that user is root
#
[ $SDK_VER -ge 40100 ] && check_root

#------------------------------------------------------------------------
# Make sure vfxsched is stopped - case this shell called by the user
#-----------------------------------------------------------------------
srv_running=no
if [ -x "$VSIFAX/bin/vfxsched" -o -x "$VSIFAX/bin/vfxsched.exe" ]
then
   vfxsched -s check && srv_running=yes
   if [ "$srv_running" = "yes" ]
   then
       echo ""
       echo "The $PKGNAME servers must be shutdown."
       echo "Shutting down the $PKGNAME servers."

       is_ok "stopserver-ok"
       if [ $? -eq 0 ]
       then
          vfxsched -s stop
       else
           echo "Please run this installation again after"
           echo "shutting down the $PKGNAME Servers."
           abort_install
        fi
    fi
fi


if [ "$DELETE" = "no" ]
then
	#
	#update path in vsibost.sh
	#
	sed_var=`echo "$BOSTON_DIR" | sed -e 's/\//\\\\\//g'`
	sed -e "s/\(BRKTD_LICENSE_FILE=\)\(.*\)$/\1$sed_var/" -e "s/\(BOSTON_DIR=\)\(.*\)$/\1$sed_var/"  $VSIFAX/lbin/vsibost.sh > $VSIFAX/lbin/vsibost.tmp

	cp -f $VSIFAX/lbin/vsibost.tmp $VSIFAX/lbin/vsibost.sh
	rm -f $VSIFAX/lbin/vsibost.tmp

	$VSIFAX/lbin/vsibost.sh stop
	setup_boston || abort_install
	$VSIFAX/lbin/vsibost.sh start
	setup_config
	setup_devices
        configure_autostart on
else
	$VSIFAX/lbin/vsibost.sh stop
        configure_autostart off
	setup_rm_boston
	remove_boston
fi
[ "$srv_running" = "yes" ] && vfxsched -s start
