// +++FHDR+++
//
//
//   Class Name : Registry
//   Super Class: n/a
//
//   Description:
//	The Registry class provides a simple mechanism for managing the
//	bindings of names to objects. Methods are provided for binding,
//	unbinding,  looking up and listing name to object bindings.
//
//
//   Notes      : none
//
//
//   Method - Description
//   ------------------------------------------------------------------------
//
//
// ---FHDR---


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "Registry.h"


// +++PHDR+++
//
//  Method     : Registry::Registry()
//
//  Visibility : public
//
//  Description: Default constructor for the Regsitry class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

Registry::Registry ()
{
	_capacity = 0;
	_capacityIncrement = 1;
	_num_elements = 0;
	_names = NULL;
	_objects = NULL;
	ensureCapacity(10);
}

// +++PHDR+++
//
//  Method     : Registry::Registry()
//
//  Visibility : public
//
//  Description: Alternate constructor for Registry class. This constructor
//               creates a Regsitry object with the specified capacity.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  capacity        int                     desired capacity
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

Registry::Registry (int capacity)
{
	_capacity = 0;
	_capacityIncrement = 1;
	_num_elements = 0;
	_names = NULL;
	_objects = NULL;
	ensureCapacity(capacity);
}


// +++PHDR+++
//
//  Method     : Registry::~Registry()
//
//  Visibility : public
//
//  Description: Default destructor for the Registry class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

Registry::~Registry ()
{
	delete _objects;

	for (int i = 0; i < _num_elements; i++) {
		delete _names[i];
	}
	delete _names;
}


// +++PHDR+++
//
//  Method     : Registry::capacity()
//
//  Visibility : public
//
//  Description: Returns the current capacity.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   The current capacity.
//
//
//  Notes:     none
//
// ---PHDR---

int Registry::capacity () {
	return (_capacity);
}


// +++PHDR+++
//
//  Method     : Registry::ensureCapacity()
//
//  Visibility : public
//
//  Description: Ensures that the Registry object has at least the specified
//               capacity.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  capacity        int                     desired capacity
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void Registry::ensureCapacity (int new_capacity)
{
	/*
	 * Already have sufficient capacity, do nothing
	 */
	if (new_capacity <= _capacity) {
		return;
	}

	/*
	 * Create new arrays to hold the names and objects. Copy the
	 * current names and objects into the arrays
	 */
	char ** tmp_names = new char * [new_capacity];
	void ** tmp_objects = new void * [new_capacity];

	int i;
	for (i = 0; i < _num_elements; i++) {
		tmp_names[i] = _names[i];
		tmp_objects[i] = _objects[i];
	}

	for (i = _num_elements; i < new_capacity; i++) {
		tmp_names[i] = NULL;
		tmp_objects[i] = NULL;
	}

	/*
	 * Delete the current arrays and assign the new ones to them
	 */
	if (_names != NULL)
		delete _names;
	_names = tmp_names;
	
	if (_objects!= NULL)
		delete _objects;
	_objects = tmp_objects;

	_capacity = new_capacity;
}


// +++PHDR+++
//
//  Method     : Registry::bind()
//
//  Visibility : public
//
//  Description: Binds (maps) the specified name to the object. 
//               A name may be bound to only one object at a time.
//
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  name            char *                  name to bind
//  object          void *                  object to bind
//
//
//  Returns:   TRUE on success, FALSE on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int Registry::bind (char * name, void * object)
{
	/*
	 * See if the item already exists
	 */
	if (lookup(name) != NULL) {
		return (0);
	}

	/*
	 * Increase the capacity if needed
	 */
	if (_num_elements == _capacity) {
		ensureCapacity(_capacity + _capacityIncrement);
	}

	/*
	 * Save a copy of the name
	 */
	_names[_num_elements] = new char[strlen(name) + 1];
	strcpy(_names[_num_elements], name);
	
	_objects[_num_elements] = object;
	_num_elements++;

	return (1);
}



// +++PHDR+++
//
//  Method     : Registry::unbind()
//
//  Visibility : public
//
//  Description: Removes the specified name from the bindings. 
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  name            char *                  the name to unbind
//
//
//  Returns:   TRUE on success, FALSE on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int Registry::unbind (char * name)
{
	/*
	 * Look thru the names array for a match on name, if not found
	 * get outa here
	 */
	int i;
	int index;
	for (i = 0, index = -1; index == -1 && i < _num_elements; i++) {
		if (strcmp(name, _names[i]) == 0) {
			index = i;
		}
	}

	if (index == -1) {
		return (0);
	}


	/*
	 * Shift everything down in the arrays
	 */
	_num_elements--;
	for (i = index; i < _num_elements; i++) {
		_names[i] = _names[i + 1];
		_objects[i] = _objects[i + 1];
	}

	_names[_num_elements] = NULL;
	_objects[_num_elements] = NULL;

	return (1);
}


// +++PHDR+++
//
//  Method     : Registry::lookup()
//
//  Visibility : public
//
//  Description: Returns the object bound to the specified name. If the 
//               name is not bound, NULL will be returned.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  name            char *                  the name to lookup
//
//
//  Returns:   The object bound to name. If name not bound then NULL
//             is returned.
//
//
//  Notes:     none
//
// ---PHDR---

void * Registry::lookup (char * name)
{
	/*
	 * Look thru the names array for a match if found return the
	 * pointer to the object
	 */
	for (int i = 0; i < _num_elements; i++) {
		if (strcmp(name, _names[i]) == 0) {
			return (_objects[i]);
		}
	}

	return (NULL);
}


// +++PHDR+++
//
//  Method     : Registry::list()
//
//  Visibility : public
//
//  Description: Returns an enumeration of the currently bound objects.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   An array of char *, one entry for each bound object name.
//
//
//  Notes:     none
//
// ---PHDR---

char ** Registry::list ()
{
	if (_num_elements == 0) {
		return (NULL);
	}

	/*
	 * Copy the array of names (but not the names themselves)
	 */
	char ** names = new char * [_num_elements + 1];
	names[_num_elements] = NULL;

	for (int i = 0; i < _num_elements; i++) {
		names[i] = _names[i];
	}

	return (names);
}

