package com.vsi.xmlf;

/**
 * This class encapsulates the command line passed to
 * the main() function of an executable. It processes
 * the arguments in the same way as Unixes getopt().
 *
 */
public
class CmdLine {

	public static final char NUM_CHR  = '#';
	public static final char ARG_XPTD = ':';
	public static final int EOF   = -1;
	public static final int ERROR = -2;


	protected String _optarg;
	protected int _optind;
	protected int _charpos;

	protected char _optopt;
	protected char _optsw;

	protected String _optstr;
	protected char _switches[];

	protected String _msgbuf;


	/**
	 * No args constructor.
	 */
	public CmdLine () {
		this(null);
	}


	/**
	 * Construct a CmdLine object with the specified option switches.
	 *
	 * @param  optstr  the option string (see Unix getopt(3))
	 */
	public CmdLine (String optstr) {
		_optstr   = optstr;
		_optind   = 0;
		_charpos  = 0;
		_optopt   = '\0';
		_optsw    = '\0';
		_optarg   = null;
		_msgbuf   = null;

		_switches = new char[1];
		_switches[0] = '-';
	}




	/**
	 * Process cmd line arguments (getopt substitute).<p>
	 *
	 * Notes:  This routine is a replacement for the UNIX routine getopt
	 * (which is not universally available).
	 *
	 * @param  argv	 argument vector
	 * @returns  option found, ERROR if error or EOF if at end
	 */
	public int getOpt (String argv[]) {
		return (getOpt(argv, null));
	}


	/**
	 * Process cmd line arguments (getopt substitute).<p>
	 *
	 * Notes:  This routine is a replacement for the UNIX routine getopt
	 * (which is not universally available).
	 *
	 * @param  argv	 argument vector
	 * @param  opstr  
	 * @returns  option found, ERROR if error or EOF if at end
	 */
	public int getOpt (String argv[], String optstr) {

		int argc = argv.length;
		String op;

		if (optstr == null) {
			optstr = _optstr;
		}

		_optarg = null;

		/* check if we should start over */
		if (_optind == 0) {
			_charpos = 0;
		}

		/* if past all arguments, done */
		if (_optind >= argc) {
			_charpos = 0;
			return (EOF);
		}

		/* check if new argument */
		if (_charpos == 0) {
			/* check if option & longer than 1 char */
			char ch0 = argv[_optind].charAt(0); 
			for (int i = 0; i < _switches.length; i++) {
				if (_switches[i] == ch0) {
					_charpos = 1;
					_optsw = ch0;
					break;
				}
			}

			if (_charpos == 0) {
				return (EOF);
			}
		}

		/* get next option & bump ptr */

		_optopt = argv[_optind].charAt(_charpos++); 

		/* check for double option chars */

		if (_optopt == _optsw) {
			_charpos = 0;
			_optind++;
			return (EOF);
		}

		/* check for number */

		if (Character.isDigit(_optopt)) {
			for (int i = 0; i < optstr.length(); i++) {
				char c = optstr.charAt(i);
				if (c == NUM_CHR) {
					_optopt = NUM_CHR;
					_optarg = argv[_optind].substring(
							_charpos - 1);
					_charpos = 0;
					_optind++;
					return ((int)_optopt);
				}
			}
		}

		/* check if at end of string of options */

		if (argv[_optind].length() == _charpos) {
			_charpos = 0;
			_optind++;
		}

		/* search for next option */

		int optstr_len = optstr.length();
		for (int i = 0; i < optstr_len; i++) {
			char ch = optstr.charAt(i);
			if (ch == ARG_XPTD) {
				continue;
			}

			if (_optopt != ch) {
				continue;
			}

			/* got a matching letter in string */

			/* check if no arg expected */
			if (i + 1 >= optstr_len ||
				optstr.charAt(i + 1) != ARG_XPTD) {
					return ((int)_optopt);
			}

			/* process option argument */

			if (_optind == argc) {
				/* at end of list */

				_msgbuf = "option requires an argument -- " +
					 _optopt;
				return (ERROR);
			}

			_optarg  = argv[_optind].substring(_charpos);
			_charpos = 0;
			_optind++;
			return ((int)_optopt);
		}

		_msgbuf = "invalid option -- " + _optopt;
		return (ERROR);
	}


	/**
	 * Gets the argument for the last parsed option.
	 */
	public String getOptArg () {
		return (_optarg);
	}



	/**
	 * Gets the current option index.
	 */
	public int getOptIndex () {
		return (_optind);
	}


	/**
	 * Checks if an arg is present.
	 *
	 * @param   argv  args from cmd line.
	 * @param   arg	  arg to look for.
	 * @returns true if found, false if not.
	 */
	public boolean containsArg (String argv[], char arg) {

		int argc = argv.length;

		char opts[] = new char[1];;
		opts[0] = arg;
		String optstr = new String(opts);

		boolean found = false;
		while (!found) {
			if (_optind >= argc) {
				break;
			}

			int c = getOpt(argv, optstr);
			if (c == arg) {
				found = true;
			} else if (c == EOF) {
				_optind++;
			}
		}

		_optind = 0;		/* reset for real pass */
		return (found);
	}


	/**
	 * Checks if an arg is present.
	 *
	 * @param  argv	 args from cmd line.
	 * @param  arg   arg to look for.
	 * @returns  A pointer to arg if present, null if not.
	 */
	public String getStringArg (String argv[], char arg) {

		int argc = argv.length;

		char opts[] = new char[2];
		opts[0] = arg;
		opts[1] = ':';
		String optstr = new String(opts);

		while (true) {
			if (_optind >= argc)
				break;

			int c = getOpt(argv, optstr);
			if (c == arg) {
				_optind = 0;
				return (_optarg);
			} else if (c == EOF) {
				_optind++;
			}
		}

		_optind = 0;		/* reset for real pass */
		return (null);
	}


	/**
	 * Gets the last reported error message.
	 *
	 * @returns  A pointer to error message
	 */
	public String getErrorMsg () {
		return (_msgbuf);
	}


	/**
	 * Sets option switches to look for.
	 *
	 * @param   str	 char array of switch characters.
	 */
	public void setSwitches (char switches[]) {
		_switches = switches;
	}

	
	/**
	 * Builtin tester
	 */
	public static void main (String args[]) {

		String optstr = args[0];
		String argv[] = new String[args.length - 1];
		for (int i = 1; i < args.length; i++) {
			argv[i - 1] = args[i];
		}

		CmdLine opts = new CmdLine(optstr);
		while (true) {

			int c = opts.getOpt(argv);

			if (c == CmdLine.ERROR) {
				System.err.println(opts.getErrorMsg());
				break;
			}

			if (c == CmdLine.EOF) {
				break;
			}

			System.out.println("c=" + (char)c + ", optarg=" +
				opts.getOptArg());
		}
	}
}
