package com.vsi.xmlf;

/**
 * This class takes a URL string and parses it into it's
 * protocol, host, filename, and port parts.
 *
 */
public
class Url {

	protected String _host     = null;
	protected String _protocol = null;
	protected String _filename = null;
	protected int _port        = -1;


	/**
	 * Construct a URL object from the URL specification.
	 *
	 * @param  url_spec  the URL specification
	 */
	Url (String url_spec) {

		if (url_spec == null) {
			return;
		}

		parse(url_spec);
	}


	/**
	 * Parses a URL specification into its protocol, host, filename,
	 * and port parts.
	 *
	 * @param  spec  the URL specification to parse.
	 */
	protected void parse (String spec) {

		_protocol = null;
		_host     = null;
		_filename = null;
		_port     = -1;

		/*
		 * Eliminate leading/trailing whitespace
		 */
		spec = spec.trim();
	
		int i;
		char c;
		int start = 0;
		int limit = spec.length();

		/*
		 * Look for a protocol specifier
		 */
	        for (i = start; i < limit && (c = spec.charAt(i)) != '/'; i++) {
	                if (c == ':') {
				_protocol = spec.substring(start, i - start);
                    		start = i + 1;
				break;
                	}
		}

		/*
		 * Parse the hostname
		 */
		char c1 = spec.charAt(start);
		char c2;
		if (start + 1 < limit)
			c2 = spec.charAt(start + 1);
		else
			c2 = '\0';
		if ((c1 == '/' && c2 == '/') ||
			(_protocol == null && c1 != '/')) {

			if (_protocol != null) {
				start += 2;
			}

			int end_host = limit;
	        	for (i = start; i < limit; i++) {
				c = spec.charAt(i);
				if (c == '/' || c == ':') {
					end_host = i;
					break;
				}
			}

			_host = spec.substring(start, end_host);
			start = end_host;
		}
				
	
		/*
		 * Parse the filename
		 */
		int end_filename = limit;
        	for (i = start; i < limit; i++) {
        	        if (spec.charAt(i) == ':') {
				end_filename = i;
				break;
			}
		}

		/*
		 * If we have a filename save it
		 */
		if (end_filename != start) {
			_filename = spec.substring(start, end_filename);
		}

	
		/*
		 * And finally parse the port
		 */
		start = end_filename + 1;
		if (start < limit) {
			_port = Integer.parseInt(spec.substring(start));
		}
	}


	/**
	 * Converts the Url object to a string (URL specification).
	 *
	 * @return  the URL string for the object
	 */
	public String toString () {

		StringBuffer url_buf = new StringBuffer();

		if (_protocol != null) {
			url_buf.append(_protocol);
			url_buf.append(":");
		}

		if (_host != null) {
			url_buf.append("//");
			url_buf.append(_host);
		}

		if (_filename != null) {
			if (_host != null) {
				url_buf.append("/");
			}

			url_buf.append(_filename);
		}

		if (_port != -1) {
			url_buf.append(":");
			url_buf.append(_port);
		}

		return (url_buf.toString());
	}


	/**
	 * Gets the protocol specified by the URL.
	 *
	 * @return  A String containing the protocol.
	 */
	public String getProtocol () {
		return (_protocol);
	}


	/**
	 * Gets the host specified by the URL.
	 *
	 * @return   A String containing the host.
	 */
	public String getHost () {
		return (_host);
	}


	/**
	 * Gets the filename specified by the URL.
	 *
	 * @return  A String containing the filename.
	 */
	public String getFilename () {
		return (_filename);
	}


	/**
	 * Gets the port specified by the URL.
	 *
	 * @return  The port.
	 */
	public int getPort () {
		return (_port);
	}


	/**
	 * Sets the protocol for the URL.
	 *
	 * @param  protocol  the new protocol for the URL.
	 */
	public void setProtocol (String protocol) {
		_protocol = protocol;
	}


	/**
	 * Sets the host for the URL.
	 *
	 * @param  protocol  the new host for the URL .
	 */
	public void setHost (String host) {
		_host = host;
	}


	/**
	 * Sets the filename for the URL.
	 *
	 * @param  filename  the new filename for the URL.
	 */
	public void setFilename (String filename) {
		_filename = filename;
	}


	/**
	 * Sets the port for the URL.
	 *
	 * @param  port  the new port for the URL.
	 */
	public void setPort (int port) {
		_port = port;
	}
}


