/*------------------------------------------------------------------------
 * Program to list the status of a fax request
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"s:t:WU:H:Z:?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] [reqid]\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
	fprintf(fp, "  -t secs       Time between polls\n");
	fprintf(fp, "  -s time       Start from time\n");
}

/*------------------------------------------------------------------------
 * extract a name from a tag list
 */
static const char *
tag_name (V_TAG *tags, const char *name)
{
	const char *p = VSI_Tag_Get_Data_By_Name(tags, name, 0, 0);

	if (p == 0 || *p == 0)
		return ("<none>");
	return (p);
}

/*------------------------------------------------------------------------
 * display an OLOG entry
 */
static int
display_olog (V_TAG *tags)
{
	const char *seqno	= tag_name(tags, V_TAG_OLOG_SEQNO);
	const char *active 	= tag_name(tags, V_TAG_OLOG_ACTIVE);
	const char *rresult	= tag_name(tags, V_TAG_OLOG_RRESULT);
	const char *rresstr	= tag_name(tags, V_TAG_OLOG_RRESSTR);

	if (atoi(rresult) == V_RRESULT_SENDING)
	{
		const char *	npages	= tag_name(tags, V_TAG_OLOG_NPAGES);
		const char *	pct	= tag_name(tags, V_TAG_OLOG_NAKS);

		printf("%-8s %-8s %s  %s  %s\n",
			seqno, active, rresstr, npages, pct);
	}
	else
	{
		printf("%-8s %-8s %s\n", seqno, active, rresstr);
	}

	return (strcmp(active, "done") == 0);
}

/*------------------------------------------------------------------------
 * display a list of OLOG tags
 */
static void
display_list (V_LIST *list)
{
	V_LIST * l;

	for (l=list; l; l=VSI_List_Next(l))
	{
		V_TAG * tags = VSI_List_Tags(l);

		display_olog(tags);
	}

	printf("------------------------------\n");
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	V_LIST *	list;
	V_TAG *		tags;
	const char *	pgmname		= argv[0];
	const char *	reqid		= 0;
	int		timeout		= 30;
	time_t		start		= 0;
	time_t		now;
	int		c;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * already processed
			 */
			break;

		case 's':
			/*------------------------------------------------
			 * set start time
			 */
			start = VSI_Util_Str_To_Time(VSI_Opt_Argstr(), 0);
			break;

		case 't':
			/*------------------------------------------------
			 * set timeout
			 */
			timeout = atoi(VSI_Opt_Argstr());
			if (timeout <= 0)
				timeout = 1;
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * check if a reqid was specified
	 */
	c = VSI_Opt_Index();
	if (c < argc)
	{
		reqid = argv[c++];
	}

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * loop on messages
	 */
	while (TRUE)
	{
		if (reqid == 0)
		{
			now = VSI_Server_Newmsg(vptr, 0, V_MSG_SNTFAX, start,
				0);
			if (now == (time_t)-1)
				break;

			if (now == 0)
			{
				start = time(0);
			}
			else
			{
				list = VSI_Server_Newmsg_Get(vptr, 0,
					V_MSG_SNTFAX, start, 0);
				start = now;

				if (list != 0)
				{
					display_list(list);
					list = VSI_List_Free(list, 0);
				}
			}
		}
		else
		{
			tags = VSI_Server_Status(vptr, reqid, 0);
			if (tags == 0)
				break;
			if (display_olog(tags))
				break;
		}

		VSI_Util_Sleep(timeout * 1000);
	}

	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	return (EXIT_SUCCESS);
}
