/*------------------------------------------------------------------------
 * Program to convert a file and submit a fax request
 *
 * This program converts external tagnames to internal names.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"dt:WU:H:Z:?"

/*------------------------------------------------------------------------
 * tag conversion list
 */
struct tag_list
{
	char *	tag_user;		/* user tag name	*/
	char *	tag_name;		/* our  tag name	*/
	char *	tag_dflt;		/* default if not null	*/
};
typedef struct tag_list TAG_LIST;

static const TAG_LIST tag_list[] =
{
	/*----------------------------------------------------------------
	 * DATA tag		OUR tag			default
	 */
	{ "TOFAXNUM",		V_TAG_ENV_TOFNUM,	0 },
	{ "TONAME",		V_TAG_ENV_TONAME,	0 },
	{ "TOCOMPANY",		V_TAG_ENV_TOCOMP,	0 },

	{ "FROMFAXNUM",		V_TAG_ENV_FRFNUM,	0 },
	{ "FROMNAME",		V_TAG_ENV_FRNAME,	0 },
	{ "FROMCOMPANY",	V_TAG_ENV_FRCOMP,	0 },

	{ "NOTE",		V_TAG_ENV_NOTEFILE,	0 },
	{ "UNIQUEID",		V_TAG_ENV_APPREF,	0 },
	{ "WHO",		V_TAG_ENV_CLIENTID,	0 },

	{ 0, 0, 0 }
};

/*------------------------------------------------------------------------
 * conversion data
 */
#define NOTIFY_NONE	0x00			/* no notification	*/
#define NOTIFY_OK	0x01			/* notify if successful	*/
#define NOTIFY_FAIL	0x02			/* notify if failed	*/

struct cvt_data
{
	char	data_path[V_TAG_DATALEN];	/* new file to send	*/
	char	note_path[V_TAG_DATALEN];	/* coverpage notefile	*/
	FILE *	fp_note;			/* notefile stream	*/
	int	notify;				/* notify mode		*/
	V_TAG *	tags;				/* envelope tags	*/
};
typedef struct cvt_data CVT_DATA;

/*------------------------------------------------------------------------
 * process tags
 */
static int
process_tag (void *udata, const char *name, const char *data, char *msgbuf)
{
	CVT_DATA * cd = (CVT_DATA *)udata;
	const TAG_LIST * tl;

	/*----------------------------------------------------------------
	 * special case of notefile
	 *
	 * Open the notefile if not opened & add this line to it.
	 */
	if (strcmp(name, "NOTE") == 0)
	{
		if (cd->fp_note == 0)
			cd->fp_note = fopen(cd->note_path, "w");
		if (cd->fp_note != 0)
			fprintf(cd->fp_note, "%s\n", data);
		return (0);
	}

	/*----------------------------------------------------------------
	 * special case of notify modes
	 */
	if (strcmp(name, "NOTIFY_OK") == 0)
	{
		cd->notify |= NOTIFY_OK;
		return (0);
	}

	if (strcmp(name, "NOTIFY_ERROR") == 0)
	{
		cd->notify |= NOTIFY_FAIL;
		return (0);
	}

	/*----------------------------------------------------------------
	 * find tag in list & bail if not there (not an error)
	 */
	for (tl=tag_list; tl->tag_user; tl++)
	{
		if (strcmp(name, tl->tag_user) == 0)
			break;
	}

	if (tl->tag_user == 0)
		return (0);

	/*----------------------------------------------------------------
	 * add tag to list
	 */
	if (tl->tag_dflt == 0)
		cd->tags = VSI_Tag_Add(cd->tags, tl->tag_name, data, 0);
	else
		cd->tags = VSI_Tag_Add(cd->tags, tl->tag_name, tl->tag_dflt, 0);

	return (0);
}

/*------------------------------------------------------------------------
 * line convert routine
 */
static int
convert_line (void *udata, const char *line, FILE *fp_out, char *msgbuf)
{
	char		tag_buf[BUFSIZ];
	const char *	l;
	char *		b;
	char *		t;
	char *		tag_name;
	char *		tag_data;
	int		rc = 0;

	/*----------------------------------------------------------------
	 * skip "userid:" line (What is this line?)
	 */
	if (strncmp(line, "USERID:", 7) == 0)
		return (0);

	/*----------------------------------------------------------------
	 * just copy any lines with no tags
	 */
	if (*line != '<')
	{
		fputs(line, fp_out);
		return (0);
	}

	/*----------------------------------------------------------------
	 * got a tag line, so do all tags in it
	 */
	while (TRUE)
	{
		/*--------------------------------------------------------
		 * point to next tag
		 */
		l = strchr(line, '<');
		if (l == 0)
			break;

		/*--------------------------------------------------------
		 * collect tag in buffer
		 */
		b = tag_buf;
		for (l++; *l; l++)
		{
			if (*l == '>')
			{
				l++;
				break;
			}
			*b++ = *l;
		}
		*b = 0;

		line = l;

		/*--------------------------------------------------------
		 * strip off trailing white-space
		 */
		t = tag_buf;
		for (b=tag_buf; *b; b++)
		{
			if (! isspace(*b))
				t = b+1;
		}
		*t = 0;

		/*--------------------------------------------------------
		 * break out <name: data>
		 */
		b = strchr(tag_buf, ':');
		if (b == 0)
		{
			tag_name = tag_buf;
			tag_data = "";
		}
		else
		{
			tag_name = tag_buf;
			*b++ = 0;
			for (; *b; b++)
			{
				if (! isspace(*b))
					break;
			}
			tag_data = b;
		}

		/*--------------------------------------------------------
		 * now process this tag
		 */
		rc = process_tag(udata, tag_name, tag_data, msgbuf);
		if (rc)
			break;
	}

	return (rc);
}

/*------------------------------------------------------------------------
 * file convert routine
 */
static V_TAG *
convert_file (V_TAG *tags, const char *pathname, const char *tmp_dir,
	char *msgbuf)
{
	CVT_DATA	cd;
	FILE *		fp_inp;
	FILE *		fp_out;
	int		rc;

	*msgbuf = 0;

	/*----------------------------------------------------------------
	 * fill in cvt-data struct
	 */
	{
		const char *	p;
		const char *	s;
		char *		t;

		/*--------------------------------------------------------
		 * get temp directory to use
		 */
		if (tmp_dir == 0 || *tmp_dir == 0)
		{
			tmp_dir = VSI_Util_Temp_Dir(0);
		}

		/*--------------------------------------------------------
		 * get basename of path
		 */
		p = pathname;
		for (s=p; s; )
		{
			s = strpbrk(p, "/\\");
			if (s != 0)
				p = s+1;
		}

		/*--------------------------------------------------------
		 * create base part of temp names (dir/name)
		 */
		sprintf(cd.data_path, "%s/%s", tmp_dir, p);
		t = strrchr(cd.data_path, '.');
		if (t != 0)
			*t = 0;

		/*--------------------------------------------------------
		 * our temp data file is "<tmp-dir>/<basename(file).pcl"
		 * our temp note file is "<tmp-dir>/<basename(file).ntf"
		 */
		strcpy(cd.note_path, cd.data_path);
		strcat(cd.data_path, ".pcl");
		strcat(cd.note_path, ".ntf");
	}

	cd.fp_note = 0;
	cd.notify  = NOTIFY_NONE;
	cd.tags    = tags;

	/*----------------------------------------------------------------
	 * open input file
	 */
	fp_inp = fopen(pathname, "r");
	if (fp_inp == 0)
	{
		sprintf(msgbuf, "Cannot open input %s", pathname);
		return (cd.tags);
	}

	/*----------------------------------------------------------------
	 * open output file
	 */
	fp_out = fopen(cd.data_path, "w");
	if (fp_out == 0)
	{
		fclose(fp_inp);
		sprintf(msgbuf, "Cannot open output %s", cd.data_path);
		return (cd.tags);
	}

	/*----------------------------------------------------------------
	 * now process all lines in the file
	 */
	rc = 0;
	while (TRUE)
	{
		char	line[BUFSIZ];

		if (fgets(line, sizeof(line), fp_inp) == 0)
			break;

		rc = convert_line(&cd, line, fp_out, msgbuf);
		if (rc)
			break;
	}

	fclose(fp_inp);
	fclose(fp_out);

	/*----------------------------------------------------------------
	 * If any error, delete all temp files & return.
	 */
	if (rc)
	{
		VSI_Tag_Delete_Files(cd.tags, tmp_dir, 0);
		return (cd.tags);
	}

	/*----------------------------------------------------------------
	 * Add any notification that was specified
	 */
	{
		const char *	np = "";

		switch (cd.notify)
		{
		case NOTIFY_NONE:		np = V_ENUM_NOTIFY_NONE; break;
		case NOTIFY_OK:			np = V_ENUM_NOTIFY_OK;   break;
		case NOTIFY_FAIL:		np = V_ENUM_NOTIFY_FAIL; break;
		case NOTIFY_OK | NOTIFY_FAIL:	np = V_ENUM_NOTIFY_BOTH; break;
		}

		cd.tags = VSI_Tag_Add(cd.tags, V_TAG_ENV_NOTIFY, np, 0);
	}

	/*----------------------------------------------------------------
	 * Add notefile if one was opened
	 */
	if (cd.fp_note != 0)
	{
		fclose(cd.fp_note);
		cd.fp_note = 0;
		cd.tags = VSI_Tag_Add(cd.tags,
			V_TAG_ENV_NOTEFILE, cd.note_path, 0);
	}

	/*----------------------------------------------------------------
	 * Add the data file & original pathname
	 */
	cd.tags = VSI_Tag_Add(cd.tags, V_TAG_ENV_FILELCL,  cd.data_path, 0);
	cd.tags = VSI_Tag_Add(cd.tags, V_TAG_ENV_ORIGPATH, pathname, 0);

	return (cd.tags);
}

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] file\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -t tag=val    Tag to add\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
	fprintf(fp, "  -d            Delete file if successful\n");
	fprintf(fp, "file            File to convert\n");
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	char		reqid[V_TAG_DATALEN];
	V_TAG *		tags		= 0;
	const char *	pgmname		= argv[0];
	const char *	pathname	= 0;
	const char *	tmp_dir		= VSI_Util_Temp_Dir(0);
	int		do_delete	= FALSE;
	int		opt_index;
	int		c;
	int		rc;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * initialize tag list with entries from system ini file
	 */
	tags = VSI_Tag_Add_Ini_Sys(tags, 0);

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'd':
			/*------------------------------------------------
			 * delete file if successful
			 */
			do_delete = TRUE;
			break;

		case 't':
			/*------------------------------------------------
			 * process tag in form "tag=value"
			 */
			tags = VSI_Tag_Add_Str(tags, VSI_Opt_Argstr(), msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * processed already
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * should be exactly one arg here ...
	 */
	opt_index = VSI_Opt_Index();
	if ((argc - opt_index) != 1)
	{
		fprintf(stderr, "%s: Invalid number of args\n", pgmname);
		usage(pgmname, stderr);
		return (EXIT_FAILURE);
	}

	pathname = argv[opt_index++];

	/*----------------------------------------------------------------
	 * Now convert the file & submit it if conversion is successful.
	 */
	tags = convert_file(tags, pathname, tmp_dir, msgbuf);
	if (*msgbuf != 0)
	{
		rc = -1;
	}
	else
	{
		/*--------------------------------------------------------
		 * login to the server
		 */
		vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
		if (vptr == 0)
		{
			rc = -1;
		}
		else
		{
			/*------------------------------------------------
			 * Submit the tag list to be faxed & logout.
			 */
			rc = VSI_Server_Submit(vptr, reqid, tags, msgbuf);
			VSI_Server_Logout(vptr, 0);
		}
	}

	/*----------------------------------------------------------------
	 * delete any temp files we created
	 */
	VSI_Tag_Delete_Files(tags, tmp_dir, 0);

	/*----------------------------------------------------------------
	 * free the tag list
	 */
	tags = VSI_Tag_Free(tags, 0);

	/*----------------------------------------------------------------
	 * delete orig file if requested & submit was successful
	 */
	if (rc == 0 && do_delete)
		remove(pathname);

	/*----------------------------------------------------------------
	 * output error msg or request ID
	 */
	if (rc)
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
	else
		printf("%s\n", reqid);

	/*----------------------------------------------------------------
	 * free our context
	 */
	VSI_Context_Free(vctx, 0);

	return (rc == 0 ? EXIT_SUCCESS : EXIT_FAILURE);
}
