// +++FHDR+++
//
//
//   Class Name : Url
//   Super Class: n/a
//
//   Description: This class takes a URL string and parses it into it's
//                protocol, host, filename, and port parts.
//
//
//   Notes      : none
//
//
//   Method - Description
//   ------------------------------------------------------------------------
//
//
// ---FHDR---


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "Url.h"


static char * stringCopy (char * src);



// +++PHDR+++
//
//  Method     : Url::Url()
//
//  Visibility : public
//
//  Description: Public constructor for Url class
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  url_spec        char *                  URL specification string
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

Url::Url (char * url_spec)
{
	_protocol = NULL;
	_host     = NULL;
	_filename = NULL;
	_port = -1;

	if (url_spec == NULL) {
		return;
	}

	parse(url_spec, &_protocol, &_host, &_filename, &_port);
}


// +++PHDR+++
//
//  Method     : Url::~Url
//
//  Visibility : public
//
//  Description: Destructor for Url class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

Url::~Url ()
{
	if (_protocol != NULL) {
		delete (_protocol);
		_protocol = NULL;
	}

	if (_host!= NULL) {
		delete (_host);
		_host = NULL;
	}

	if (_filename != NULL) {
		delete (_filename);
		_filename = NULL;
	}
}



// +++PHDR+++
//
//  Method     : Url::parse(
//
//  Visibility : public
//
//  Description: Parses a URL specification into its protocol, host, filename,
//               and port parts.
//
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  spec            char *                  the URL specification to parse
//  protocol        char **                 storage for the protocol
//  host            char **                 storage for the host
//  filename        char **                 storage for the filename
//  port            int  *                  storage for the port
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---


void Url::parse (char * spec, char ** protocol, 
	char ** host, char ** filename, int * port)
{
	*protocol = NULL;
	*host = NULL;
	*filename = NULL;
	*port = -1;

	/*
	 * Eliminate trailing whitespace
	 */
	int limit = strlen(spec);
	while ((limit > 0) && (spec[limit - 1] <= ' ')) {
	        limit--;
	}

        /*
	 * Eliminate leading whitespace
	 */
	int start = 0;
	while ((start < limit) && (spec[start] <= ' ')) {
		start++;
	}

	
	int i;
	int c;

	/*
	 * Look for a protocol specifier
	 */
        for (i = start; i < limit && (c = spec[i]) != '/'; i++) {
                if (c == ':') {
               		*protocol = new char[i - start + 1];
			strncpy(*protocol, spec + start, i - start);
			(*protocol)[i - start] = '\0';
                    	start = i + 1;
			break;
                }
	}

	/*
	 * Parse the hostname
	 */
	if ((spec[start] == '/' && spec[start + 1] == '/') ||
		(*protocol == NULL && spec[start] != '/')) {

		if (*protocol != NULL) {
			start += 2;
		}

		int end_host = limit;
        	for (i = start; i < limit; i++) {
			if (spec[i] == '/' || spec[i] == ':') {
				end_host = i;
				break;
			}
		}

		*host = new char[end_host - start + 1];
		strncpy(*host, spec + start, end_host - start);
		(*host)[end_host - start] = '\0';
		start = end_host;
	}
				
	
	/*
	 * Parse the filename
	 */
	int end_filename = limit;
        for (i = start; i < limit; i++) {
                if (spec[i] == ':') {
			end_filename = i;
			break;
		}
	}

	/*
	 * If we have a filename save it
	 */
	if (end_filename != start) {
		*filename= new char[end_filename - start + 1];
		strncpy(*filename, spec + start, end_filename - start);
		(*filename)[end_filename - start] = '\0';
	}

	
	/*
	 * And finally parse the port
	 */
	start = end_filename + 1;
	if (start < limit) {
		*port = (int) atoi(spec + start);
	}
}


// +++PHDR+++
//
//  Method     : Url::toString()
//
//  Visibility : public
//
//  Description: Converts the Url object to a string (URL specification).
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   A char * containg the URL specificatiog for the Url object.
//
//
//  Notes:     none
//
// ---PHDR---

char * Url::toString ()
{
	int len = 0;
	len += (_protocol != NULL ? strlen(_protocol) + 1 : 0);
	len += (_host != NULL ? strlen(_host) + 2 : 0);
	len += (_filename != NULL ? strlen(_filename) + 1 : 0);
	len += (_port != -1 ? 8 : 0);

	char * spec = new char[len + 1];
	*spec = '\0';

	if (_protocol != NULL) {
		strcat(spec, _protocol);
		strcat(spec, ":");
	}

	if (_host != NULL) {
		strcat(spec, "//");
		strcat(spec, _host);
	}

	if (_filename != NULL) {
		if (_host != NULL)
			strcat(spec, "/");

		strcat(spec, _filename);
	}

	if (_port != -1) {
		char port[8];
		sprintf(port, ":%d", _port);
		strcat(spec, port);
	}

	return (spec);
}


// +++PHDR+++
//
//  Method     : Url::getProtocol()
//
//  Visibility : public
//
//  Description: Returns the protocol specified by the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   A char * containing the protocol.
//
//
//  Notes:     none
//
// ---PHDR---

char * Url::getProtocol ()
{
	return (_protocol);
}


// +++PHDR+++
//
//  Method     : Url::getHost()
//
//  Visibility : public
//
//  Description: Returns the host specified by the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   A char * containing the host.
//
//
//  Notes:     none
//
// ---PHDR---

char * Url::getHost ()
{
	return (_host);
}


// +++PHDR+++
//
//  Method     : Url::getFilename()
//
//  Visibility : public
//
//  Description: Returns the filename specified by the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   A char * containing the filename.
//
//
//  Notes:     none
//
// ---PHDR---

char * Url::getFilename ()
{
	return (_filename);
}


// +++PHDR+++
//
//  Method     : Url::getPort()
//
//  Visibility : public
//
//  Description: Returns the port specified by the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   The port.
//
//
//  Notes:     none
//
// ---PHDR---

int Url::getPort ()
{
	return (_port);
}


// +++PHDR+++
//
//  Method     : Url::setProtocol()
//
//  Visibility : public
//
//  Description: Sets the protocol for the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  protocol        char *                  the new protocol for the URL 
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void Url::setProtocol (char * protocol)
{
	_protocol = stringCopy(protocol);
}


// +++PHDR+++
//
//  Method     : Url::setHost()
//
//  Visibility : public
//
//
//  Description: Sets the host for the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  protocol        char *                  the new host for the URL 
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void Url::setHost (char * host)
{
	_host = stringCopy(host);
}


// +++PHDR+++
//
//  Method     : Url::setFilename()
//
//  Visibility : public
//
//  Description: Sets the filename for the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  filename        char *                  the new filename for the URL 
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void Url::setFilename (char * filename)
{
	_filename = stringCopy(filename);
}


// +++PHDR+++
//
//  Method     : Url::setPort()
//
//  Visibility : public
//
//  Description: Sets the port for the URL.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  port            int                     the new port for the URL 
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void Url::setPort (int port)
{
	_port = port;
}


// +++PHDR+++
//
//  Method     : stringCopy()
//
//  Visibility : static
//
//  Description: Duplicates a string using the new operator for storage.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  src             char *                  The string to duplicate
//
//
//  Returns:   A copy of the source string.
//
//
//  Notes:     none
//
// ---PHDR---

static char * stringCopy (char * src)
{	
	if (src == NULL) {
		return (NULL);
	}

	char * dst = new char[strlen(src) + 1];
	strcpy(dst, src);
	return (dst);
}

