package com.vsi.xmlf.beans;

import com.vsi.xmlf.*;


import javax.swing.Box;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import javax.swing.border.BevelBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.ComponentUI;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import java.io.IOException;
import java.util.Vector;


/**
 * This component present a Wizard UI and manages a set of panels
 * with a "Next" and "Back" button. Panels displayed by this component
 * must implement the WizardPanel interface.
 */
public
class WizardManager extends JPanel implements java.io.Serializable {


	public static final int FORWARD =  1;
	public static final int BACK    = -1;
	

	protected transient JDialog _dialog = null;

	protected transient JPanel  _cards  = null;
	protected transient CardLayout _layout = null;

	protected transient JLabel _bitmap_label = null;
	protected boolean _bitmap_visible = false;

	protected transient JPanel _buttons     = null;
	protected transient JButton _back_btn   = null;
	protected transient JButton _next_btn   = null;
	protected transient JButton _help_btn   = null;
	protected transient JButton _cancel_btn = null;

	protected transient Component _help_spacer = null;
	protected boolean _help_btn_visible = false;

	protected Vector _panels   = new Vector();
	protected int _panel_index = -1;


	protected Vector _cancel_listeners = new Vector();
	protected Vector _finish_listeners = new Vector();


	public WizardManager () {
		createForm();
	}


	private void createForm () {
		setLayout(new GridBagLayout());

		/*
		 * We use a CardLayout to manage the panels
		 */
		_layout = new CardLayout();
		_cards = new JPanel(_layout);
		_cards.setBorder(new EmptyBorder(0, 0, 0, 0));

		_bitmap_label = new JLabel("");
		_bitmap_label.setName("Bitmap");
		_bitmap_label.setBorder(new BevelBorder(BevelBorder.LOWERED));
		_bitmap_label.setVisible(false);
		

		FlowLayout fl = new FlowLayout(FlowLayout.RIGHT);
		fl.setHgap(0);
		fl.setVgap(0);
		_buttons = new JPanel(fl);
		_buttons.setBorder(new EmptyBorder(8, 0, 0, 0));

		_back_btn = new JButton("< Back");
		_back_btn.setMnemonic('B');
		ActionListener al = new BackAction();
		_back_btn.addActionListener(al);
		_back_btn.setEnabled(false);
		_buttons.add(_back_btn);

		_next_btn = new JButton("Finish");
		al = new NextAction();
		_next_btn.addActionListener(al);
		_buttons.add(_next_btn);

		_buttons.add(Box.createHorizontalStrut(32));

		_cancel_btn = new JButton("Cancel");
		al = new CancelAction();
		_cancel_btn.addActionListener(al);
		registerKeyboardAction(al,
			KeyStroke.getKeyStroke((char)KeyEvent.VK_ESCAPE), 
			WHEN_IN_FOCUSED_WINDOW);
		_buttons.add(_cancel_btn);

		_help_spacer = Box.createHorizontalStrut(16);

		_help_btn = new JButton("Help");
		_help_btn.setMnemonic('H');
		al = new HelpAction();
		_help_btn.addActionListener(al);
		_help_btn.setEnabled(false);

		JPanel bottom = new JPanel(new BorderLayout());
		bottom.setBorder(new EmptyBorder(0, 0, 0, 0));
		bottom.add(new JSeparator(), BorderLayout.NORTH);
		bottom.add(_buttons, BorderLayout.SOUTH);

		GridBagConstraints gbc = new GridBagConstraints();
		gbc.anchor = GridBagConstraints.NORTHWEST;
		gbc.fill = GridBagConstraints.NONE;
		gbc.gridy = 0;
		gbc.gridx = 0;
		gbc.weightx = 0.0;
		gbc.weighty = 0.0;
		gbc.insets.right  = 0;
		gbc.insets.left   = 8;
		gbc.insets.top    = 8;
		gbc.insets.bottom = 0;
		add(_bitmap_label, gbc);

		gbc.insets.right  = 8;
		gbc.insets.left   = 16;
		gbc.insets.top    = 8;
		gbc.insets.bottom = 0;
		gbc.weightx = 1.0;
		gbc.weighty = 1.0;
		gbc.anchor  = GridBagConstraints.NORTHWEST;
		gbc.fill    = GridBagConstraints.BOTH;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbc.gridy   = 0;
		gbc.gridx   = 1;
		add(_cards, gbc);

		gbc.insets.right  = 8;
		gbc.insets.left   = 8;
		gbc.insets.top    = 16;
		gbc.insets.bottom = 8;
		gbc.gridwidth = GridBagConstraints.REMAINDER;
		gbc.anchor = GridBagConstraints.SOUTHEAST;
		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.gridx = 0;
		gbc.gridy = 1;
		add(bottom, gbc);


		/*
		 * Lock the size of the Next button so it won't resize
		 * between "Next" and "Finish"
		 */
		Dimension s = _next_btn.getPreferredSize();
		_next_btn.setText("Next >");
		_next_btn.setMnemonic('N');
		Dimension s1 = _next_btn.getPreferredSize();

		if (s1.width > s.width && s1.width > 0) {
			_next_btn.setPreferredSize(s1);
		} else if (s.width > 0) {
			_next_btn.setPreferredSize(s);
		}
	}


	private void writeObject(java.io.ObjectOutputStream out)
		throws IOException {

		JComponent parent = this; 
	 	for (Container p = getParent(); p != null; p = p.getParent()) {
		        if (p instanceof JComponent) {
		            parent = (JComponent)p;
		        }
		}

		ComponentUI ui = UIManager.getUI(parent);
		ui.uninstallUI(parent);

		try {
			out.defaultWriteObject();
		} catch (IOException ioe) {
			throw new IOException(ioe.getMessage());	
		} finally {
			ui.installUI(parent);
		}
	}


	/**
	 * Show the panel at the specified index.
	 *
	 * @param  index  the index of the panel to show.
	 */
	public void show (int index) {

		/*
		 * Return if the panel index hasn't changed
		 */
		if (index == _panel_index) {
			return;
		}

		/*
		 * Validate the index
		 */
		if (index < 0 || index >= getNumPanels()) {
			return;
		}

		int direction;
		if (index > _panel_index) {
			direction = WizardManager.FORWARD;
		} else {
			direction = WizardManager.BACK;
		}

		WizardPanel wp = getPanelAt(_panel_index);
		if (wp != null && !wp.leave(direction)) {
			return;
		}

		Panel p = (Panel)_panels.elementAt(index);
		wp = p.getPanel();
		
		if (wp.getHasHelp()) {
			_help_btn.setEnabled(true);
		} else {
			_help_btn.setEnabled(true);
		}

		_layout.show(_cards, p.getName());

		if (index > 0 && _panel_index < 1) {
			_back_btn.setEnabled(true);
		} else if (index == 0) {
			_back_btn.setEnabled(false);
		} 

		if (index == getNumPanels() - 1) {
			_next_btn.setText("Finish");
			_next_btn.setMnemonic('F');
		} else if (_panel_index == getNumPanels() - 1 &&
			getNumPanels() > 1) {
			_next_btn.setText("Next >");
			_next_btn.setMnemonic('N');
		}

		_panel_index = index;

		_next_btn.requestFocus();

		wp.enter(direction);
	}



	public WizardPanel getPanelAt (int index) {

		if (index < 0 || index >= getNumPanels()) {
			return (null);
		}

		Panel p = (Panel)_panels.elementAt(index);
		return (p.getPanel());
	}
	

	/**
	 * Go back one panel.
	 */
	public void back () {
		show(_panel_index - 1);
	}


	/**
	 * Go forward one panel.
	 */
	public void next () {
		if (_panel_index == getNumPanels() - 1) {
			fireFinishActionPerformed();
			return;
		}

		show(_panel_index + 1);
	}


	/**
	 * This class listens for the "Back" button.
	 */
	class BackAction implements ActionListener, java.io.Serializable {

		BackAction () {
		}

	        public void actionPerformed(ActionEvent e) {
			back();
	        }
	}


	/**
	 * This class listens for the "Next" button.
	 */
	class NextAction implements ActionListener, java.io.Serializable {

		NextAction () {
		}

	        public void actionPerformed(ActionEvent e) {
			next();
	        }
	}


	/**
	 * This class listens for the "Cancel" button.
	 */
	class CancelAction implements ActionListener, java.io.Serializable {

		CancelAction () {
		}

	        public void actionPerformed(ActionEvent e) {
			fireCancelActionPerformed();
	        }
	}


	/**
	 * This class listens for the "Help" button.
	 */
	class HelpAction implements ActionListener, java.io.Serializable {

		HelpAction () {
		}

	        public void actionPerformed(ActionEvent e) {
			WizardPanel wp = getPanelAt(_panel_index);
			if (wp.getHasHelp()) {
				wp.doHelp(WizardManager.this);
			}
	        }
	}


	/**
	 * This class simply is used as a map of panel component to panel name.
	 */
	class Panel implements java.io.Serializable {

		WizardPanel _panel = null;
		String _name = null;

		Panel (WizardPanel p, String name) {

			_panel = p;
			_name  = name;
		}

		WizardPanel getPanel () {
			return (_panel);
		}

		String getName () {
			return (_name);
		}
	}


	/**
	 * Get the current look and feel.
	 */
	public String getLookAndFeel () {
		return (GuiUtils.getLookAndFeel());
	}


	/**
	 * Set the look and feel.<p>
	 * NOTE: This sets the look and feel for all components using Swing.
	 */
	public void setLookAndFeel (String plaf_name) {

		try {
			GuiUtils.setLookAndFeel(plaf_name, this);
		} catch (Exception ignore) {
		}
	}


	/**
	 * Get the number of managed panels.
	 */
	public int getNumPanels () {
		return (_panels.size());
	}


	/**
	 *
	 */
	public void setBitmapIcon (Icon icon) {

		_bitmap_label.setIcon(icon);

		if (icon == null) {
			return;
		}

		Dimension d = new Dimension(
			icon.getIconWidth(), icon.getIconHeight());
		_bitmap_label.setPreferredSize(d);
		_bitmap_label.setMinimumSize(d);

		revalidate();
	}


	/**
	 *
	 */
	public Icon getBitmapIcon () {
		return (_bitmap_label.getIcon());
	}


	/**
	 *
	 */
	public void setBitmapVisible (boolean flag) {
		if (flag == _bitmap_label.isVisible()) {
			return;
		}

		_bitmap_label.setVisible(flag);

		Container p = getParent();
		if (p instanceof JComponent) {
			((JComponent)p).revalidate();
		}
	}


	/**
	 *
	 */
	public boolean isBitmapVisible () {
		return (_bitmap_label.isVisible());
	}


	/**
	 * Add a panel.
	 */
	public void addPanel (WizardPanel p, String name) {

		Panel panel = new Panel(p, name);
		_panels.addElement(panel);
		_cards.add(p.getComponent(), name);
	}


	/**
	 * Add a panel.
	 */
	public void addPanel (WizardPanel p) {

		Component c = p.getComponent();
		Panel panel = new Panel(p, c.getName());
		_panels.addElement(panel);
		_cards.add(p.getComponent(), c.getName());
	}



	/**
	 * Returns true if the "Help" button is setup to be visible.
	 */
	public boolean isHelpButtonVisible () {
		return (_help_btn_visible);
	}


	/**
	 * Displays hides the "Help" button.
	 */
	public void setHelpButtonVisible (boolean flag) {

		if (flag == _help_btn_visible) {
			return;
		}

		if (flag) {
			_buttons.add(_help_spacer);
			_buttons.add(_help_btn);
			_help_btn_visible = true;
		} else {
			_buttons.remove(_help_spacer);
			_buttons.remove(_help_btn);
			_help_btn_visible = false;
		}

		revalidate();
	}


	/**
	 *
	 */
	public void setNextButtonEnabled (boolean enabled) {
		_next_btn.setEnabled(enabled);
	}


	/**
         * Add a finish action listener.
         *
         * @param  l  The ActionListener to add.
         */
	public synchronized void addFinishActionListener (ActionListener l) {

               if (!_finish_listeners.contains(l)) {
                       _finish_listeners.addElement(l);
               }
        }


	/**
	 * Remove a finish action listener.
	 *
	 * @param  l  The ActionListener to remove.
	 */
	public synchronized void removeFinishActionListener (ActionListener l) {

               if (!_finish_listeners.contains(l)) {
                       _finish_listeners.removeElement(l);
               }
        }


        /**
         * Notify listening objects of a finish action
         *
         */
        public void fireFinishActionPerformed () {

                /*
                 * Make a copy of the listener object vector so that it cannot
                 * be changed while we are firing events
                 */
                Vector v;
                synchronized(this) {
                        if (_finish_listeners.size() < 1) {
                                return;
                        }

                        v = (Vector)_finish_listeners.clone();
                }

                /*
                 * Create the event object
                 */
                ActionEvent evt = new ActionEvent(
			this, ActionEvent.ACTION_PERFORMED, "finish");


                /*
                 * Fire the event to all listeners
                 */
                int count = v.size();
                for (int i = 0; i < count; i++) {
                        ActionListener l = (ActionListener)v.elementAt(i);
                        l.actionPerformed(evt);
                }
        }


	/**
	 * Add a cancel action listener.
	 *
	 * @param  l  The ActionListener to add.
	 */
	public synchronized void addCancelActionListener (ActionListener l) {

               if (!_cancel_listeners.contains(l)) {
                       _cancel_listeners.addElement(l);
               }
        }


        /**
         * Remove a cancel action listener.
         *
         * @param  l  The ActionListener to remove.
         */
        public synchronized void removeCancelActionListener (ActionListener l) {

               if (!_cancel_listeners.contains(l)) {
                       _cancel_listeners.removeElement(l);
               }
        }


        /**
         * Notify listening objects of a cancel action
         *
         */
        public void fireCancelActionPerformed () {

                /*
                 * Make a copy of the listener object vector so that it cannot
                 * be changed while we are firing events
                 */
                Vector v;
                synchronized(this) {
                        if (_cancel_listeners.size() < 1) {
                                return;
                        }

                        v = (Vector)_cancel_listeners.clone();
                }

                /*
                 * Create the event object
                 */
                ActionEvent evt = new ActionEvent(
			this, ActionEvent.ACTION_PERFORMED, "cancel");


                /*
                 * Fire the event to all listeners
                 */
                int count = v.size();
                for (int i = 0; i < count; i++) {
                        ActionListener l = (ActionListener)v.elementAt(i);
                        l.actionPerformed(evt);
                }
        }
}

