package com.vsi.xmlf;

import java.io.InputStream;
import java.io.CharArrayWriter;
import java.io.ByteArrayInputStream;
import org.xml.sax.*;
import org.xml.sax.helpers.*;

/**
 * This class encapsulates the SAX API. Decoders are registered and passed
 * into this classes constructor. When the <code>parse</code> method is
 * called it will eventually call the decoder to create a XmlfObject.
 *
 */
public
class XmlParser extends HandlerBase {


	protected Registry _decoders           = null;
	protected XmlElement _root_element     = null;
	protected XmlElement _current_element  = null;
	protected CharArrayWriter _data_buf    = null;

	protected String _parserClass = "com.ibm.xml.parser.SAXDriver";
	protected org.xml.sax.Parser _parser = null;
	

	/**
	 * Create a XmlParser object that can parse XML documents specified
	 * by the Registry of XML decoders.
	 *
	 * @param  decoders  the Registry of bound XML document decoders
	 */
	public XmlParser (Registry decoders) {
		_decoders = decoders;
	}


	/**
	 * Gets the parse class name.
	 *
	 * @return the parser class name
	 * @see #setParserClassName
	 */
	public String getParserClassName () {
		return (_parserClass);
	}


	/**
	 * Sets the parser class name. The class name must specify a valid
	 * SAX parser.
	 *
	 * @param  parserClass the name of the parser class
	 */
	public void setParserClassName (String parserClass) {
		_parserClass = parserClass;
		_parser = null;
	}


	/**
	 * Gets a Parser for this object.
	 *
	 * @return  the SAX parser encapsulated by this object
	 */
	protected Parser getParser () {

		if (_parser != null) {
			return (_parser);
		}

		try {
			_parser = ParserFactory.makeParser(
					getParserClassName());
		} catch (Exception e) {
		}

		return (_parser);
	}


	/**
	 * Parses the XML data from the InputStream and returns a
	 * XmlfObject for the data.
	 *
	 * @param  xml_stream  the input stream containing the XML data to parse
	 * @return  a XmlfObject encapsulating the XML data parsed
	 */
	XmlfObject parse (InputStream xml_stream) {
	
		Parser parser = getParser();
		if (parser == null) {
			return (null);
		}


		_current_element = null;
		_root_element    = null;
		_data_buf        = null;
		parser.setDocumentHandler(this);

		try {
			parser.parse(new InputSource(xml_stream));
		} catch (Exception e) {
			_root_element = null;
		}

		if (_root_element == null) {
			return (null);
		}

		try {	
			DocumentDecoder decoder = (DocumentDecoder)
				_decoders.lookup(_root_element.getName());
			return (decoder.createObject(_root_element));

		} catch (NotBoundException nbe) {
			return (null);
		}
	}


	/**
	 * Parses the XML data and returns an XmlfObject for the data.
	 *
	 * @param  xml_data  the XML data to parse
	 * @return  a XmlfObject encapsulating the XML data parsed
	 */
	public XmlfObject parse (String xml_data) {

		InputStream xml_stream = new ByteArrayInputStream(
			xml_data.getBytes());
		return (parse(xml_stream));
	}


	/**
	 * This method is called when a start element is found by the parser.
	 */
	public void startElement (String name, org.xml.sax.AttributeList attrs)
		throws SAXException {

		AttributeList attributes;
		if (attrs.getLength() < 1) {
			attributes = null;
		} else {
			attributes = new AttributeList();
			for (int i = 0; i < attrs.getLength(); i++) {
				String attr_name = attrs.getName(i);
	     			String attr_value = attrs.getValue(i);
				attributes.add(attr_name, attr_value);
			}
		}

		_current_element = new XmlElement(
			_current_element, name, null, attributes);

		if (_root_element == null)
			_root_element = _current_element;
	}



	/**
	 * This method is called when character data is found by the parser.
	 */
	public void characters(char ch[], int start, int length)
		throws SAXException {

		if (_data_buf == null) {
			_data_buf = new CharArrayWriter(length);
		}

		_data_buf.write(ch, start, length);
	}


	/**
	 * This method is called when an end element is found by the parser.
	 */
	public void endElement(String name) throws SAXException {

		if (_data_buf != null) {
			_current_element.setData(_data_buf.toCharArray());
			_data_buf = null;
		}

		_current_element = _current_element.getParent();
	}
}


