package com.vsi.xmlf;

import java.util.Vector;

/**
 * This class encapsulates an XML element parsed by the XmlParser class.
 *
 */
public
class XmlElement {

	protected String _name   = null;
	protected char [] _data  = null;
	protected AttributeList _attributes = null;
	
	protected XmlElement _parent = null;
	protected Vector _sub_elements = new Vector();


	/**
	 * Construct an XmlElement with no name or data assigned to it.
	 */
	public XmlElement () {
	}


	/**
	 * This constructor creates a new XmlElement with the specified
	 * parent, name, data, and attributes.
	 *
	 * @param  parent  the parent XmlElement object (null if this
	 *                 is the root element
	 * @param  name    the name for the element
	 * @param  data    the data for the element
	 * @param  attributes  the attribute list for the element
	 */
	public XmlElement (XmlElement parent, String name, char data[],
		AttributeList attributes) {

		_parent = parent;
		_name   = name;
		_data   = data;
		_attributes = attributes;

		if (parent != null) {
			parent.addSubElement(this);
		}
	}


	/**
	 * Gets the number of children elements for this element.
	 *
	 * @return  the number of children (sub) elements
	 */
	public int getNumSubElements () {
		return (_sub_elements.size());
	}


	/**
	 * Gets an array of sub elements which are the children elements
	 * of this element.
	 *
	 * @return  an array containing the children elements
	 */
	public XmlElement [] getSubElements () {

		int num_elements = getNumSubElements();
		
		XmlElement sub_elements[] = new XmlElement[num_elements];
		_sub_elements.copyInto(sub_elements);

		return (sub_elements);
	}


	/**
	 * Gets the parent of this element.
	 *
	 * @return  the parent element
	 */
	public XmlElement getParent () {

		return (_parent);
	}


	/**
	 * Gets the name of this element.
	 *
	 * @return  the name of this element
	 */
	public String getName () {
		return (_name);
	}
		

	/**
	 * Gets the data contained by this element.
	 *
	 * @return  the data contained by this element
	 * @see     #getUntrimmedData
	 */
	public String getData () {

		if (_data != null) {
			return (new String(_data).trim());
		} else {
			return (null);
		}
	}


	/**
	 * Gets the data (untrimmed) contained by this element.
	 *
	 * @return  the untrimmed string data contained by this element
	 * @see     #getData
	 */
	public String getUntrimmedData () {

		if (_data != null) {
			return (new String(_data));
		} else {
			return (null);
		}
	}


	/**
	 * Gets the value of the specified attribute name.
	 *
	 * @param  attr_name  the name of the attribute to get the value for
	 * @return  the value of the attribute
	 */
	public String getAttribute (String attr_name) {

		if (_attributes != null) {
			return (_attributes.getValue(attr_name));
		} else {
			return (null);
		}
	}


	/**
	 * Sets the name of this element.
	 *
	 * @param  name  the new name for this element
	 */
	public void setName (String name) {
		_name = name;
	}


	/**
	 * Sets the data for this element.
	 *
	 * @param  data  the new data to be contained by this element
	 */
	public void setData (char data[]) {
		_data = data;
	}


	/**
	 * Sets the attributes list for this element.
	 *
	 * @param  attributes  the new attrbutes list for this element
	 */
	public void setAttributes (AttributeList attributes) {
		_attributes = attributes;
	}


	/**
	 * Sets the parent element of this element.
	 *
	 * @param  parent  the new parent element
	 */
	public void setParent (XmlElement parent) {
		_parent = parent;
	}


	/**
	 * Adds the element as a child of this element.
	 *
	 * @param  sub_element  the element to add 
	 */
	public void addSubElement (XmlElement sub_element) {
		_sub_elements.addElement(sub_element);
	}


	/**
	 * Gets the child element specified by name.
	 *
	 * @param  name  the name of the child element to get
	 * @return  the child element
	 */
	public XmlElement findElement (String name) {

		if (name == null) {
			return (null);
		}

		XmlElement xml[] = getSubElements();
		int num_elements = xml.length;
		for (int i = 0; i < num_elements; i++) {
			if (name.equalsIgnoreCase(xml[i].getName())) {
				return (xml[i]);
			}
		}

		return (null);
	}



	/**
	 * Gets the child element specified by name from the array
	 * of elements.
	 *
	 * @param  xml  the array of elements
	 * @param  name  the element name to get
	 * @return  the XmlElement object for name
	 */
	public static XmlElement findElement (XmlElement xml[], String name) {

		if (xml == null || name == null) {
			return (null);
		}

		int num_elements = xml.length;
		for (int i = 0; i < num_elements; i++) {
			if (name.equalsIgnoreCase(xml[i].getName())) {
				return (xml[i]);
			}
		}

		return (null);
	}
	

	/**
	 * Gets the data contained by element specified by name in the
	 * element array.
	 *
	 * @param   xml  the element array
	 * @param   name  the element name to get the data from
	 * @return  the data contained by this element
	 * @see     #getUntrimmedData(XmlElement xml[], String name)
	 */
	public static String getData (XmlElement xml[], String name) {

		XmlElement found = findElement(xml, name);
		if (found != null) {
			return (found.getData());
		} else {
			return (null);
		}
	}


	/**
	 * Gets the data (untrimmed) contained by the specified element
	 * in the element array.
	 *
	 * @param   xml  the element array
	 * @param   name  the element name to get the data from
	 * @return  the untrimmed string data contained by this element
	 * @see     #getData(XmlElement xml[])
	 */
	public static String getUntrimmedData (XmlElement xml[], String name) {

		XmlElement found = findElement(xml, name);
		if (found != null) {
			return (found.getUntrimmedData());
		} else {
			return (null);
		}
	}
			

	/**
	 * Gets the value of the specified attribute from the specified
	 * element name in the element array.
	 *
	 * @param  xml  the array of elements
	 * @param  name the name of the element to get the attribute from
	 * @param attr_name  the name of the attribute to get the value for
	 */
	public static String getAttribute (XmlElement xml[], 
		String name, String attr_name) {

		XmlElement found = findElement(xml, name);
		if (found != null) {
			return (found.getAttribute(attr_name));
		} else {
			return (null);
		}
	}
}


