package com.vsi.xmlf;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Vector;

/**
 * This class defines various utilitiy methods that are used throughout
 * the <em>com.vsi.xmlf</em> package.
 *
 */
public
class Utilities {


	/**
	 * This class is not meant to be instantiated.
	 */
	private Utilities () {
	}


	/**
	 * Native line separator ("\n" on Unix).
	 */
	protected static String _line_separator = 
		System.getProperty("line.separator");


	/**
	 * Get the native line separator ("\n" on Unix).
	 */
	public static String getLineSeparator () {
		return (_line_separator);
	}

	
	/**
	 * Returns a date-time string formatted in accordance with RFC 822
	 * for the current time.
	 */
	public static String getFormattedDate () {
		return (getFormattedDate(null));
	}


	/**
	 * Returns a date-time string formatted in accordance with RFC 822.
	 *
	 * @param  datetime  The <code>Date</code> object to format.
	 * the string for. Use null for the current date/time.
	 */
	public static String getFormattedDate (java.util.Date datetime) {
		
		java.text.SimpleDateFormat df = 
			new java.text.SimpleDateFormat(
			"EEE, dd MMM yyyy HH:mm:ss z", java.util.Locale.US);
		
		if (datetime == null) {
			datetime = new java.util.Date();
		}

		return (df.format(datetime));
	}


	/**
	 * Remove the beginning and trailing quotes from a string.
	 *
	 * @param  in  the string to remove the quotes from.
	 * @returns  The de-quotified string. If the string is malformed 
	 * (begins with but does not end with a quote) null is returned.
	 */
	public static String removeQuotes (String in) {

		if (in == null) {
			return (null);
		}


		/*
		 * Parse off quotes
		 */
		in = in.trim();
		int index = in.indexOf("\"");
		if (index == 0) {
			index = in.indexOf("\"", 1);
			if (index < 0) {
				return (null);
			}

			in = in.substring(1, index);
		}

		return (in);
	}


	/**
	 * Create a unique file in the specified directory based on
	 * <code>name</code>.
	 *
	 * @param  directory  the directory for the unique file
	 * @param  name       the original filename for the unique file.
	 * @exception IOException when an exception occurs.
	 */
	public static final synchronized String createUniqueFile (
		String directory, String name) throws IOException {

		File f = getUniqueFile(directory, name, 0);
		FileOutputStream fos = new FileOutputStream(f);
		fos.close();
		fos = null;
		return (f.getPath());
	}


	/**
	 * Generate a unique filename in the specified directory.
	 *
	 * @param  directory  the directory for the unique file
	 * @param  name       the original filename for the unique file.
	 * @exception IOException when an exception occurs.
	 */
	public static final String getUniqueFilename (
		String directory, String name) {

		File f = getUniqueFile(directory, name, 0);
		return (f.getPath());
	}


	/**
	 * Generate a <code>File</code> object that points to a unique
	 * filename in the specified directory.
	 *
	 * @param  directory  the directory for the unique file
	 * @param  name       the original filename for the unique file.
	 */
	public static final File getUniqueFile (
		String directory, String name) {
	
		return (getUniqueFile(directory, name, 0));
	}


	/**
	 * This method does the actual work of generating the unique filename.
	 */
	private static File getUniqueFile (
		String directory, String name, int times) {

		if (times == 0) {
			File f = new File(directory, name);
			if (!f.exists())
				return (f);
		}

		times++;

		String tmpname;
		int index = name.lastIndexOf(".");
		if (index >= 0) {
			tmpname = name.substring(0, index) + 
						times + name.substring(index);
		} else {
			tmpname = name + times;
		}
				
		File f = new File(directory, tmpname);
		if (!f.exists())
			return (f);
			
		return (getUniqueFile(directory, name, times));	
	}


	/**
	 * Splits up a delimited string into an array of strings.
	 *
	 * @param  list  The delimited string to split up.
	 * @param  delimiter  The delimiter to split the string up on.
	 */
	public static String [] splitDelimitedString (
		String list, String delimiter) {

		Vector fields = new Vector();

		list = list.trim();
		if (delimiter == null) {
			delimiter = ";";
		}

		int first_delim = 0;
		int next_delim = list.indexOf(delimiter, 1);
		while (next_delim >= 0) {
			fields.addElement(
				list.substring(first_delim, next_delim));
			first_delim = next_delim + 1;
			next_delim = list.indexOf(delimiter, first_delim);
		}

		/*
		 * Add the last (or only) field
		 */
		fields.addElement(list.substring(first_delim));

		/*
		 * Convert the Vector into String []
		 */
		String strings[] = new String[fields.size()];
		fields.copyInto(strings);
		return (strings);
	}


	/**
	 * Gets the host name for the local machine.
	 *
	 * @return A string with the local hostname or null on error.
	 */
	public static String getLocalHostName () {

		java.net.InetAddress addr = null;
		try {
			addr = java.net.InetAddress.getLocalHost();
		} catch (Exception ignore) {
			return (null);
		}

		return (addr.getHostName());
	}
}
