/*------------------------------------------------------------------------
 * Program to list the status of a fax request
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"WU:H:Z:?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] [reqid]\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
	fprintf(fp, "reqid           Request-id to query\n");
}

/*------------------------------------------------------------------------
 * extract a name from a tag list
 */
static const char *
tag_name (V_TAG *tags, const char *name)
{
	const char *p = VSI_Tag_Get_Data_By_Name(tags, name, 0, 0);

	if (p == 0 || *p == 0)
		return ("<none>");
	return (p);
}

/*------------------------------------------------------------------------
 * display server state
 */
static void
display_state (V_TAG *tags)
{
	const char *hostname	= tag_name(tags, V_TAG_SRVI_HOSTNAME);
	const char *startup	= tag_name(tags, V_TAG_SRVI_STARTUP);
	const char *schedtime	= tag_name(tags, V_TAG_SRVI_SCHEDTIME);
	const char *state	= tag_name(tags, V_TAG_SRVI_STATE);
	const char *version	= tag_name(tags, V_TAG_SRVI_VERSION);
	const char *build	= tag_name(tags, V_TAG_SRVI_BUILD);
	const char *bldinfo	= tag_name(tags, V_TAG_SRVI_BLDINFO);
	const char *runinfo	= tag_name(tags, V_TAG_SRVI_RUNINFO);
	const char *serial	= tag_name(tags, V_TAG_SRVI_SERIAL);
	const char *licmode	= tag_name(tags, V_TAG_SRVI_LICMODE);
	const char *numusers	= tag_name(tags, V_TAG_SRVI_NUMUSERS);
	const char *numports	= tag_name(tags, V_TAG_SRVI_NUMPORTS);
	const char *srvexpdt	= tag_name(tags, V_TAG_SRVI_SRVEXPDT);

	printf("hostname        = %s\n", hostname);
	printf("start time      = %s\n", startup);
	printf("server time     = %s\n", schedtime);
	printf("state           = %s\n", state);
	printf("version         = %s\n", version);
	printf("build           = %s\n", build);
	printf("build info      = %s\n", bldinfo);
	printf("run   info      = %s\n", runinfo);
	printf("serial number   = %s\n", serial);
	printf("license mode    = %s\n", licmode);
	printf("number of users = %s\n", numusers);
	printf("number of ports = %s\n", numports);
	printf("server exp date = %s\n", srvexpdt);
}

/*------------------------------------------------------------------------
 * display a status entry
 */
static void
display_status (V_TAG *tags)
{
	const char *seqno	= tag_name(tags, V_TAG_OLOG_SEQNO);
	const char *reqtype	= tag_name(tags, V_TAG_OLOG_REQTYPE);
	const char *gseqno	= tag_name(tags, V_TAG_OLOG_GSEQNO);
	const char *memno	= tag_name(tags, V_TAG_OLOG_MEMNO);
	const char *clientid	= tag_name(tags, V_TAG_OLOG_CLIENTID);
	const char *mailaddr	= tag_name(tags, V_TAG_OLOG_MAILADDR);
	const char *subtime	= tag_name(tags, V_TAG_OLOG_SUBMITTIME);
	const char *queue	= tag_name(tags, V_TAG_OLOG_QUEUE);
	const char *active 	= tag_name(tags, V_TAG_OLOG_ACTIVE);
	const char *rresult	= tag_name(tags, V_TAG_OLOG_RRESSTR);

	const char *toname	= tag_name(tags, V_TAG_OLOG_TONAME);
	const char *tocomp	= tag_name(tags, V_TAG_OLOG_TOCOMP);
	const char *tofnum	= tag_name(tags, V_TAG_OLOG_TOFNUM);
	const char *tovnum	= tag_name(tags, V_TAG_OLOG_TOVNUM);

	const char *frname	= tag_name(tags, V_TAG_OLOG_FRNAME);
	const char *frcomp	= tag_name(tags, V_TAG_OLOG_FRCOMP);
	const char *frfnum	= tag_name(tags, V_TAG_OLOG_FRFNUM);
	const char *frvnum	= tag_name(tags, V_TAG_OLOG_FRVNUM);

	const char *subject	= tag_name(tags, V_TAG_OLOG_SUBJECT);
	const char *numpgs	= tag_name(tags, V_TAG_OLOG_NUMPGS);

	int   is_grp		= 0;

	/*----------------------------------------------------------------
	 * Rules concerning record type:
	 *
	 *	"reqtype"	"memno"		entry-type
	 *	---------	-------		----------
	 *	reg		N/A		regular fax job
	 *	group		0		group master record
	 *	group		1-n		group member record
	 */
	if (strcmp(reqtype, "group") == 0 && strcmp(memno, "0") == 0)
	{
		is_grp = 1;
		printf("Status for group fax job %s",
			gseqno);
	}
	else
	{
		printf("Status for fax job %s",
			seqno);
		if (strcmp(reqtype, "group") == 0)
		{
			printf(" (member %s of group %s)",
				memno, gseqno);
		}
	}
	printf("\n");

	printf("\n");

	printf("Submitted by: %-20.20s  E-mail      : %s\n",
		clientid, mailaddr);

	printf("Submit time : %-20.20s  Queue       : %s\n",
		subtime, queue);

	printf("Status      : %-20.20s",
		active);
	if (! is_grp)
		printf("  Result      : %s", rresult);
	printf("\n");

	printf("\n");

	printf("To name     : %-20.20s  From name   : %s\n",
		toname, frname);

	printf("To company  : %-20.20s  From company: %s\n",
		tocomp, frcomp);

	printf("To fax #    : %-20.20s  From fax #  : %s\n",
		tofnum, frfnum);

	printf("To voice #  : %-20.20s  From voice #: %s\n",
		tovnum, frvnum);

	printf("\n");

	printf("Subject     : %s\n",
		subject);

	printf("Num pages   : %s\n",
		numpgs);

	if (is_grp)
	{
		const char *num_mems = tag_name(tags, V_TAG_OLOG_NUMMEMS);
		const char *num_pend = tag_name(tags, V_TAG_OLOG_NUMQUEUED);
		const char *num_sent = tag_name(tags, V_TAG_OLOG_NUMSENT);
		const char *num_fail = tag_name(tags, V_TAG_OLOG_NUMFAILED);

		printf("\n");
		printf("Members  Pending  Normal  Failed\n");

		printf("%5s    %5s    %5s   %5s\n",
			num_mems,
			num_pend,
			num_sent,
			num_fail);
	}
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	const char *	reqid		= 0;
	V_TAG *		tag_list	= 0;
	const char *	pgmname		= argv[0];
	int		opt_index;
	int		c;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * already processed
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * get reqid argument
	 */
	opt_index = VSI_Opt_Index();
	if (argc > opt_index)
	{
		reqid = argv[opt_index++];
	}

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * get list of status tags or server state
	 */
	if (reqid == 0)
		tag_list = VSI_Server_Info(vptr, msgbuf);
	else
		tag_list = VSI_Server_Status(vptr, reqid, msgbuf);

	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	/*----------------------------------------------------------------
	 * check if any errors
	 */
	if (*msgbuf != 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}
	/*----------------------------------------------------------------
	 * display status info or state
	 */
	if (reqid == 0)
		display_state(tag_list);
	else
		display_status(tag_list);

	/*----------------------------------------------------------------
	 * free the tag list
	 */
	tag_list = VSI_Tag_Free(tag_list, 0);

	return (EXIT_SUCCESS);
}
