/*
 *
 *
 *	Description:
 *
 *	This program simply sends an XMLF file to a xml server
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "TransportManager.h"
#include "VxmldTransport.h"
#include "Base64Codec.h"
#include "TmpStream.h"
#include "CmdLine.h"

#define XMLF_VERSION	"1.0"

static void
usage (const char *progname, int flag)
{
	if (flag)
	{
	cout << "Usage: " << progname << 
			" [options] [inp-file] [embed files]" << endl;
	cout << "options:" << endl;
	cout << "  -o out-file  output filename (default=stdout)" << endl;
	cout << "  -t format    response format (html, text, or xml)" << endl;
	cout << "  -h url       server URL" << endl;
#if 0
	cout << "  -v           verify input file (not implemented)" << endl;
#endif
	}
	else
	{
		cerr << "Type \"" << progname << " -help\" for help" << endl;
	}
}


/*------------------------------------------------------------------------
 * This function determines the document type contained in the passed in
 * XML string data. It lower cases the document type and returns that string.
 */
static char *
get_doc_type (char * buf)
{
	static char doc_type_buf[32];
	if (*doc_type_buf)
		return (doc_type_buf);

	/*
	 * All document types start with "<fax-" or "<FAX-"
	 */
	char * p = strstr(buf, "<fax-");
	if (p == NULL)
	{
		p = strstr(buf, "<FAX-");
		if (p == NULL)
			return (NULL);
	}

	/*
	 * figure out the end of the start tag name
	 */
	char * p1 = strchr(p, ' ');
	if (p1 == NULL)
		p1 = strchr(p, '>');
	if (p1 == NULL)
		p1 = strchr(p, '\0');
	if (p1 == NULL || (unsigned int)(p1 - p) >= sizeof(doc_type_buf))
		return (NULL);

	/*
	 * now copy the document type (lower case)
	 */
	char * doc_type = doc_type_buf;
	for (p++; p < p1; p++, doc_type++)
		*doc_type = tolower(*p);

	*doc_type = '\0';
	return (doc_type_buf);
}


/*------------------------------------------------------------------------
 * This function is given XML data from the start of the document to the
 * end of the root tag, replaces the response format and then outputs
 * the new XML data to the specified stream.
 */
static void
output_doc_type(char *bufp, const char *rsp_fmt, fstream *outin)	
{
#define RF_ATTR "response-format"

	/*
	 * if no response format was specified output the buffer and return
	 */
	if (rsp_fmt == NULL)
	{
		outin->write(bufp, strlen(bufp));
		outin->put('\n');
		return;
	}


	/*
	 * if we have a response-format attribute already, replace it,
	 * output the rest of the data and return.
	 */
	char * p = strstr(bufp, "response-format=");
	if (p == NULL)
	{
		p = strstr(bufp, "RESPONSE-FORMAT=");
	}

	if (p != NULL)
	{

		*p = '\0';

		/*
		 * output the new response format
		 */		
		outin->write(bufp, strlen(bufp));
		outin->write(RF_ATTR, strlen(RF_ATTR));
		outin->put('=');
		outin->put('"');
		outin->write(rsp_fmt, strlen(rsp_fmt));
		outin->put('"');

		/*
		 * skip over the response-format attribute name
		 */
		p += strlen(RF_ATTR) + 1;
		if (*p == '"' || *p == '\'')
			p++;

		/*
		 * locate the end of the response format attribute value
		 */
		while (*p != '\0' && *p != ' ' && *p != '\n' && *p != '>')
		{
			if (*p == '"' || *p == '\'')
			{
				p++;
				break;
			}

			p++;
		}

		if (*p == '\r')
			p++;


		/*
		 * output any remaining data
		 */
		if (*p != '\0')
		{
			outin->write(p, strlen(p));
			outin->put('\n');	
		}

		return;
	}


	/*
	 * now figure out the end of the element, output the element and
	 * add the response format
	 */
	p = strrchr(bufp, '>');
	if (p != NULL)
		*p = '\0';	
	outin->write(bufp, strlen(bufp));
	outin->put(' ');
	outin->write(RF_ATTR, strlen(RF_ATTR));
	outin->put('=');
	outin->put('"');
	outin->write(rsp_fmt, strlen(rsp_fmt));
	outin->put('"');

	/*
	 * output the closing bracket if we had one
	 */
	if (p != NULL)
	{
		outin->put('>');

		if (p[1] != '\0')
		{
			p++;
			outin->write(p, strlen(p));
		}
	}

	outin->put('\n');
}


/*------------------------------------------------------------------------
 * Base64 encode and embed the passed in files
 */
static void
embed_files (ifstream ** files, char ** filenames, fstream * out)
{
	int i;
	char buf[1024];
	for (i = 0; files[i] != NULL; i++)
	{
		sprintf(buf, "<body %s filename=\"%s\">\n", 
			"content-transfer-encoding=\"base64\"", filenames[i]);
		out->write(buf, strlen(buf));

		Base64Codec::encode(files[i], out);

		sprintf(buf, "\n</body>\n");
		out->write(buf, strlen(buf));
	}
}


/*------------------------------------------------------------------------
 * This function will embed the response-format and body files for
 * a "fax-submit" document.
 */
static ifstream *
embed_stuff (const char * in_filename, ifstream * in,
		const char * rsp_fmt, ifstream ** files, char ** filenames)
{
	/*
	 * anything to embed?
	 */
	if (rsp_fmt == NULL && files == NULL)
		return (in);

	/*
	 * don't allow embeding when input is stdin
	 */	
	if (strcmp("stdin", in_filename) == 0 && (rsp_fmt || files))
	{
		cerr <<
			"cannot embed files or set response format on stdin" <<
			endl;

		delete in;
		return (NULL);
	}

	
	/*
	 * create a temporary stream to output the new XML document to
	 */
	TmpStream * outin = new TmpStream();

	char buf[8192];
	int buf_len = sizeof(buf) - 1;
	*buf = '\0';

#if _WIN32
	if (!outin->is_open() || outin->bad())
#else
	if (outin->bad())
#endif
	{
		cerr <<
			"cannot create temporary file" <<
			endl;

		delete in;
		return (NULL);
	}

	ostrstream doc_type_buf;
	char * doc_type = NULL;
	int found_doc_type = 0;
	
	/*
	 * read the input XML file and embed the specified response-format
	 * and/or files.
	 */
	while (!in->eof())
	{
		in->getline(buf, buf_len);	
		if (in->bad())
			continue;

		int len = strlen(buf);		
		if (len > 0)
		{
			if (buf[len - 1] == '\r')
			{
				len--;
				buf[len] = '\0';
			}
		}


		/*
		 * are we still looking for the document type?
		 */
		if (doc_type == NULL)
		{
			doc_type_buf << buf;

			if (!found_doc_type && (strstr(buf, "<fax-") != NULL ||
				strstr(buf, "<FAX-") != NULL))
			{
					found_doc_type = 1;
			} 

			if (found_doc_type && strchr(buf, '>') != NULL)
			{
				doc_type_buf << ends;
				char * p = doc_type_buf.str();
				doc_type = get_doc_type(p);

				/*
				 * Make sure that the doc type is fax-submit
				 * if we have files to embed, exit if not
				 */
				if (files != NULL &&
					strcmp("fax-submit", doc_type))
				{

					cerr << "can only embed files if "
						<< "input document is a "
						<< "fax-submit document."
						<< endl;
					delete outin;
					return (NULL);
				}

				output_doc_type(p, rsp_fmt, outin);
			}
			else
			{
				doc_type_buf << "\n";
			}


			continue;
		}


		/*
		 * see if this is the </content> end tag, if so embed the files
		 */
		if (files != NULL && doc_type != NULL)
		{
			char * p = strstr(buf, "</content");
			if (p == NULL)
				p = strstr(buf, "</content");

			if (p != NULL)
			{
				if (p != buf)
				{
					*p = '\0';
					outin->write(buf, strlen(buf));
				}

				embed_files(files, filenames, outin);

				*p = '<';
				outin->write(p, strlen(p));
				outin->put('\n');

				/*
				 * set files to NULL which indicates no more
				 * files to embed
				 */
				files = NULL;
				continue;
			}
		}

		outin->write(buf, len);
		outin->put('\n');
	}


	/*
	 * no longer need the input file
	 */
	delete in;

	/*
	 * close the temporary stream, then reopen it for input
	 */
	outin->close();
	outin->open(outin->getFilename(), ios::in);

	return ((ifstream *)outin);
}	


int
main (int argc, char ** argv)
{
	ifstream ** embed_files = NULL;
	char ** embed_filenames = NULL;
	char *  progname = argv[0];
	char *  url_spec = (char *)"localhost";
	char *  inp_file = (char *)"-";
	char *  out_file = (char *)"-";
	char *  rsp_fmt  = NULL;
	int     verify   = 0;
	int     c;
	int     i;

	/*----------------------------------------------------------------
	 * collect arguments
	 */
	progname = argv[0];

	if (argc > 1)
	{
		if (strcmp(argv[1], "-?"   )  == 0 ||
		    strcmp(argv[1], "-help")  == 0 ||
		    strcmp(argv[1], "--help") == 0 )
		{
			usage(progname, 1);
			return (0);
		}

		if (strcmp(argv[1], "-??"     )  == 0 ||
		    strcmp(argv[1], "-helpall")  == 0 ||
		    strcmp(argv[1], "--helpall") == 0 )
		{
			usage(progname, 2);
			return (0);
		}

		if (strcmp(argv[1], "-V"     )   == 0 ||
		    strcmp(argv[1], "-version")  == 0 ||
		    strcmp(argv[1], "--version") == 0 )
		{
			cout << progname << ": version " << XMLF_VERSION <<endl;
			return (0);
		}
	}
	else
	{
		usage(progname, 0);
		return (1);
	}

	CmdLine options("o:t:h:v");
	while ((c = options.getOpt(argc, argv)) != EOF)
	{

		switch (c)
		{
		  case 'o':
			out_file = options.getOptArg();
			break;

		  case 'v':
			verify = 1;
			break;

		  case 'h':
			url_spec = options.getOptArg();
			break;
			
		  case 't':
			rsp_fmt = options.getOptArg();
			if (strcmp("xml", rsp_fmt) == 0)
			{
				/* valid response format */
			}
			else if (strcmp("text", rsp_fmt) == 0)
			{
				/* valid response format */
			}
			else if (strcmp("html", rsp_fmt) == 0)
			{
				/* valid response format */
			}
			else
			{
				cerr << "invalid response format: " <<
					"\"" << rsp_fmt << "\"" << endl;
				return (1);
			}

			break;
			
			
		  default:
			cerr << progname << ": " << 
				options.getErrorMsg() << endl;	
			usage(progname, 0);
			return (1);
		}
	}
	
	i = options.getOptIndex();

	/*
	 * if next parameter is not NULL then it is the XMLF file to send
	 */
	if (i < argc)
		inp_file = argv[i++];


	/*
	 * remaining parameters are files to embed
	 */
	if (i < argc)
	{
		int num_files = argc - i;
		embed_filenames = new char *[num_files + 1];
		embed_filenames[num_files] = NULL;
		embed_files = new ifstream *[num_files + 1];
		embed_files[num_files] = NULL;
		for (num_files = 0; i < argc; i++, num_files++)
		{
			embed_filenames[num_files] = argv[i];
#if _WIN32
			embed_files[num_files] = new ifstream(argv[i],
				ios::in | ios::binary);
			if (embed_files[num_files]->bad() || 
				!embed_files[num_files]->is_open())
#else
			embed_files[num_files] = new ifstream(argv[i],
				ios::in);
			if (embed_files[num_files]->bad())
#endif
			{
				cerr << "couldn't open embed file: \"" << 
					argv[i] << "\" for input" << endl;
				return (1);
			}
		}
	}
	

	/*----------------------------------------------------------------
	 * Open the transport manager, bind the vxmld transport and
	 * set vxmld as the default transport
	 */
	TransportManager *	mgr		= new TransportManager();
	VxmldTransport *	transport	= new VxmldTransport();

	mgr->bindTransport((char *)"vxmld", transport);
	mgr->setDefaultTransport((char *)"vxmld");

	/*----------------------------------------------------------------
	 * Open the XML file to send
	 */
	ifstream * xml_stream;

	if (strcmp(inp_file, "-") == 0)
	{
		xml_stream = new ifstream(0, ios::in);
		inp_file = (char *)"stdin";
	}
	else
	{
#if _WIN32
		xml_stream = new ifstream(inp_file,
			ios::in | ios::binary);
#else
		xml_stream = new ifstream(inp_file,
			ios::in);
#endif
	}

#if _WIN32
	if (xml_stream->bad() || !xml_stream->is_open())
#else
	if (xml_stream->bad())
#endif
	{
		cerr << "couldn't open: " << inp_file << " for input" << endl;
		return (1);
	}


	/*----------------------------------------------------------------
	 * set the format and embed any specified files
	 */
	xml_stream = embed_stuff(inp_file, xml_stream, 
			rsp_fmt, embed_files, embed_filenames);

	if (embed_files)
	{
		for (i = 0; embed_files[i] != NULL; i++)
		{
			delete embed_files[i];
		}

		delete embed_filenames;
		delete embed_files;
	}

	if (xml_stream == NULL)
	{
		return (1);
	}

	
	/*----------------------------------------------------------------
	 * Now send the XML file to the fax server
	 */
	char * response = mgr->send(url_spec, xml_stream);
	delete xml_stream;


	/*----------------------------------------------------------------
	 * response will be NULL on error
	 */
	if (response == NULL)
	{
		cerr << mgr->getLastError() << endl;
		return (1);
	}

	/*----------------------------------------------------------------
	 * output response
	 */
	if (strcmp(out_file, "-") == 0)
	{
		cout << response << endl;
	}
	else
	{
		ofstream * out_stream;
#if _WIN32
		out_stream = new ofstream(out_file, ios::out | ios::binary);
		if ( out_stream->bad() || !out_stream->is_open() )
		{
#else
		out_stream = new ofstream(out_file, ios::out);
		if ( out_stream->bad() )
		{
#endif
			cerr << "couldn't open: " << inp_file << 
				" for output" << endl;
			return (1);
		}

		*out_stream << response;
		out_stream->close();
		delete out_stream;
	}

	return (0);
}
