// +++FHDR+++
//
//
//   Class Name : TcpSocket
//   Super Class: n/a
//
//   Description: This class implements a platform independent TCP socket.
//
//
//   Notes      : none
//
//
//   Method - Description
//   ------------------------------------------------------------------------
//
//
// ---FHDR---


#include "OutputString.h"
#include "TcpSocket.h"

#if _WIN32

#include <winsock.h>
#include <stdio.h>

#define H_ERRNO		h_errno

#else

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>

#include <string.h>
#include <strings.h>
#include <unistd.h>

#ifdef __cplusplus
}
#endif

#ifndef INVALID_SOCKET
#define INVALID_SOCKET	(-1)
#endif

#ifndef SOCKET_ERROR
#define SOCKET_ERROR	(-1)
#endif

#ifndef INADDR_NONE
#define INADDR_NONE	(-1)
#endif

#ifdef SOCKET
#  undef SOCKET
#endif
#define SOCKET		int

#define H_ERRNO		errno

#define closesocket	::close
#define ioctlsocket	ioctl

#endif // _WIN32 


int TcpSocket::_sockets_initialized = 0;



// +++PHDR+++
//
//  Method     : TcpSocket::TcpSocket()
//
//  Visibility : public
//
//  Description: Default constructor for the TcpSocket class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

TcpSocket::TcpSocket()
{
	_read_timeout = 60;
	_write_timeout = 10;
	_read_buf_size = 16384;
	_lowlevel_socket = -1;

	_host = NULL;
	_port = -1;
}


// +++PHDR+++
//
//  Method     : TcpSocket::TcpSocket()
//
//  Visibility : public
//
//  Description: Alternate constructor for the TcpSocket class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  host            char *                  host name for the socket connection
//  port            int                     port for the socket connection
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

TcpSocket::TcpSocket(char * host, int port)
{
	_read_timeout = 60;
	_write_timeout = 10;
	_read_buf_size = 16384;
	_lowlevel_socket = -1;
	_host = NULL;
	_port = -1;

	open(host, port);
}


// +++PHDR+++
//
//  Method     : TcpSocket::~TcpSocket()
//
//  Visibility : public
//
//  Description: Destructor for the TcpSocket class.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

TcpSocket::~TcpSocket()
{
	close();

	if (_host != NULL) {
		delete (_host);
		_host = NULL;
	}
}


// +++PHDR+++
//
//  Method     : TcpSocket::getHost()
//
//  Visibility : public
//
//  Description: Gets the current host name for the server.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   char * containing the host name of the server.
//
//
//  Notes:     none
//
// ---PHDR---

char * TcpSocket::getHost ()
{
	return (_host);
}


// +++PHDR+++
//
//  Method     : TcpSocket::getPort()
//
//  Visibility : public
//
//  Description: Gets the current port for the server.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   an int with the port number.
//
//
//  Notes:     none
//
// ---PHDR---

int TcpSocket::getPort ()
{
	return (_port);
}




// +++PHDR+++
//
//  Method     : TcpSocket::getLastError()
//
//  Visibility : public
//
//  Description: Returns the last error message from the socket.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   A char * containing the last error message generated.
//
//
//  Notes:     none
//
// ---PHDR---

char * TcpSocket::getLastError ()
{
	return (_err_msg);
}


// +++PHDR+++
//
//  Method     : TcpSocket::ok_to_write()
//
//  Visibility : public
//
//  Description: Checks to see if it is ok to write on the socket.
// 
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   1 on success, 0 on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int TcpSocket::ok_to_write ()
{
	int rc;
	struct timeval timeout;
	fd_set fdset;
	SOCKET s = (SOCKET)_lowlevel_socket;

	/*
	 * Check if OK to write
	 */
	FD_ZERO(&fdset);
	FD_SET(s, &fdset);

	timeout.tv_sec  = _write_timeout;
	timeout.tv_usec = 0;

	rc = select(s + 1, (fd_set *)NULL, &fdset, (fd_set *)NULL, &timeout);
	if (rc == -1) {
		_err_msg << "Select error on socket " << s << ends;
		return (0);
	}

	if (rc == 0) {
		_err_msg << "Select on socket " << s << " timed out" << ends;
		return (0);
	}

	return (1);
}


// +++PHDR+++
//
//  Method     : TcpSocket::write()
//
//  Visibility : public
//
//  Description: Sends data to the socket.
// 
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  buf             char *                  the data to write
//  len             int                     the length of the data to write
//
//
//  Returns:   1 on success, 0 on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int TcpSocket::write (char * buf, int len)
{
	SOCKET s = (SOCKET)_lowlevel_socket;

	/*
	 * Write the buffer to the socket
	 */
	int rc;
	while (len > 0) {
		rc = send(s, (char *)buf, len, 0);
		if (rc == -1) {
			break;
		}

		if (rc == len)
			break;
		buf += rc;
		len -= rc;
	}

	if (rc == -1) {
		_err_msg << "Send error on socket " << s << ends;
		return (0);
	}

	return (1);
}


// +++PHDR+++
//
//  Method     : TcpSocket::read()
//
//  Visibility : public
//
//  Description: Reads data from the socket and returns it as
//               a NULL terminated char *.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   A NULL terminated char * containing the data read from the
//             socket.
//
//
//  Notes:     none
//
// ---PHDR---

char * TcpSocket::read ()
{
	int rc;
	struct timeval timeout;
	fd_set fdset;
	SOCKET s = (SOCKET)_lowlevel_socket;


	/*
	 * Wait for data to be ready for reading
	 */
	FD_ZERO(&fdset);
	FD_SET(s, &fdset);

	timeout.tv_sec  = _read_timeout;
	timeout.tv_usec = 0;

	rc = select(s + 1, &fdset, (fd_set *)NULL,
		(fd_set *)NULL, &timeout);
	if (rc == -1) {
		_err_msg << "Select error on socket " << s << ends;
		return (NULL);
	}

	if (rc == 0) {
		_err_msg << "Select on socket " << s << 
			" timed out" << ends;
		return (NULL);
	}
	

	/*
	 * Now read the data from the socket
	 */
	char * buf = new char [_read_buf_size + 1];
	int buf_size = _read_buf_size;
	int buf_ptr = 0;
	char * read_buf = new char[_read_buf_size];
	int nread;
	do {
		nread = recv(s, read_buf, _read_buf_size - 1, 0);
		if (nread > 0) {
			if (buf_ptr + nread >= buf_size) {
				buf_size += _read_buf_size + 1;
				char * tmp = new char[buf_size];
				memcpy(tmp, buf, buf_ptr);
				delete buf;
				buf = tmp;
			}

			memcpy(buf + buf_ptr, read_buf, nread);
			buf_ptr += nread;
			buf[buf_ptr] = '\0';
		}
	} while (nread > 0);

	delete read_buf;

	if (nread == -1) {
		_err_msg << "Recv error on socket " << s << ends;
		delete buf;
		return (NULL);
	}

	return (buf);
}


// +++PHDR+++
//
//  Method     : TcpSocket::open()
//
//  Visibility : public
//
//  Description: Opens a TCP connection to the specified host and port.
//
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  to_addr         char *                  hostname of server to connect to
//  to_port         int                     port of server to connect to
//
//
//  Returns:   A SOCKET connected to the specified server, or INVALID_SOCKET
//             on failure.
//
//
//  Notes:     none
//
// ---PHDR---


int TcpSocket::open (char * to_addr, int to_port)
{
        u_long addr;
	struct sockaddr_in server;
	SOCKET s;
	int rc;
	int len;

	/*
	 * Initialize the OS'es socket layer
	 */
	if (!TcpSocket::initializeSockets()) {
		return (0);
	}

	/*
	 * Close the socket if already open
	 */
	close();

	/*
	 * Must have a to address
	 */
	if (to_addr == (const char *)NULL) {
		_err_msg << "no host address specified" << ends;
		return (0);
	}

	if (to_port < 1) {
		_err_msg << "invalid port specified" << ends;
		return (0);
	}

	/*
	 * Convert the address on failure try the address as a hostname
	 */
	addr = inet_addr((char *)to_addr);
	if (addr == INADDR_NONE) {
		struct hostent * hp = gethostbyname(to_addr);
		if (hp == NULL) {
			_err_msg << "invalid host IP: " << to_addr << ends;
			return (0);
		}

		struct in_addr inaddr;
		memcpy((char *)&inaddr, (char *)hp->h_addr, hp->h_length);
		char * ip = inet_ntoa(inaddr);
		addr = inet_addr((char *)ip);
		if (addr == INADDR_NONE) {
			_err_msg << "invalid host IP: " << to_addr << ends;
			return (0);
		}

	}


	/*
	 * Create the socket
	 */
	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s == INVALID_SOCKET) {
		_err_msg << "cannot open socket" << ends;
		return (0);
	}

	/*
	 * Connect the socket to the other end
	 */
	server.sin_family = AF_INET;
	server.sin_port   = htons((u_short)to_port);
	memcpy((char *)&server.sin_addr, (char *)&addr, sizeof(addr));

	len = sizeof(server);
	rc = connect(s, (struct sockaddr *)&server, len);
	if (rc < 0) {
		_err_msg << "problem connecting to: " << to_addr << 
			", error=" << H_ERRNO << ends;
		closesocket(s);
		return (0);
	}

	_lowlevel_socket = (long)s;	

	/*
	 * Save the host and port info
	 */
	_host = new char[strlen(to_addr) + 1];
	strcpy(_host, to_addr);
	_port = to_port;

	return (1);
}


// +++PHDR+++
//
//  Method     : TcpSocket::close()
//
//  Visibility : public
//
//  Description: Closes the low level socket if open.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void TcpSocket::close ()
{
	if (_lowlevel_socket != -1) {
		closesocket((SOCKET)_lowlevel_socket);
		_lowlevel_socket = -1;
	}
}


// +++PHDR+++
//
//  Method     : TcpSocket::initializeSockets()
//
//  Visibility : public (static)
//
//  Description: Initializes the TCP socket layer for the OS.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   1 on success, 0 on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int TcpSocket::initializeSockets ()
{
	if (_sockets_initialized) {
		return (1);
	}

#if _WIN32

	WORD	wVersionRequested = MAKEWORD(1,1);
	WSADATA	wsaData;
	if ( WSAStartup(wVersionRequested, &wsaData) != 0 ) {
		return (0);
	}

	if ( LOBYTE(wsaData.wVersion) != 1 || HIBYTE(wsaData.wVersion) != 1) {
		WSACleanup();
		return (0);
	}

#endif
	_sockets_initialized = 1;
	return (1);
}


// +++PHDR+++
//
//  Method     : TcpSocket::deinitializeSockets()
//
//  Visibility : public (static)
//
//  Description: Uninitializes the socket layer for the OS.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  none
//
//
//  Returns:   n/a
//
//
//  Notes:     none
//
// ---PHDR---

void TcpSocket::deinitializeSockets()
{
	if (!_sockets_initialized) {
		return;
	}

#if _WIN32
	WSACleanup();
#endif

	_sockets_initialized = 0;
}


