// +++FHDR+++
//
//
//   Class Name : Base64Codec
//   Super Class: n/a
//
//   Description:
//
//	Decode/Encode a base64 file
//
//
//   Method - Description
//   ------------------------------------------------------------------------
//
//
//   Notes      :
//
//	The basic base64 algorithm is as follows:
//
//	Bits in a byte are referred to in the following as follows:
//
//		       +----+----+----+----+ +----+----+----+----+
//		byte = | b1 | b2 | b3 | b4 | | b5 | b6 | b7 | b8 |
//		       +----+----+----+----+ +----+----+----+----+
//
//	output:	Take 3 bytes (a, b, c) and convert to 4 bytes (w, x, y, z)
//		as follows:
//
//		Translation is to take a value in the range [0-64]
//		and use it as the index into the string "A-Za-z0-9+/".
//
//		a & b & c present:
//
//			w = 0 0 a1 a2 a3 a4 a5 a6
//			x = 0 0 a7 a8 b1 b2 b3 b4
//			y = 0 0 b5 b6 b7 b8 c1 c2
//			z = 0 0 c3 c4 c5 c6 c7 c8
//
//			w = OXLATE (w)
//			x = OXLATE (x)
//			y = OXLATE (y)
//			z = OXLATE (z)
//
//		a & b present:
//
//			w = 0  0 a1 a2 a3 a4 a5 a6
//			x = 0  0 a7 a8 b1 b2 b3 b4
//			y = 0  0 b5 b6 b7 b8  0  0
//
//			w = OXLATE (w)
//			x = OXLATE (x)
//			y = OXLATE (y)
//			z = '='
//
//		a present:
//
//			w = 0 0 a1 a2 a3 a4 a5 a6
//			x = 0 0 a7 a8  0  0  0  0
//
//			w = OXLATE (w)
//			x = OXLATE (x)
//			y = '='
//			z = '='
//
//	input:	Take 4 bytes (w, x, y, z) and convert them into 3 bytes
//		(a, b, c) as follows:
//
//		Translation is to convert a character from the string
//		"A-Za-z0-9+/" into its index its entry in the string,
//		which will be a value in the range [0-64].
//
//		w, x, y, z:
//
//			w = IXLATE (w)
//			x = IXLATE (x)
//			y = IXLATE (y)
//			z = IXLATE (z)
//
//			a = w3 w4 w5 w6 w7 w8 x3 x4
//			b = x5 x6 x7 x8 y3 y4 y5 y6
//			c = y7 y8 z3 z4 z5 z6 z7 z8
//
//		w, x, y, =:
//
//			w = IXLATE (w)
//			x = IXLATE (x)
//			y = IXLATE (y)
//
//			a = w3 w4 w5 w6 w7 w8 x3 x4
//			b = x5 x6 x7 x8 y3 y4 y5 y6
//
//		w, x, =, =:
//
//			w = IXLATE (w)
//			x = IXLATE (x)
//
//			a = w3 w4 w5 w6 w7 w8 x3 x4
//
// ---FHDR---

#include "Base64Codec.h"

/*------------------------------------------------------------------------
 * input conversion table
 */
#define EQ	64

int Base64Codec::_decode_map[] = { 
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
	52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,EQ,-1,-1,
	-1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
	15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
	-1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
	41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
	-1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1
};

#define IXLATE(c)  (_decode_map[(c)])

/*------------------------------------------------------------------------
 * output conversion table
 */
unsigned char Base64Codec::_encode_map[] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

#define OXLATE(c)  (_encode_map[(c)])



// +++PHDR+++
//
// Function:	Base64Codec::decode
//
// Description:	convert a line encoded in base64 to a buffer
//
// Argument	Type			Description
// ------------	-----------------------	-------------------------------------
// buf_inp	const unsigned char *	input  buffer
// buf_out	unsigned char *		output buffer
// num_inp	int			number of chars in input buffer
//
// Returns:	-1 error
//		 0 no chars
//		>0 number of chars in output buffer
//
// Notes:	This routine assumes num_inp is a mult of 4
//
// ---PHDR---

int
Base64Codec::decode (const unsigned char *buf_inp, unsigned char *buf_out,
	int num_inp)
{
	unsigned char *abc = buf_out;

	/*----------------------------------------------------------------
	 * make sure input count is multiple of four
	 */
	if ((num_inp % 4) != 0)
	{
		return (-1);
	}

	/*----------------------------------------------------------------
	 * now process buffer four chars at a time
	 */
	while (num_inp > 0)
	{
		int w, x, y, z;

		/*--------------------------------------------------------
		 * get next 4 chars (each in range [0-64] [0x00 - 0x3f])
		 */
		w = IXLATE( buf_inp[0] );
		x = IXLATE( buf_inp[1] );
		y = IXLATE( buf_inp[2] );
		z = IXLATE( buf_inp[3] );

		/*--------------------------------------------------------
		 * validate all four chars
		 */
		if (w == -1 || x == -1 || y == -1 || z == -1)
		{
			return (-1);
		}

		/*--------------------------------------------------------
		 * now process according to the algorithm
		 */
		if (y != EQ && z != EQ)
		{
			*abc++ = ((w       ) << 2) | ((x & 0x30) >> 4);
			*abc++ = ((x & 0x0f) << 4) | ((y & 0x3c) >> 2);
			*abc++ = ((y & 0x03) << 6) | ((z       )     );
		}

		else if (y != EQ && z == EQ)
		{
			*abc++ = ((w       ) << 2) | ((x & 0x30) >> 4);
			*abc++ = ((x & 0x0f) << 4) | ((y & 0x3c) >> 2);
		}

		else /* if (y == EQ && z == EQ) */
		{
			*abc++ = ((w       ) << 2) | ((x & 0x30) >> 4);
		}

		/*--------------------------------------------------------
		 * bump ptr, decrement count, and do it again
		 */
		buf_inp += 4;
		num_inp -= 4;
	}

	return (abc - buf_out);
}


// +++PHDR+++
//
// Function:	Base64Codec::encode
//
// Description:	convert a buffer to a line encoded in base64
//
// Argument	Type			Description
// ------------	-----------------------	-------------------------------------
// buf_inp	const unsigned char *	input  buffer
// buf_out	unsigned char *		output buffer
// num_inp	int			number of chars in input buffer
//
// Returns:	-1 error
//		 0 no chars
//		>0 number of chars in output buffer
//
// Notes:
//
// ---PHDR---

int
Base64Codec::encode (const unsigned char *buf_inp, unsigned char *buf_out,
	int num_inp)
{
	const unsigned char *b = buf_inp;
	unsigned char *wxyz = buf_out;

	/*----------------------------------------------------------------
	 * check for invalid number of chars
	 */
	if (num_inp < 0)
		return (-1);

	/*----------------------------------------------------------------
	 * now process all chars in input buffer
	 */
	while (num_inp > 0)
	{
		unsigned char w;
		unsigned char x;
		unsigned char y;
		unsigned char z;
		int b0;
		int b1;
		int b2;
		int n;

		/*--------------------------------------------------------
		 * process next [1-3] chars according to algorithm
		 */
		if (num_inp >= 3)
		{
			/*------------------------------------------------
			 * process 3 chars
			 */
			b0 = b[0];
			b1 = b[1];
			b2 = b[2];

			w = OXLATE(((b0       ) >> 2)                     );
			x = OXLATE(((b0 & 0x03) << 4) | ((b1 & 0xf0) >> 4));
			y = OXLATE(((b1 & 0x0f) << 2) | ((b2 & 0xc0) >> 6));
			z = OXLATE(((b2 & 0x3f)     )                     );

			n = 3;
		}
		else if (num_inp == 2)
		{
			/*------------------------------------------------
			 * process 2 chars
			 */
			b0 = b[0];
			b1 = b[1];

			w = OXLATE(((b0       ) >> 2)                     );
			x = OXLATE(((b0 & 0x03) << 4) | ((b1 & 0xf0) >> 4));
			y = OXLATE(((b1 & 0x0f) << 2)                     );
			z = '=';

			n = 2;
		}
		else
		{
			/*------------------------------------------------
			 * process 1 char
			 */
			b0 = b[0];

			w = OXLATE(((b0       ) >> 2)                      );
			x = OXLATE(((b0 & 0x03) << 4)                      );
			y = '=';
			z = '=';

			n = 1;
		}

		wxyz[0] = w;
		wxyz[1] = x;
		wxyz[2] = y;
		wxyz[3] = z;

		wxyz    += 4;
		b       += n;
		num_inp -= n;
	}
	*wxyz = 0;

	return (wxyz - buf_out);
}


// +++PHDR+++
//
//  Method     : Base64Codec::encode
//
//  Visibility : public
//
//  Description: Reads from one stream encoding the read data into Base64
// 		 and writes the output to the output stream.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  in              istream *               input stream to encode
//  out             ostream *               output stream to write encoded data
//
//
//  Returns:   1 on success, 0 on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int
Base64Codec::encode (istream * in, ostream * out)
{
	unsigned char inpbuf[60];
	unsigned char outbuf[128];
	int inlen;
	int outlen;

	/*
	 * Now read and encode a chunk at a time
	 */
	while (!in->eof()) {

		in->read((char *)inpbuf, sizeof(inpbuf));
		inlen = in->gcount();

		outlen = encode(inpbuf, outbuf, inlen);
		if (outlen == -1) {
			return (0);
		} else if (outlen > 0) {
			out->write((char *)outbuf, outlen);
			if (out->bad()) {
				return (0);
			}

			out->put('\r');
			out->put('\n');
		}
	}

	return (1);
}


// +++PHDR+++
//
//  Method     : Base64Codec::decode
//
//  Visibility : public
//
//  Description: Reads from a stream containing Base64 encoded data decodes
//               the data and writes it on the output stream.
//
//  Parameter       Type                    Description
//  --------------- ----------------------- ----------------------------------
//  in              istream *               input stream to decode
//  out             ostream *               output stream to write decoded data
//
//
//  Returns:   1 on success, 0 on failure.
//
//
//  Notes:     none
//
// ---PHDR---

int
Base64Codec::decode (istream * in, ostream * out)
{
	unsigned char inpbuf[128];
	unsigned char outbuf[60];
	int inlen;
	int outlen;

	/*
	 * Now read and decode a chunk at a time
	 */
	while (!in->eof()) {

		in->read((char *)inpbuf, sizeof(inpbuf));
		inlen = in->gcount();

		outlen = decode(inpbuf, outbuf, inlen);
		if (outlen == -1) {
			return (0);
		} else if (outlen > 0) {
			out->write((char *)outbuf, outlen);
			if (out->bad()) {
				return (0);
			}
		}
	}

	return (1);
}

