/*------------------------------------------------------------------------
 * Program to try and/or modify a fax request
 */
#include <stdio.h>
#include <stdlib.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"n:t:c:WU:H:Z:?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] reqid\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -n number     New phone-number\n");
	fprintf(fp, "  -t tag=val    Tag to add or change\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
	fprintf(fp, "reqid           Request-id to try or modify\n");
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_CTX *		vctx;
	V_SYS *		vptr;
	char		msgbuf[V_MSG_LEN];
	const char *	reqid;
	V_TAG *		tag_list	= 0;
	const char *	pgmname		= argv[0];
	int		opt_index;
	int		c;
	int		rc;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		const char *opt = VSI_Opt_Argstr();

		switch (c)
		{
		case 'W':
			/*------------------------------------------------
			 * Who Am I
			 */
			{
				char	clientid[V_TAG_DATALEN];
				char	hostname[V_TAG_DATALEN];

				VSI_WhoAmI(vctx, clientid, hostname, 0);
				printf("%s@%s\n", clientid, hostname);
			}
			return (EXIT_SUCCESS);

		case 'n':
			/*------------------------------------------------
			 * process phone-number
			 */
			tag_list = VSI_Tag_Add(tag_list, V_TAG_ENV_TOFNUM,
				opt, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 't':
			/*------------------------------------------------
			 * process tag in form "tag=value"
			 */
			tag_list = VSI_Tag_Add_Str(tag_list, opt, msgbuf);
			if (*msgbuf != 0)
			{
				fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
				return (EXIT_FAILURE);
			}
			break;

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * already processed
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * get request id
	 */
	opt_index = VSI_Opt_Index();
	if ((argc - opt_index) != 1)
	{
		fprintf(stderr, "%s: No request ID provided\n", pgmname);
		return (EXIT_FAILURE);
	}

	reqid = argv[opt_index++];

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * submit the modify request
	 */
	rc = VSI_Server_Modify(vptr, reqid, tag_list, msgbuf);

	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	/*----------------------------------------------------------------
	 * free the tag list
	 */
	tag_list = VSI_Tag_Free(tag_list, 0);

	/*----------------------------------------------------------------
	 * display error msg if request failed
	 */
	if (rc)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	return (EXIT_SUCCESS);
}
