#!/bin/sh
# ------------------------------------------------------------------------
#
# This is a sample script for use by the external FIM mechanism to
# demonstrate the external FIM usage.
#
# ------------------------------------------------------------------------
#
# args:
#
#	start <dest>
#
#		Invoked once at startup time.
#			Args:
#				dest	destination name
#			Returns:
#				0	OK
#				!= 0	shutdown
#			Stdout:
#				Optional semicolon-separated list of
#				environment vars to set, in form "name=value".
#				If any field is "NO-STATUS", then no
#				status polling will be done.
#
#	stop  <dest>
#
#		Invoked once at stop time.
#			Args:
#				dest	destination name
#			Returns: ignored
#			Stdout:  ignored
#
#	send  <dest> <reqid> <file ...>
#
#		Send a fax request.
#			Args:
#				dest	destination name
#				reqid	assigned request-id
#				file	list of files to send
#			Returns:
#				0	OK
#				!= 0	error
#			Stdout:
#				EJID (external job ID)
#
#		Note that the phone-number to send the job to can
#		be gotten from the OLOG record.  The TFN tag will contain
#		the original dialstring and the DIA tag will contain the
#		converted dialstring.
#
#
#	recv  <dest>
#
#		Poll for any received faxes.
#			Args:
#				dest	destination name
#			Returns:
#				0	fax received
#				!= 0	no fax received
#			Stdout:
#				pathname of rcvd fax
#
#	cancel <dest> <reqid> <ejid>
#
#		Cancel a sent fax job.
#			Args:
#				dest	destination name
#				reqid	assigned request-id
#				ejid	external job-id
#			Returns:
#				0	job canceled
#				!= 0	error
#			Stdout: ignored
#
#	status <dest> <reqid> <ejid>
#
#		Get any status for a job.
#			Args:
#				dest	destination name
#				reqid	assigned request-id
#				ejid	external job-id
#			Returns:
#				0	successful query
#				!= 0	error
#			Stdout:
#				aresult-code (if available)
#
# ------------------------------------------------------------------------

# ------------------------------------------------------------------------
# usage
#
usage ()
{
	echo "usage: $PGM <cmd> <args ...>"
	echo "cmds:"
	echo "  start  <dest>"
	echo "  stop   <dest>"
	echo "  send   <dest> <reqid> <files ...>"
	echo "  recv   <dest>"
	echo "  cancel <dest> <reqid> <ejid>"
	echo "  status <dest> <reqid> <ejid>"
}

# ------------------------------------------------------------------------
# get name of our temp directory
#
get_tempdir ()
{
	echo "$VSIFAX/spool/temp/$base"
}

# ------------------------------------------------------------------------
# get name of our cache file for a job
#
get_cache_file ()
{
	reqid=$1
	echo "$EX_TEMP_DIR/$reqid.txt"
}

# ------------------------------------------------------------------------
# log routines
#
log_pathname ()
{
	dest=$1
	echo "$VSIFAX/spool/logs/$dest"
}

log_write ()
{
	echo "$*" >>$EX_LOG_PATH
}

log_error ()
{
	log_write "$@"
	echo "$*" >&2
}

log_separator ()
{
	log_write "-----------------------------------------"
}

# ------------------------------------------------------------------------
# start routine
#
do_start ()
{
	dest=$1

	# ----------------------------------------------------------------
	# start mechanism:
	#
	#	1.	startup the log file
	#	2.	create a temp dir for our stuff
	#	3.	output all env vars we want set
	#
	EX_LOG_PATH=`log_pathname $dest` export EX_LOG_PATH
	EX_TEMP_DIR=`get_tempdir $base`  export EX_TEMP_DIR

	log_separator
	log_write "starting"
	log_separator

	if [ ! -d "$EX_TEMP_DIR" ]
	then
		mkdir "$EX_TEMP_DIR" ||
		{
			log_error "Cannot create temp dir $EX_TEMP_DIR"
			exit 1
		}
	fi

	# ----------------------------------------------------------------
	# Note that here we set the log-path & temp-dir for all other calls.
	# Only the start routine needed to know where they are.
	#
	echo "EX_LOG_PATH=$EX_LOG_PATH;EX_TEMP_DIR=$EX_TEMP_DIR"
}

# ------------------------------------------------------------------------
# stop routine
#
do_stop ()
{
	dest=$1

	# ----------------------------------------------------------------
	# stop mechanism:
	#
	#	1.	write final log entry
	#
	log_separator
	log_write "stopping"
}

# ------------------------------------------------------------------------
# send routine
#
do_send ()
{
	dest=$1
	reqid=$2
	shift 2
	files="$@"

	# ----------------------------------------------------------------
	# send mechanism:
	#
	#	1.	create dummy ejid as "ex-$reqid"
	#	2.	cache fax record in temp dir
	#		This consists of the file "$reqid.txt" with
	#		the contents being $ejid.
	#	3.	output ejid
	#
	log_write "Sending job $reqid"

	file=`get_cache_file $reqid`
	ejid="ex-$reqid"
	echo "$ejid" >$file

	log_write "ejid: $ejid"

	echo "$ejid"
}

# ------------------------------------------------------------------------
# recv routine
#
do_recv ()
{
	dest=$1

	# ----------------------------------------------------------------
	# recv mechanism:
	#
	#	1.	Check for the file "$VSIFAX/spool/in/<dest>.tif
	#	2.	If found, pass the name back up
	#
	fax_path="$VSIFAX/spool/in/$dest.tif"
	[ ! -f "$fax_path" ] ||
	{
		log_write "Received $faxpath"
		echo "$fax_path"
	}
}

# ------------------------------------------------------------------------
# cancel routine
#
do_cancel ()
{
	dest=$1
	reqid=$2
	ejid="$3"

	# ----------------------------------------------------------------
	# cancel mechanism:
	#
	#	1.	validate the reqid
	#	2.	If found, delete it
	#
	log_write "Cancel $reqid [$ejid]"

	file=`get_cache_file $reqid`
	if [ ! -f "$file" ]
	then
		log_error "file $file not found"
		exit 1
	fi

	rm -f "$file"
}

# ------------------------------------------------------------------------
# status routine
#
do_status ()
{
	dest=$1
	reqid=$2
	ejid="$3"

	# ----------------------------------------------------------------
	# status mechanism:
	#
	#	1.	validate the reqid
	#	2.	say it succeeded
	#	3.	delete our temp file
	#
	file=`get_cache_file $reqid`
	if [ ! -f "$file" ]
	then
		log_error "file $file not found"
		exit 1
	fi

	status="normal"
	log_write "job $reqid: status $status"

	rm -f $file
	echo "$status"
}

# ------------------------------------------------------------------------
# main routine
#
PGM=`basename $0`

# ------------------------------------------------------------------------
# get cmd to execute
#
if [ $# -lt 2 ]
then
	echo "$PGM: invalid number of args" >&2
	usage >&2
	exit 1
fi

cmd=$1		export cmd
shift

# ------------------------------------------------------------------------
# get "base" name of destination
#
# i.e., "sample-hold" -> "sample"
#
# This name is used to find the temp dir for the send FIM
# and its associated status FIM.
#
dest=$1		export dest

base=`echo $dest | sed -e 's/-hold//'`	export base

# ------------------------------------------------------------------------
# execute proper cmd
#
case "$cmd" in

	# ----------------------------------------------------------------
	# start FIM processing
	#
	start)	do_start  "$@"	;;

	# ----------------------------------------------------------------
	# stop FIM processing
	#
	stop)	do_stop   "$@"	;;

	# ----------------------------------------------------------------
	# submit a fax job
	#
	send)	do_send   "$@"	;;

	# ----------------------------------------------------------------
	# poll for a received fax
	#
	recv)	do_recv   "$@"	;;

	# ----------------------------------------------------------------
	# cancel a fax job
	#
	cancel)	do_cancel "$@"	;;

	# ----------------------------------------------------------------
	# get status for a pending job
	#
	status)	do_status "$@"	;;

	# ----------------------------------------------------------------
	# display usage
	#
	help)	usage
		exit 0
		;;

	# ----------------------------------------------------------------
	# bogus cmd
	#
	*)	echo "$PGM: invalid cmd \"$cmd\"" >&2
		usage >&2
		exit 1
		;;
esac
