# +++PHDR+++
#
# Procedure:	tail_log__ReadFile
#
# Description:	Reads new data from a log file
#
#
# Argument		Description
# --------------------	----------------------------------------------------
# count                 Id of log
#
#
# Returns:	nothing
#
#
# Notes:
#
# ---PHDR---

proc tail_log__ReadFile { count } {
	global tail_log

	if { $tail_log($count,id) == "" } {
		return
	}

	set tail_log($count,id) ""

	set form $tail_log($count,form)

	$form.log.text configure -state normal

	#
	# See if a scrollbar controll is pressed
	#
	set sb_is_active [ $form.log.sb activate ]

	#
	# Read to EOF
	#
	while { [ gets $tail_log($count,fd) line ] > 0 } {

		#
		# increment the line count and remove the 1st line if we are
		# at the max line count
		#
		incr tail_log($count,lines)
		if { $tail_log($count,lines) > $tail_log(max_lines) } {
			$form.log.text delete 1.0 2.0
		}

		#
		# Insert the new line of text
		#
		$form.log.text insert end "$line\n"

		if { $sb_is_active == "" } {
			$form.log.text see end
		}
	
	}

	#
	# Update the clock
	#
	$form.log.text configure -state disabled
	$form.controls.clock configure \
		-text [clock format [clock seconds] -format "%T" -gmt 0]

	update

	#
	# Set the after command and save its ID
	#
	set tail_log($count,id) [ after 500 [list tail_log__ReadFile $count] ]
}


# +++PHDR+++
#
# Procedure:	tail_log__Done
#
# Description:	command for the "Done" button
#
#
# Argument		Description
# --------------------	----------------------------------------------------
# count                 Log id
#
#
# Returns:	nothing
#
#
# Notes:
#
# ---PHDR---

proc tail_log__Done { count } {
	global tail_log

	tail_log__Close $count

	if { $tail_log($count,shell) == "." } {
		exit 0
	} else {
		catch { destroy $tail_log($count,shell) }
	}
}


# +++PHDR+++
#
# Procedure:	tail_log__Close
#
# Description:	Close the logged file and cancels the timer
#
#
# Argument		Description
# --------------------	----------------------------------------------------
# count			Log id
#
#
# Returns:	nothing
#
#
# Notes:
#
# ---PHDR---

proc tail_log__Close { count } {
	global tail_log

	if { $tail_log($count,id) != "" } {
		after cancel $tail_log($count,id)
		set tail_log($count,id) ""
	}

	if { $tail_log($count,fd) != "" } {
		close $tail_log($count,fd)
		set tail_log($count,fd) ""
	}
}


# +++PHDR+++
#
# Procedure:	tail_log__Reopen
#
# Description:	Reopens the log file
#
#
# Argument		Description
# --------------------	----------------------------------------------------
# count			Log id
#
#
# Returns:	nothing
#
#
# Notes:
#
# ---PHDR---

proc tail_log__Reopen { count } {
	global tail_log

	tail_log__Close $count
	tail_log__Open  $count $tail_log($count,fname)
}


# +++PHDR+++
#
# Procedure:	tail_log__Open
#
# Description:	Opens a file in a log tail window
#
#
# Argument		Description
# --------------------	----------------------------------------------------
# count			Log id
# filename              Name of file to open (blank for file selection)
#
#
# Returns:	nothing
#
#
# Notes:
#
# ---PHDR---

proc tail_log__Open { count filename } {
	global tail_log

	set vfaxdir [vcl vfaxdir]
	if { $vfaxdir == "" } {
		set logdir "."
	} else {
		set logdir "$vfaxdir/spool/logs"
	}

	set tail_log($count,fname) ""
	if { $filename == "" } {
		#
		# Get the file to open
		#
		set curdir [pwd]
		cd "$logdir"
		set filename [tk_getOpenFile -parent $tail_log($count,shell)]
		cd $curdir
	
		if { $filename == "" } {
			return
		}
	} 

	tail_log__Close $count

	#
	# Open the log file
	#
	set tail_log($count,fd) ""
	set curdir [pwd]
	cd "$logdir"
	catch { set tail_log($count,fd) [ open $filename r ] } errmsg
	cd $curdir
	if { $tail_log($count,fd) == "" } {
		tk_messageBox -title "Open Log" -type ok \
			-icon error -parent $tail_log($count,shell) \
			-message "Couldn't open \"$filename\""
		return
	}	

	set tail_log($count,fname)  $filename

	#
	# Set the file to non-blocking
	#
	fconfigure $tail_log($count,fd) -blocking 0

	#
	# Put the title on the shell (with the filename)
	#
	wm title $tail_log($count,shell) \
		"VSI-FAX Log Viewer    Log: [file tail $filename]"

	#
	# Clear the current contents of the log text
	#
	$tail_log($count,form).log.text configure -state normal
	$tail_log($count,form).log.text delete 1.0 end

	#
	# Display the log file
	#
	set tail_log($count,id) none
	set tail_log($count,lines) 0
	tail_log__ReadFile $count

	return
}		


# +++PHDR+++
#
# Procedure:	tail_log__Popup
#
# Description:	Creates and pops up a log window
#
#
# Argument		Description
# --------------------	----------------------------------------------------
# fname                 Name of log file to open
# shell                 Name of the shell to create the log in
#
# Returns:	nothing
#
#
# Notes:
#
# ---PHDR---

proc tail_log__Popup { fname shell } {
	global tail_log

	#
	# Increment the log count and initialize the log array
	#
	incr tail_log(count)
	set tail_log($tail_log(count),shell) ""
	set tail_log($tail_log(count),id)    ""
	set tail_log($tail_log(count),fd)    ""
	set tail_log($tail_log(count),fname) ""

	#
	# Build th main form for the log
	#
	tail_log__BuildMainForm $tail_log(count) $shell

	#
	# Open the first file
	#
	tail_log__Open $tail_log(count) $fname
}


# +++PHDR+++
#
# Procedure:	tail_log__BuildMainForm
#
# Description:	Builds a log tail window
#
#
# Argument		Description
# --------------------	----------------------------------------------------
# count                 Log id
# shell                 Name of shell to put log in
#
# Returns:	nothing
#
#
# Notes:
#
# ---PHDR---

proc tail_log__BuildMainForm { count shell } {
	global tail_log

	#
	# Create the top level shell if the shell is not "."
	#
	set form ".log$count"

	if { $shell == "" } {
		set shell $form
		catch { destroy $shell }
		toplevel $form
	} else {
		catch { destroy $form }
		frame $form
	}

	set tail_log($count,form)  $form
	set tail_log($count,shell) $shell

	#
	# Put the title on the shell
	#
	wm title $shell "VSI-FAX Log Viewer"

	#
	# Setup a handler for the window manager "Close" function
	#
	wm protocol $shell WM_DELETE_WINDOW [ list tail_log__Done $count ]


	frame $form.log
	text $form.log.text -height 10 -width 70 -state normal \
		-yscrollcommand "$form.log.sb set"
	scrollbar $form.log.sb -command "$form.log.text yview"
	pack $form.log.sb -side right -fill y 
	pack $form.log.text -side left -fill both -expand 1
	pack $form.log -side top -fill both -expand 1


	frame $form.controls
	button $form.controls.exit -width 7 -text "Done" \
		-command [ list tail_log__Done $count ]
	button $form.controls.reopen -width 7 -text "Reopen" \
		-command [ list tail_log__Reopen $count ]
	button $form.controls.open -width 7 -text "Open..." \
		-command [ list tail_log__Open $count "" ]
	pack $form.controls.exit $form.controls.reopen \
		$form.controls.open -side left

	label $form.controls.clock
	pack $form.controls.clock -side right -fill x

	pack $form.controls -side bottom -fill x -anchor sw
	
	catch { pack $form -fill both -expand 1 }
	update

	#
	# Set the current size as the minimum size for the form
	#
	set h [ winfo height $shell ]
	set w [ winfo width  $shell ]
	wm minsize $shell $w $h
}


global tail_log

#
# Set the max display line count
#
set tail_log(max_lines) 1024

#
# Reset the log count
#
set tail_log(count) 0

return
