#!/bin/sh
#
#+++FHDR+++
#
#	Description:
#
#	configuration for the VFS package
#
#---FHDR---

# ------------------------------------------------------------------------
# echo an error msg line
#
echoerr ()
{
	echo "$*" >&2
}

# ------------------------------------------------------------------------
# Abort the installation
#
abort_install ()
{
	echoerr ""
	echoerr "Installation aborted."
	exit 1
}

# ------------------------------------------------------------------------
# check if running in windows
#
in_windows ()
{
	[ -d "c:/" ]
}

# ------------------------------------------------------------------------
# add a directory to the path
#
# Note that the directory is ALWAYS added to the end.
#
addpath ()
{
	dir="$1"

	sep=":"
	arg=""
	if in_windows
	then
		sep=";"
		arg="-i"
	fi

	foo=`echo "$PATH" | egrep $arg "^$dir$sep|$sep$dir$sep|$sep$dir$"`
	if [ $? -eq 0 ]
	then
		echo "$PATH"
	else
		echo "$PATH$sep$dir"
	fi

	unset dir sep arg foo
}

# ------------------------------------------------------------------------
# get name of the console device
#
get_dev_tty ()
{
	if in_windows
	then
		echo "con:"
	else
		echo "/dev/tty"
	fi
}

# ------------------------------------------------------------------------
# read user input from a response file or the keyboard & save the response
#
read_user_input ()
{
	id="$1"
	found=no

	CONSOLE=`get_dev_tty`

	if [ "$id" = "" -o "$LOAD_RESPONSE_FILE" = "" ]
	then
		read response <$CONSOLE
	else
		response=`grep "^$id:" "$LOAD_RESPONSE_FILE"`
		if [ $? -eq 0 ]
		then
			response=`echo "$response" | sed -e 's/^[^:]*://'`
			echo "$response" >&2
			found=yes
		else
			read response <$CONSOLE
		fi
	fi

	if [ "$found" = "no" -a "$id" != "" -a "$SAVE_RESPONSE_FILE" != "" ]
	then
		sed -e "/^$id:/d" "$SAVE_RESPONSE_FILE" >"$SAVE_RESPONSE_FILE.tmp"
		echo "$id:$response" |
			cat "$SAVE_RESPONSE_FILE.tmp" - >"$SAVE_RESPONSE_FILE"
		rm -f "$SAVE_RESPONSE_FILE.tmp"
	fi

	echo "$response"
}

# ------------------------------------------------------------------------
# check if running in BSD
#
in_bsd ()
{
	[ \( -f /vmunix -a ! -f /osf_boot \) -o \
		-f /netbsd -o \
		-f /kernel -o \
		-f /mach_kernel -o \
		`uname -s` = "FreeBSD" -o \
		`uname -s` = "OpenBSD" -o \
		`uname -s` = "NetBSD" ]
}

# ------------------------------------------------------------------------
# check if running in linux
#
in_linux ()
{
	[ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
}

# ------------------------------------------------------------------------
# echo a line without a final CR
#
echonl ()
{
	if in_linux
	then
		# --------------------------------------------------------
		# Linux
		#
		/bin/echo -e "${*}\c"

	elif in_bsd
	then
		# --------------------------------------------------------
		# BSD
		#
		echo -n "${*}"

	else
		# --------------------------------------------------------
		# System V
		#
		echo "${*}\c"
	fi
}

# ------------------------------------------------------------------------
# getyesorno() -- this prompts the user to hit Yes or No, and it returns
# true if the user hits yes, false if the user hits no, and it loops if
# the user hits something else.
#
getyesorno ()
{
	default=y
	if [ x"$1" = x"-y" ]
	then
		default=y
		shift
	elif [ x"$1" = x"-n" ]
	then
		default=n
		shift
	fi

	id="$1"
	shift

	while true
	do
		echonl "$@ (y/n) [$default] : "
		answer=`read_user_input "$id"`

		if [ "$answer" = "" ]
		then
			answer=$default
		fi

		case "$answer" in

		[yY]*)
			return 0
			;;

		[nN]*)
			return 1
			;;
		esac
	done
}

# ------------------------------------------------------------------------
# This prompts the user for OK to proceed
#
is_ok ()
{
	getyesorno "$1" "Is this OK?"
}

# ------------------------------------------------------------------------
# output a prompt & get a response from the user
#
get_user_input ()
{
	id="$1"
	dflt="$2"
	msg="$3"

	ok_id="$id-ok"
	[ "$id" = "" ] && ok_id=""

	CONSOLE=`get_dev_tty`

	while true
	do
		if [ x"$dflt" = x"!" ]
		then
			echonl "Enter $msg: "
		else
			echonl "Enter $msg [$dflt]: "
		fi >$CONSOLE

		response=`read_user_input "$id"`
		[ "$response" = "" -a x"$dflt" != x"!" ] && response="$dflt"

		echo "$msg = \"$response\"" >$CONSOLE
		is_ok "$ok_id" >$CONSOLE && { echo "$response"; break; }

		dflt="$response"
	done
}

# ------------------------------------------------------------------------
# Return the package name to use
#
pkgname ()
{
	echo "${VSI_PKGNAME:-VSI-FAX}"
}

# ------------------------------------------------------------------------
# get Windows directory
#
get_windir ()
{
	{
		if [ "$windir" != "" ]
		then
			echo "$windir"

		elif [ "$WINDIR" != "" ]
		then
			echo "$WINDIR"

		elif [ "$SystemRoot" != "" ]
		then
			echo "$SystemRoot"

		elif [ "$SYSTEMROOT" != "" ]
		then
			echo "$SYSTEMROOT"

		elif [ -d "c:/windows" ]
		then
			echo "c:/windows"

		elif [ -d "c:/winnt" ]
		then
			echo "c:/winnt"

		else
			echo "c:/windows"
		fi
	} | sed -e 's,\\\\*,/,g'
}

# ------------------------------------------------------------------------
# get Windows drive
#
get_windrive()
{
	get_windir | sed -e 's,[/\\].*,,'
}

# ------------------------------------------------------------------------
# Return the product name to use
#
prdname ()
{
	echo "${VSI_PRDNAME:-vsifax}"
}

# ------------------------------------------------------------------------
# get response dir/file info
#
response_info ()
{
	if in_windows
	then
		RESPONSE_DIR="`get_windrive`/Program Files/Common Files/VSI-FAX Shared"
	else
		RESPONSE_DIR="/var/spool/pkg"
	fi
	RESPONSE_FILE="`prdname`-install.log"

	echo "RESPONSE_DIR=\"$RESPONSE_DIR\""
	echo "SAVE_RESPONSE_FILE=\"$RESPONSE_DIR/$RESPONSE_FILE\""
}

# ------------------------------------------------------------------------
# check if a boot entry exists
#
check_boot ()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# check if partial install OK
	#
	PART=no
	if [ x"$1" = x"-p" ]
	then
		PART=yes
		shift
	fi

	# ----------------------------------------------------------------
	# get name to look for
	#
	NAME=$1

	# ----------------------------------------------------------------
	# Check for Linux
	#
	if [ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
	then
		if [ "$PART" = "no" ]
		then
			[ -x /etc/rc.d/rc2.d/S99$NAME -a \
			  -x /etc/rc.d/rc2.d/K01$NAME ]
		else
			[ -x /etc/rc.d/rc2.d/S99$NAME -o \
			  -x /etc/rc.d/rc2.d/K01$NAME -o \
			  -x /etc/rc.d/rc0.d/K01$NAME ]
		fi
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for BSD
	#
	if [ -f /etc/rc.local ]
	then
		grep $NAME /etc/rc.local >/dev/null 2>&1
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for standard System V
	#
	if [ -d /etc/rc2.d ]
	then
		if [ "$PART" = "no" ]
		then
			[ -x /etc/rc2.d/S99$NAME -a \
			  -x /etc/rc2.d/K01$NAME ]
		else
			[ -x /etc/rc2.d/S99$NAME -o \
			  -x /etc/rc2.d/K01$NAME -o \
			  -x /etc/rc0.d/K01$NAME ]
		fi
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for OSF or HP
	#
	if [ -d /sbin/rc2.d ]
	then
		if [ "$PART" = "no" ]
		then
			[ -x /sbin/rc2.d/S99$NAME -a \
			  -x /sbin/rc2.d/K01$NAME ]
		else
			[ -x /sbin/rc2.d/S99$NAME -o \
			  -x /sbin/rc2.d/K01$NAME -o \
			  -x /sbin/rc0.d/K01$NAME ]
		fi
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for old System V
	#
	if [ -d /etc/rc.d/7 ]
	then
		[ -x /etc/rc.d/7/S99$NAME ]
		return $?
	fi

	# ----------------------------------------------------------------
	# Check for AIX
	#
	if [ -f /etc/rc ]
	then
		grep $NAME.sh /etc/rc >/dev/null 2>&1
		return $?
	fi

	# ----------------------------------------------------------------
	# Cannot determine auto-boot location
	#
	return 1
}

# ------------------------------------------------------------------------
# Set a boot entry for auto-start/stop
#
set_boot ()
{
	# ----------------------------------------------------------------
	# bail if in windows
	#
	in_windows && return 0

	# ----------------------------------------------------------------
	# get options
	#
	NAME=$1
	OPTION=$2

	SCRIPT="$VSIFAX/lbin/$NAME.sh"

	# ----------------------------------------------------------------
	# Check for Linux
	#
	if [ -f /etc/lilo.conf -o `uname -s` = "Linux" ]
	then
	     rm -f /etc/rc.d/rc5.d/S99$NAME
	     rm -f /etc/rc.d/rc3.d/S99$NAME
	     rm -f /etc/rc.d/rc2.d/S99$NAME
	     rm -f /etc/rc.d/rc6.d/K01$NAME
	     rm -f /etc/rc.d/rc5.d/K01$NAME
	     rm -f /etc/rc.d/rc3.d/K01$NAME
	     rm -f /etc/rc.d/rc2.d/K01$NAME
	     rm -f /etc/rc.d/rc1.d/K01$NAME
	     rm -f /etc/rc.d/rc0.d/K01$NAME	# TEMP hack

	     if [ "$OPTION" = "on" ]
	     then
	      [ -d /etc/rc.d/rc5.d ] && ln -s $SCRIPT /etc/rc.d/rc5.d/S99$NAME
	      [ -d /etc/rc.d/rc3.d ] && ln -s $SCRIPT /etc/rc.d/rc3.d/S99$NAME
	      [ -d /etc/rc.d/rc2.d ] && ln -s $SCRIPT /etc/rc.d/rc2.d/S99$NAME
	      [ -d /etc/rc.d/rc6.d ] && ln -s $SCRIPT /etc/rc.d/rc6.d/K01$NAME
	      [ -d /etc/rc.d/rc5.d ] && ln -s $SCRIPT /etc/rc.d/rc5.d/K01$NAME
	      [ -d /etc/rc.d/rc3.d ] && ln -s $SCRIPT /etc/rc.d/rc3.d/K01$NAME
	      [ -d /etc/rc.d/rc2.d ] && ln -s $SCRIPT /etc/rc.d/rc2.d/K01$NAME
	      [ -d /etc/rc.d/rc1.d ] && ln -s $SCRIPT /etc/rc.d/rc1.d/K01$NAME
	      [ -d /etc/rc.d/rc0.d ] && ln -s $SCRIPT /etc/rc.d/rc0.d/K01$NAME
	     fi

	     return 0
	fi

	# ----------------------------------------------------------------
	# Check for BSD
	#
	if [ -f /etc/rc.local ]
	then
		cp /etc/rc.local /etc/rc.local-
		sed -e "/$NAME/d" /etc/rc.local- > /etc/rc.local
		if [ "$OPTION" = "on" ]
		then
			echo "$SCRIPT start" >> /etc/rc.local
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for standard System V
	#
	if [ -d /etc/rc2.d ]
	then
		rm -f /etc/rc2.d/S99$NAME
		rm -f /etc/rc2.d/K01$NAME
		rm -f /etc/rc0.d/K01$NAME	# TEMP hack

		if [ "$OPTION" = "on" ]
		then
			ln -s $SCRIPT /etc/rc2.d/S99$NAME
			ln -s $SCRIPT /etc/rc2.d/K01$NAME
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for OSF or HP
	#
	if [ -d /sbin/rc2.d ]
	then
		rm -f /sbin/rc2.d/S99$NAME
		rm -f /sbin/rc2.d/K01$NAME
		rm -f /sbin/rc0.d/K01$NAME	# TEMP hack

		if [ "$OPTION" = "on" ]
		then
			ln -s $SCRIPT /sbin/rc2.d/S99$NAME
			ln -s $SCRIPT /sbin/rc2.d/K01$NAME
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for old System V
	#
	if [ -d /etc/rc.d/7 ]
	then
		if [ "$OPTION" = "on" ]
		then
			ln -s $SCRIPT /etc/rc.d/7/S99$NAME
		else
			rm -f /etc/rc.d/7/S99$NAME
		fi

		return 0
	fi

	# ----------------------------------------------------------------
	# Check for AIX
	#
	if [ -f /etc/rc ]
	then
		cp /etc/rc /etc/rc-
		sed -e "/$NAME/d" -e '/^exit 0/d' /etc/rc- > /etc/rc
		if [ "$OPTION" = "on" ]
		then
			echo "$SCRIPT start" >> /etc/rc
		fi
		echo "exit 0" >> /etc/rc

		return 0
	fi

	# ----------------------------------------------------------------
	# Cannot determine auto-boot location
	#
	echoerr "Cannot determine auto-start at boot time procedure"
	return 1
}

# ------------------------------------------------------------------------
# help msg
#
usage ()
{
	echo "usage: $PGM [options]"
	echo "options:"
	echo "  -r     read response file"
}

# ------------------------------------------------------------------------
# helpall msg
#
helpall ()
{
	usage

	echo "  -D     debug install"
	echo "  -x     remove any configurations"
}

# ------------------------------------------------------------------------
# check if server is set in global .ini file
#
configure_hostname ()
{
	HOSTNAME=`vadmin config -clq LOGIN Hostname`
	if [ $? -ne 0 ]
	then
		SERVERS=`vadmin server query`
		set -- $SERVERS
		if [ $# -eq 1 ]
		then
			HOSTNAME=$1
		else
			if [ $# -gt 1 ]
			then
				echo "The following $PKGNAME servers were found:"
				for s in $SERVERS
				do
					echo "  $s"
				done

				echo ""
				echo "Please choose which $PKGNAME server" \
					"to connect to."
			else
				echo "Please enter the name of the machine" \
					"running the $PKGNAME server."
			fi

			HOSTNAME=`get_user_input "vfs-hostname" \
				"$HOSTNAME" "$PKGNAME server"`
		fi

		vadmin config -ca LOGIN Hostname $HOSTNAME
	fi
}

# ------------------------------------------------------------------------
# configure autostart at boot
#
configure_pickup_start ()
{
	if check_boot vsipickup
	then
		:
	else
		echo ""
		echo "Setting the $PKGNAME VFS Pickup Service to " \
			"start automatically at system boot time."
		is_ok "pickup-start-ok" && set_boot vsipickup on
	fi
}

configure_autostart ()
{
	if [ "$1" = "on" ]
	then
		if in_windows
		then
			vfxpickup -s install
		else
			configure_pickup_start
		fi
	else
		if in_windows
		then
			vfxpickup -s remove
		else
			set_boot vsipickup off
		fi
	fi
}

# ------------------------------------------------------------------------
# main()
#
PATH=`addpath "$VSIFAX/bin"`
PATH=`addpath "$VSIFAX/lbin"`

PKGNAME=`pkgname`

eval `response_info`

# ------------------------------------------------------------------------
# process options
#
PGM=`basename $0`

DEBUG=
DELETE=no

case "$1" in
	'-?' | -help | --help)
		usage
		exit 0
		;;

	'-??' | -helpall | --helpall)
		helpall
		exit 0
		;;

	'-V' | -version | --version)
		echo "$PGM: `vadmin version`"
		exit 0
		;;
esac

ARGS=`getopt "Drx" $*` || { usage >&2; exit 1; }
set -- $ARGS

while [ "$1" != "" ]
do
	case "$1" in

	-D)
		DEBUG=-x
		shift
		;;

	-r)
		LOAD_RESPONSE_FILE="$SAVE_RESPONSE_FILE"
		shift
		;;

	-x)
		DELETE=yes
		shift
		;;

	--)
		shift
		break
		;;

	*)
		echoerr "Invalid argument \"$1\""
		abort_install
		;;
	esac
done

export SAVE_RESPONSE_FILE
export LOAD_RESPONSE_FILE

if [ "$SAVE_RESPONSE_FILE" != "" ]
then
	[ ! -d "$RESPONSE_DIR" ] && mkdir -p "$RESPONSE_DIR"
	touch "$SAVE_RESPONSE_FILE"
	chmod 664 "$SAVE_RESPONSE_FILE"
fi

[ "$DEBUG" != "" ] && set $DEBUG

# ------------------------------------------------------------------------
# now do/undo all configurations
#
if [ "$DELETE" = "no" ]
then
	configure_hostname
	configure_autostart on
else
	configure_autostart off
fi
