package com.vsi.xmlf;

import java.io.*;
import java.net.Socket;

/**
 * This class implements a DocumentTransport for transporting
 * XML-F documents to the VSI-FAX vxmld daemon.
 *
 */
public
class VxmldTransport implements DocumentTransport {

	OutputString _err_msg = null;

	static final int DEFAULT_XMLD_PORT = 2996;
	static final int XML_READ_BUF_SIZE = 16384;


	/**
	 * No args constructor.
	 */
	public VxmldTransport() {
		_err_msg = new OutputString();
	}


	/**
	 * Gets the last error message from the transport.
	 *
	 * @return A String containing the last error message generated
	 * by the transport.
	 */
	public String getLastError () {
		return (_err_msg.toString());
	}


	/**
	 * Writes the specified XML data to the specified URL.
	 *
	 * @param  url_spec  the URL of the vxmld server.
	 * @param  xml_data  the XML data to send.
	 *
	 * @return  an InputStream to read the response from.
	 */
	public InputStream write (String url_spec, String xml_data) {

		Socket s = openSocket(url_spec);
		if (s == null) {
			return (null);
		}


		/*
		 * Write the XML data to the daemon
		 */
		byte buf[] = xml_data.getBytes();
		if (buf == null) {
			closeSocket(s);
			return (null);
		}
	
		try {
			OutputStream output = s.getOutputStream();
			output.write(buf);
		} catch (IOException ioe) {

			_err_msg.append(ioe.getMessage());
			closeSocket(s);
			return (null);
		}
			
		try {
			return (s.getInputStream());
		} catch (IOException ignore) {
			closeSocket(s);
			return (null);
		}
	}


	/**
	 * Writes the data from the specified stream to the specified URL.
	 *
	 * @param  url_spec  the URL of the vxmld server.
	 * @param  xml_stream  the XML stream to send.
	 *
	 * @return  an InputStream to read the response from.
	 */
	public InputStream write (String url_spec, InputStream xml_stream) {

		Socket s = openSocket(url_spec);
		if (s == null) {
			return (null);
		}


		/*
		 * Now read and send a line at a time
		 */
		byte xml_buf[] = new byte[XML_READ_BUF_SIZE];
		int nread;

		try {
			OutputStream output = s.getOutputStream();
			while ((nread = xml_stream.read(xml_buf)) != -1) {
				output.write(xml_buf, 0, nread);
			}
		} catch (Exception e) {
			_err_msg.append(e.getMessage());
			closeSocket(s);
			return (null);
		}

		try {
			return (s.getInputStream());
		} catch (IOException ignore) {
			closeSocket(s);
			return (null);
		}
	}


	/**
	 * Creates a connection to the specified URL and returns the
	 * socket for that connection.
	 *
	 * @param  url_spec  the URL to connect to
	 * @return  the Socket for the connection
	 */
	protected Socket openSocket (String url_spec) {

		/*
		 * Parse the URL and open a connection to the xmld daemon
		 */
		Url url = new Url(url_spec);
		String hostname = url.getHost();
		if (hostname == null) {
			_err_msg.append("no hostname specified in URL");
			return (null);
		}

		int port = url.getPort();
		if (port < 1) {
			port = DEFAULT_XMLD_PORT;
		}

		try {
			return (new Socket(hostname , port));
		} catch (Exception e) {
			_err_msg.append(e.getMessage());
			return (null);
		}
	}


	/**
	 * Close the connection.
	 *
	 * @param  s  the socket for the connection
	 */
	protected void closeSocket (Socket s) {
		try {
			s.close();
		} catch (Exception ignore) {
		}
	}
}

