package com.vsi.xmlf;

import java.io.InputStream;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;
import java.util.Vector;

/**
 * The TransportManager class manages the transports that are available
 * for sending XML-F requests to the network fax server. Transports can
 * be added at runtime by binding a transport protocol name to a
 * transporting object. All transports must be sub-classed from the
 * DocumentTransport class. Currently the HTTP transport is implemented
 * and automatically bound when the TransportManager class is initialized. 
 * 
 * Additionally, the TransportManager class manages XML document decoders.
 * These document decoders take an XML document and map it into an object
 * which represents that document. Document decoders must be sub-classed
 * from the DocumentDecoder class. Document decoders are provided for the
 * three XML-F response documents (fax-submit-response,
 * fax-status-response, and fax-cancel-response documents).
 *
 */
public
class TransportManager {

	String       _default_protocol = null;
	OutputString _err_msg          = null;
	Registry     _decoders         = null;
	Registry     _transports       = null;

	Vector       _rsp_listeners = new Vector();


	/**
	 * Constructor for TransportManager class.
	 */
	public TransportManager () {
		_default_protocol = null;
		_err_msg    = new OutputString();
		_decoders   = new Registry();
		_transports = new Registry();
	}


	/**
	 * Returns the last reported error message.
	 */
	public String getLastError () {
		return (_err_msg.toString());
	}


	/**
	 * Get the default transport protocol.
	 *
	 * @returns String containing the default transport protocol or
	 * null if no default protocol has been set.
	 */
	public String getDefaultTransport () {
		return (_default_protocol);
	}


	/**
	 * Sets the default transport protocol. The default transport
	 * protocol is used whenever no protocol is present in a URL.
	 *
	 * @param  protocol  the new default protocol.
	 */
	public void setDefaultTransport (String protocol) {
		_default_protocol = protocol;
	}


	/**
	 * Binds (maps) a document type to a document decoder.
	 * A document type may be bound to only one decoder at a time.
	 *
	 *
	 * @param  doc_type   name of the document type to bind.
	 * @param  decoder    the document decoder.
	 *
	 * @returns true on success, false on failure.
	 */
	public boolean bindDecoder (String doc_type, DocumentDecoder decoder) {
		try {
			_decoders.bind(doc_type, decoder);
			return (true);
		} catch (Exception e) {
			return (false);
		}
	}


	/**
	 * Description: Binds (maps) a protocol name to a transport.
	 * A protocol name may be bound to only one transport at a time.
	 *
	 *
	 * @param  protocol   name of transport protocol to bind.
	 * @param  transport  transport for protocol.
	 *
	 * @returns true on success, false on failure.
	 */
	public boolean bindTransport (String protocol, 
		DocumentTransport transport) {

		try {
			_transports.bind(protocol, transport);
			return (true);
		} catch (Exception e) {
			return (false);
		}
	}


	/**
	 * Returns an enumeration of the currently bound document decoders.
	 *
	 * @returns  An array of String, one for each bound decoder.
	 */
	public String [] listDecoders () {
		return (_decoders.list());
	}


	/**
	 * Returns an enumeration of the currently bound transports.
	 *
	 * @returns  An array of String, one for each bound transport.
	 */
	public String [] listTransports () {
		return (_transports.list());
	}


	/**
	 * Returns the DocumentDecodert object for the specified
	 * document type. If the document type is not bound null
	 * will be returned.
	 *
	 * @param  doc_type  document type to lookup.
	 *
	 * @returns the DocumentDecoder mapped to the specified document type.
	 */
	public DocumentDecoder lookupDecoder (String doc_type) {
		try {
			return ((DocumentDecoder)_decoders.lookup(doc_type));
		} catch (Exception ignore) {
			return (null);
		}
	}


	/**
	 * Returns the DocumentTransport object for the specified
	 * protocol. If the protocol name is not bound null will
	 * be returned.
	 *
	 * @param  protocol  name of the transport protocol
	 *
	 * @returns  the DocumentTransport mapped to the specified
	 * protocol name.
	 */
	public DocumentTransport lookupTransport (String protocol) {
		try {
			return ((DocumentTransport)
				_transports.lookup(protocol));
		} catch (Exception ignore) {
			return (null);
		}
	}


	/**
	 * Sends the XML-F data (document) contained by xml_data to
	 * the fax server specified by url. Returns the data returned
	 * from the fax server.
	 *
	 * @param  url  the URL to send the XMLF data to.
	 * @param  xml_data  the XML data to send.
	 *
	 * @returns  A String containing the response from the fax server.
	 */
	public String send (String url_spec, String xml_data) {

		Url url = new Url(url_spec);
		String protocol = url.getProtocol();
		if (protocol == null) {
			protocol = getDefaultTransport();
			if (protocol == null) {
				_err_msg.append(
					"URL spec must contain a protocol");
				return (null);
			}
	
			url.setProtocol(protocol);
		}

		DocumentTransport transport = lookupTransport(protocol);
		if (transport == null) {
			_err_msg.append("invalid protocol: " + protocol);
			return (null);
		}


		InputStream in = transport.write(url.toString(), xml_data);
		if (in == null) {
			_err_msg.append(transport.getLastError());
		}

		return (read(in));
	}


	/**
	 * Sends the XML-F data (document) contained by xml_stream to
	 * the fax server specified by url. Returns the data returned
	 * from the fax server.
	 *
	 * @param  url  URL to send the XMLF data to.
	 * @param  xml_stream   the stream to read XML data from.
	 *
	 * @returns  A String containing the response from the fax server.
	 */
	public String send (String url_spec, InputStream xml_stream) {

		Url url = new Url(url_spec);
		String protocol = url.getProtocol();
		if (protocol == null) {
			protocol = getDefaultTransport();
			if (protocol == null) {
				_err_msg.append(
					"URL spec must contain a protocol");
				return (null);
			}
	
			url.setProtocol(protocol);
		}

		DocumentTransport transport = lookupTransport(protocol);
		if (transport == null) {
			_err_msg.append("invalid protocol: " + protocol);
			return (null);
		}

		InputStream in = transport.write(url.toString(), xml_stream);
		if (in == null) {
			_err_msg.append(transport.getLastError());
		}

		return (read(in));
	}


	/**
	 * Sends the XML-F document contained by the XmlfObject to the
	 * fax server specified by url. Returns the data decoded into
	 * a Response.
	 *
	 * @param  url_spec  URL to send the XMLF data to.
	 * @param  obj  the XmlfObject to send.
	 * 
	 * @returns A Response containing the response from the fax server.
	 */
	public Response send (String url_spec, XmlfObject obj) {

		Url url = new Url(url_spec);
		String protocol = url.getProtocol();
		if (protocol == null) {
			protocol = getDefaultTransport();
			if (protocol == null) {
				_err_msg.append(
					"URL spec must contain a protocol");
				return (null);
			}
	
			url.setProtocol(protocol);
		}

		DocumentTransport transport = lookupTransport(protocol);
		if (transport == null) {
			_err_msg.append("invalid protocol: " + protocol);
			return (null);
		}


		String xml_data = obj.toXml();
		InputStream in = transport.write(url.toString(), xml_data);
		if (in == null) {
			return (null);
		}

		return ((Response)decode(in));
	}


	/**
	 * Sets up and starts a thread to send the request in the background.
	 */
	public void backgroundSend (String url_spec, Request request) {
		ResponseThread rt = new ResponseThread(this, url_spec, request);
		rt.start();
	}


	/**
	 * Decodes an XMLF document into an XmlfObject.
	 *
	 * @param  xml_data  XMLF document data.
	 *
	 * @returns  A Response.
	 */
	public XmlfObject decode (String xml_data) {

		XmlParser parser = new XmlParser(_decoders);
		return (parser.parse(xml_data));
	}


	/**
	 * Decodes an XMLF document stream into an XmlfObject.
	 *
	 * @param  xml_data  XMLF document data.
	 *
	 * @returns  A Response.
	 */
	public XmlfObject decode (InputStream xml_stream) {

		XmlParser parser = new XmlParser(_decoders);
		return (parser.parse(xml_stream));
	}


	/**
	 * Removes the specified document type from the bindings.
	 *
	 * @param  doc_type  name of the document decoder.
	 *
	 * @returns  true on success, false on failure.
	 */
	public boolean unbindDecoder (String doc_type) {
		try {
			_decoders.unbind(doc_type);
			return (true);
		} catch (Exception e) {
			return (false);
		}
	}


	/**
	 * Removes the specified protocol name from the bindings. 
	 *
	 * @param  protocol  name of the transport to unbind.
	 *
	 * @returns  true on success, false on failure.
	 */
	public boolean unbindTransport (String protocol) {
		
		try {
			_transports.unbind(protocol);
			return (true);
		} catch (Exception e) {
			return (false);
		}
	}

	
	/**
	 * Reads a line at a time from the specified InputStream and
	 * returns a String containing the read lines.
	 *
	 * @param  in  the InputStream to read.
	 *
	 * @returns  A String containing the read data.
	 */
	protected String read (InputStream in) {

		if (in == null) {
			return (null);
		}
		
		try {
			BufferedReader reader = new BufferedReader(
				new InputStreamReader(in));
			StringBuffer buf = new StringBuffer();
			String line;

			while ((line = reader.readLine()) != null) {
				buf.append(line);
				buf.append("\n");
			}

			return (buf.toString());

		} catch (IOException ioe) {
			_err_msg.append(ioe.getMessage());
			return (null);
		}
	}


       /**
        * Add a response event listener.
        *
        * @param  l  The ResponseListener to add.
        */
       public synchronized void addResponseListener (ResponseListener l) {

               if (!_rsp_listeners.contains(l)) {
                       _rsp_listeners.addElement(l);
               }
       }

       /**
        * Remove a response event listener.
        *
        * @param  l  The ResponseListener to remove.
        */
       public synchronized void removeResponseListener (ResponseListener l) {

               if (!_rsp_listeners.contains(l)) {
                       _rsp_listeners.removeElement(l);
               }
       }

        /**
         * Notify listening objects of response events.
         *
         */
        public void fireResponseEvent (Response rsp) {

                /*
                 * Make a copy of the listener object vector so that it cannot
                 * be changed while we are firing events
                 */
                Vector v;
                synchronized(this) {
                        if (_rsp_listeners.size() < 1) {
                                return;
                        }

                        v = (Vector)_rsp_listeners.clone();
                }

                /*
                 * Create the event object
                 */
                ResponseEvent evt = new ResponseEvent(this, rsp);


                /*
                 * Fire the event to all listeners
                 */
                int count = v.size();
                for (int i = 0; i < count; i++) {
                        ResponseListener l = (ResponseListener)v.elementAt(i);
                        l.response(evt);
                }
        }


	/**
	 * Returns an instance of the Transportmanager class. This returned
	 * instance is setup with the default transport and decoder bindings.
	 */
	public static TransportManager getDefault () {
        
		/*
        	 * Open the transport manager
        	 */
		TransportManager mgr = new TransportManager();

	        /*
	         * Bind the vxmld (defdault) transport
        	 */
		VxmldTransport transport = new VxmldTransport();
		mgr.bindTransport("vxmld", transport);
		mgr.setDefaultTransport("vxmld");


        	/*
               	 * Bind the XMLF document decoders
		 */
		DocumentDecoder decoder;

		decoder = new SubmitRequestDecoder();
		decoder.bind(mgr);

		decoder = new StatusRequestDecoder();
		decoder.bind(mgr);

		decoder = new CancelRequestDecoder();
		decoder.bind(mgr);
	
		decoder = new SubmitResponseDecoder();
		decoder.bind(mgr);

		decoder = new StatusResponseDecoder();
		decoder.bind(mgr);

		decoder = new CancelResponseDecoder();
		decoder.bind(mgr);

		return (mgr);
	}
}
