/*------------------------------------------------------------------------
 * Program to download fax files from the server
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"vxld:u:U:H:Z:?"

/*------------------------------------------------------------------------
 * global variables
 */
static V_CTX *		vctx;
static V_SYS *		vptr;
static int		verbose		= FALSE;
static int		do_delete	= FALSE;
static int		do_list		= FALSE;
static const char *	dir		= 0;
static const char *	user		= 0;

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
	fprintf(fp, "usage: %s [options] [file ...]\n", pgm);
	fprintf(fp, "options:\n");
	fprintf(fp, "  -v            Verbose mode\n");
	fprintf(fp, "  -x            Delete file when downloaded\n");
	fprintf(fp, "  -l            List files\n");
	fprintf(fp, "  -d dir        Directory to put files in\n");
	fprintf(fp, "  -u user       access files for \"user\"\n");
	fprintf(fp, "  -U username   Username to login as\n");
	fprintf(fp, "  -H hostname   Hostname to connect to\n");
	fprintf(fp, "file            Filename to process\n");
}

/*------------------------------------------------------------------------
 * extract data for a name from a tag list
 */
static const char *
tag_data (const V_TAG *tags, const char *name)
{
	const char *p = VSI_Tag_Get_Data_By_Name(tags, name, 0, 0);

	if (p == 0 || *p == 0)
		return ("<none>");
	return (p);
}

/*------------------------------------------------------------------------
 * get filename for a file
 */
static const char *
get_filename (const V_TAG *tags)
{
	const V_TAG * tag_filename;

	tag_filename = VSI_Tag_Find_By_Name(tags, V_TAG_HDR_FILENAME, 0);

	if (tag_filename != 0)
	{
		return VSI_Tag_Data(tag_filename);
	}

	return ("");
}

/*------------------------------------------------------------------------
 * display a file header
 */
static int
list_file (const V_TAG *tags, char *msgbuf)
{
	const char *	filename;

	filename = get_filename(tags);

	if (verbose)
	{
		const char *	tsi	= tag_data(tags, V_TAG_HDR_TSI);
		const char *	rtt	= tag_data(tags, V_TAG_HDR_RTETIME);
		const char *	npg	= tag_data(tags, V_TAG_HDR_NUMPGS);

		printf("%-16s %-16s %s %4s\n",
			filename, tsi, rtt, npg);
	}
	else
	{
		printf("%s\n", filename);
	}

	return (0);
}

/*------------------------------------------------------------------------
 * delete a file
 */
static int
delete_file (const V_TAG *tags, char *msgbuf)
{
	const char *	filename;
	int		rc;

	filename = get_filename(tags);

	rc = VSI_Server_File_Delete(vptr, user, filename, msgbuf);

	return (rc);
}

/*------------------------------------------------------------------------
 * retrieve a file
 */
static int
retrieve_file (const V_TAG *tags, char *msgbuf)
{
	const char *	filename;
	char		pathname[256];
	int		rc;

	filename = get_filename(tags);
	if (dir == 0)
		strcpy(pathname, filename);
	else
		sprintf(pathname, "%s/%s", dir, filename);

	rc = VSI_Server_File_Retrieve(vptr, user, filename, pathname, 0,
		msgbuf);

	return (rc);
}

/*------------------------------------------------------------------------
 * process a file
 */
static int
process_file (const V_TAG *tags, char *msgbuf)
{
	int rc;

	if (do_list)
		rc = list_file(tags, msgbuf);
	else
		rc = retrieve_file(tags, msgbuf);

	if (do_delete)
		rc = delete_file(tags, msgbuf);

	return (rc);
}

/*------------------------------------------------------------------------
 * find a file in a list
 */
static const V_TAG *
find_file (V_LIST *files, const char *filename)
{
	V_LIST * l;

	for (l = files; l; l = VSI_List_Next(l))
	{
		const V_TAG *	tag_list;
		const char *	this_filename;

		tag_list  = VSI_List_Tags(l);
		this_filename = get_filename(tag_list);
		if (strcmp(filename, this_filename) == 0)
			return (tag_list);
	}

	return (0);
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	V_LIST *	files;
	char		msgbuf[V_MSG_LEN];
	const char *	pgmname		= argv[0];
	int		opt_index;
	int		c;
	int		rc;

	/*----------------------------------------------------------------
	 * get context
	 */
	vctx = VSI_Context_Load(argc, argv, 0, msgbuf);
	if (vctx == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		switch (c)
		{
		case 'v':
			/*------------------------------------------------
			 * verbose mode
			 */
			verbose = TRUE;
			break;

		case 'x':
			/*------------------------------------------------
			 * delete mode
			 */
			do_delete = TRUE;
			break;

		case 'l':
			/*------------------------------------------------
			 * list mode
			 */
			do_list = TRUE;
			break;

		case 'd':
			/*------------------------------------------------
			 * specify directory
			 */
			dir = VSI_Opt_Argstr();
			break;

		case 'u':
			/*------------------------------------------------
			 * specify user
			 */
			user = VSI_Opt_Argstr();
			break;

		case 'U':
		case 'H':
		case 'Z':
			/*------------------------------------------------
			 * already processed
			 */
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * login to the server
	 */
	vptr = VSI_Server_Login_By_Context(vctx, msgbuf);
	if (vptr == 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * get list of fax files
	 */
	files = VSI_Server_File_List(vptr, user, msgbuf);
	if (*msgbuf != 0)
	{
		VSI_Server_Logout(vptr, 0);
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * now process list/args
	 */
	opt_index = VSI_Opt_Index();
	if (opt_index < argc)
	{
		for (; opt_index < argc; opt_index++)
		{
			const char * filename = argv[opt_index];
			const V_TAG *	file = find_file(files, filename);

			if (file == 0)
			{
				fprintf(stderr, "%s: File %s not found\n",
					pgmname, filename);
			}
			else
			{
				rc = process_file(file, msgbuf);
				if (rc)
				{
					fprintf(stderr, "%s: %s\n",
						pgmname, msgbuf);
				}
			}
		}
	}
	else
	{
		V_LIST * l;

		for (l = files; l; l = VSI_List_Next(l))
		{
			const V_TAG * file = VSI_List_Tags(l);

			{
				rc = process_file(file, msgbuf);
				if (rc)
				{
					fprintf(stderr, "%s: %s\n",
						pgmname, msgbuf);
				}
			}
		}
	}

	/*----------------------------------------------------------------
	 * free the file list
	 */
	VSI_List_Free(files, 0);

	/*----------------------------------------------------------------
	 * logout from the server
	 */
	VSI_Server_Logout(vptr, 0);
	VSI_Context_Free(vctx, 0);

	return (EXIT_SUCCESS);
}
