/*------------------------------------------------------------------------
 * Program to copy a TIFF file
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "vsi.h"

/*------------------------------------------------------------------------
 * option string
 */
#define OPTS	"E:siv?"

/*------------------------------------------------------------------------
 * usage routine
 */
static void
usage (const char *pgm, FILE *fp)
{
fprintf(fp, "usage: %s [options] outfile inpfile ...\n", pgm);
fprintf(fp, "options:\n");
fprintf(fp, "  -E res      output resolution\n");
fprintf(fp, "  -s          disable VSI propritary tags\n");
fprintf(fp, "  -v          verbose mode\n");
fprintf(fp, "  -i          display info on output file only\n");
fprintf(fp, "  outfile     output file\n");
fprintf(fp, "  inpfile     input  file & optional page-range\n");
fprintf(fp, "              A page-range may be specified as follows:\n");
fprintf(fp, "                foo.tif        all pages\n");
fprintf(fp, "                foo.tif:1,6    pages 1 & 6\n");
fprintf(fp, "                foo.tif:1-6    pages 1 through 6\n");
fprintf(fp, "                foo.tif:1,6-   pages 1 & 6 through the end\n");
}

/*------------------------------------------------------------------------
 * main routine
 */
int
main (int argc, char **argv)
{
	char		msgbuf[V_MSG_LEN];
	const char *	pgmname		= argv[0];
	const char *	resolution	= 0;
	const char *	outpath;
	V_TAG *		tag_list	= 0;
	int		vsitags		= TRUE;
	int		verbose		= FALSE;
	int		info		= FALSE;
	int		opt_index;
	int		c;
	int		rc;

	/*----------------------------------------------------------------
	 * process any options
	 */
	while ((c = VSI_Opt_Get(argc, argv, OPTS, msgbuf)) != EOF)
	{
		switch (c)
		{
		case 'E':
			resolution = VSI_Opt_Argstr();
			break;

		case 's':
			vsitags = ! vsitags;
			break;

		case 'v':
			verbose = ! verbose;
			break;

		case 'i':
			info = ! info;
			break;

		case '?':
			/*------------------------------------------------
			 * output usage message
			 */
			usage(pgmname, stdout);
			return (EXIT_SUCCESS);

		default:
			/*------------------------------------------------
			 * output error message
			 */
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			usage(pgmname, stderr);
			return (EXIT_FAILURE);
		}
	}

	/*----------------------------------------------------------------
	 * check for output file
	 */
	opt_index = VSI_Opt_Index();
	if ((argc - opt_index) == 0)
	{
		fprintf(stderr, "%s: No output file given\n", pgmname);
		usage(pgmname, stderr);
		return (EXIT_FAILURE);
	}

	outpath = argv[opt_index++];

	/*----------------------------------------------------------------
	 * check if info wanted
	 */
	if (info)
	{
		int num;
		int i;

		tag_list = VSI_TIFF_Header(outpath, msgbuf);
		if (*msgbuf != 0)
		{
			fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
			return (EXIT_FAILURE);
		}

		num = VSI_Tag_Count(tag_list, 0);
		for (i=0; i<num; i++)
		{
			char name_buf[V_TAG_NAMELEN];
			char data_buf[V_TAG_DATALEN];

			VSI_Tag_Get_Name_By_Index(tag_list, i, name_buf, 0);
			VSI_Tag_Get_Data_By_Index(tag_list, i, data_buf, 0);

			printf("%s\t%s\n", name_buf, data_buf);
		}

		VSI_Tag_Free(tag_list, 0);

		return (EXIT_SUCCESS);
	}

	/*----------------------------------------------------------------
	 * check for input files
	 */
	if ((argc - opt_index) == 0)
	{
		fprintf(stderr, "%s: No input files given\n", pgmname);
		usage(pgmname, stderr);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * process all input files
	 */
	for (; opt_index < argc; opt_index++)
	{
		char inpfile[V_TAG_DATALEN];
		char *p;

		strcpy(inpfile, argv[opt_index]);

		p = strchr(inpfile+2, ':');
		if (p != 0)
			*p++ = 0;

		tag_list = VSI_Tag_Add(tag_list, V_TAG_ENV_FILELCL, inpfile, 0);
		if (p != 0)
		{
			tag_list = VSI_Tag_Add(tag_list, V_TAG_ENV_PGRANGE, p,
				0);
		}
	}

	/*----------------------------------------------------------------
	 * now do the copy
	 */
	rc = VSI_TIFF_Merge(outpath, tag_list, resolution, vsitags, msgbuf);
	VSI_Tag_Free(tag_list, 0);

	/*----------------------------------------------------------------
	 * check if any errors
	 */
	if (rc < 0)
	{
		fprintf(stderr, "%s: %s\n", pgmname, msgbuf);
		return (EXIT_FAILURE);
	}

	/*----------------------------------------------------------------
	 * output info if verbose
	 */
	if (verbose)
	{
		printf("%d pages copied\n", rc);
	}

	return (EXIT_SUCCESS);
}
