unit allfuncs;

// Copyright  2001 by Ziff Davis Media, Inc.
// Written by Neil J. Rubenking

// ********************************************************
// A growing collection of functions that are generally useful
// in this and other programs.
// ********************************************************

interface
uses Windows, SysUtils, Forms, Classes, Graphics, stdCtrls;

type
  gdExeType = (gdNO, gdMZ, gdNE, gdPE);
  function GetDescription(const fname : String;
    var Typ : gdExeType) : String;
  function GetDescriptionEZ(const fname : String) : String;
  function GetVersionString(const fname, ValName : String) :
    String;
  function FinalSlash(const S : String) : String;
  function NoFinalSlash(const S : String) : String;
  function GetSpecialPath(H : HWnd; nFolder : Integer) : String;
  procedure WrapLabel(const S : String; MaxW, MaxH, Play : Integer;
    CV : TCanvas; L : TLabel);
  function ReadALink(const S : String; pPath, pArgs, pWork, pIcon,
    pDesc : PChar; i : pInteger) : Boolean;
  function MakeALink(const S : String; pPath, pArgs, pWork, pIcon,
    pDesc : PChar; i : pInteger) : Boolean;
  function LocalDrives(const addl, delim : String;
    ignoreSubst : Boolean) : String;
  function StringForBytes(I : Int64) : String;
  function TempFileName(const pref : String) : String;
  function AniCurFromResource(ResName : PChar) : HCursor;
  function ShellVersionIs4_70 : boolean;
  function PickIconDlg(hOwner : LongInt; szFilename : PChar;
    cchFilename : LongInt; VAR IconIndex : LongInt) : LongBool;
  function FileSizeFromName(const S : String) : Int64;
  function CharEntity(C : Char) : String;
  function RecycleFile(Handle : THandle; const S : String) : Boolean;
  function LongPathName(const S : String) : String;
  function EE(const S : String) : String;

VAR
  IsWinNT : Boolean;

implementation
USES shlobj, shellapi, comobj, activex, IniFiles, VerResU;

TYPE
  {details of DOS header structure aren't important to us,
   so we just read the first 64 (40h) bytes}
  TDOSHeader = ARRAY[0..$3F] OF Byte;

  TNEHeader = RECORD
    {New Executable file header}
    Signature                      : Word;
    LinkerVersion,
    LinkerRevision                 : Byte;
    EntryTableRelOffset,
    EntryTableLength               : Word;
    Reserved                       : LongInt;
    Flags,
    AutomaticDSegNumber,
    LocalHeapSize,
    StackSize                      : Word;
    CSIP,
    SSSP                           : Pointer;
    SegmentTableNumEntries,
    ModuleReferenceTableNumEntries,
    NonresidentNameTableSize,
    SegmentTableRelOffset,
    ResourceTableRelOffset,
    ResidentNameTableRelOffset,
    ModuleReferenceTableRelOffset,
    ImportedNameTableRelOffset     : Word;
    NonresidentNameTableOffset     : LongInt;
    NumberOfMovableEntryPoints,
    ShiftCount,
    NumberOfResourceSegments       : Word;
    TargetOS,
    AdditionalInfo                 : Byte;
    FastLoadAreaOffset,
    FastLoadAreaSectors,
    Reserved2,
    ExpectedWindowsVersion         : Word;
  END;

function GetVersionString(const fName, ValName : String) :
  String;
begin
  WITH TVerInfoObj.Create(fName) DO
  try
    IF Status = vio_OK THEN
      Result := Values[ValName]
    ELSE Result := '';
  finally
    Free;
  end;
end;

function GetNEDescription(const fname : String) : String;
VAR
  F : File;
  W, N          : Word;
  B             : Byte;
  DH            : TDOSHeader;
  NEOffset      : Word; {Offset of NE header}
  NE            : TNEHeader;
  Desc          : ARRAY[0..MAX_PATH] OF Char;

  PROCEDURE OpenForReadOnly(VAR F : File; FName : String);
  {The calling routine must handle any exception caused
   by the attempt to open the file}
  VAR OldFileMode : Byte;
  BEGIN
    AssignFile(F, FName);
    OldFileMode := FileMode;
    FileMode := 0; {read-only}
    Reset(F,1);
    FileMode := OldFileMode;
  END;

begin
  Result := '';
  try
    OpenForReadOnly(F, fname);
    try
      BlockRead(F, DH, SizeOf(DH));
      Move(DH[0], N, 2);
      Move(DH[$18], W, 2);
      IF (N <> $5A4D{MZ}) OR ((W < $40) AND (W > 0)) THEN
        // No DOS executable header. Shouldn't happen!
        Exit;
      Move(DH[$3C], NEOffset, 2);
      Seek(F, NEOffset);
      BlockRead(F, NE, SizeOf(NE));
      IF NE.Signature = $454E{NE} THEN
        begin
          // 16-bit "New Executable"
          Seek(F, NE.NonresidentNameTableOffset);
          BlockRead(F, B, 1);
          FillChar(Desc, SizeOf(Desc),0);
          BlockRead(F, Desc, B);
          IF StrLen(Desc) <> 0 THEN
            Result := StrPas(Desc);
        end;
    finally
      CloseFile(F);
    end;
  except ON EInOutError DO
    // Let it pass - result is blank
  end;
end;

function GetDescription(const fname : String;
  var Typ : gdExeType) : String;
// Return description from header for NE (16-bit)
// Return versioninfo description for PE (32-bit)
// Return null string otherwise
VAR
  F : File;
  W, N          : Word;
  B             : Byte;
  DH            : TDOSHeader;
  NEOffset      : Word; {Offset of NE header}
  NE            : TNEHeader;
  Desc          : ARRAY[0..MAX_PATH] OF Char;

  PROCEDURE OpenForReadOnly(VAR F : File; FName : String);
  {The calling routine must handle any exception caused
   by the attempt to open the file}
  VAR OldFileMode : Byte;
  BEGIN
    AssignFile(F, FName);
    OldFileMode := FileMode;
    FileMode := 0; {read-only}
    Reset(F,1);
    FileMode := OldFileMode;
  END;

begin
  typ := gdNO;
  Result := '';
  try
    OpenForReadOnly(F, fname);
    try
      BlockRead(F, DH, SizeOf(DH));
      Move(DH[0], N, 2);
      Move(DH[$18], W, 2);
      IF N = $5A4D{MZ} THEN typ := gdMZ;
      IF (N <> $5A4D{MZ}) OR ((W < $40) AND (W > 0)) THEN
        // No DOS executable header. Shouldn't happen!
        Exit;
      Move(DH[$3C], NEOffset, 2);
      Seek(F, NEOffset);
      BlockRead(F, NE, SizeOf(NE));
      IF NE.Signature = $454E{NE} THEN
        begin
          // 16-bit "New Executable"
          typ := gdNE;
          Seek(F, NE.NonresidentNameTableOffset);
          BlockRead(F, B, 1);
          FillChar(Desc, SizeOf(Desc),0);
          BlockRead(F, Desc, B);
          IF StrLen(Desc) <> 0 THEN
            Result := StrPas(Desc);
        end
      ELSE IF NE.Signature = $4550{PE} THEN
        begin
          // 32-bit "Portable Executable"
          typ := gdPE;
          Result := GetVersionString(fName, 'FileDescription');
          IF Result = '' THEN
            Result := GetVersionString(fName, 'ProductName');
        end;
    finally
      CloseFile(F);
    end;
  except ON EInOutError DO
    // Let it pass - result is blank
  end;
end;

function GetDescriptionEZ(const fname : String) : String;
begin
  WITH TVerInfoObj.Create(fname) DO
  try
    Result := Values['FileDescription'];
    IF Result = '' THEN Result := Values['ProductName'];
    IF Result = '' THEN Result := GetNEDescription(fname);
  finally
    Free;
  end;
end;

function FinalSlash(const S : String) : String;
begin
  IF (S <> '') AND (S[Length(S)] <> '\') THEN
    Result := S + '\'
  ELSE Result := S;
end;

function NoFinalSlash(const S : String) : String;
begin
  IF (S <> '') AND (S[Length(S)] = '\') THEN
    Result := Copy(S, 1, Length(S)-1)
  ELSE Result := S;
end;

function GetSpecialPath(H : HWnd; nFolder : Integer) : String;
VAR
  thePIDL : PItemIDList;
  theBUFF : ARRAY[0..MAX_PATH] OF Char;
begin
  IF SHGetSpecialFolderLocation(H, nfolder, thePIDL) = NOERROR THEN
    begin
      IF SHGetPathFromIDList(thePIDL, theBUFF) THEN
        Result := StrPas(theBuff)
      ELSE Result := '';
    end
  ELSE Result := '';
end;

procedure WrapLabel(const S : String; MaxW, MaxH, Play : Integer;
  CV : TCanvas; L : TLabel);
// Sets the label's caption to the specified string, with
// sensible word-wrap even if the string contains no spaces.
// Adjust the label's height as necessary
VAR
  WorkS, TempS : String;
  Pos, N       : Integer;
  StartPos     : Integer;
  R            : TRect;
begin
  TempS := '';
  WorkS := S;
  // Make a guess as to where the string should break
  IF Length(WorkS) = 0 THEN StartPos := 0
  ELSE
    StartPos := (MaxW * Length(WorkS)) DIV CV.TextWidth(WorkS);
  // Build a new string, inserting spaces as needed to
  // force line breaks
  WHILE CV.TextWidth(WorkS) > MaxW DO
    begin
      Pos := StartPos;
      WHILE CV.TextWidth(Copy(WorkS, 1, Pos)) >= MaxW DO Dec(Pos);
      WHILE CV.TextWidth(Copy(WorkS, 1, Pos)) <  MaxW DO Inc(Pos);
      Dec(Pos);
      N := 0;
      WHILE (N < Pos-1) AND (N < Play) DO
        IF WorkS[Pos-N] IN [' ','.',',',';',':','/','\','-'] THEN
          Break
        ELSE Inc(N);
      IF WorkS[Pos-N] IN [' ','.',',',';',':','/','\','-'] THEN Pos := Pos-N;
      TempS := TempS + Copy(WorkS, 1, Pos);
      IF WorkS[Pos] <> ' ' THEN TempS := TempS + ' ';
      Delete(WorkS, 1, Pos);
    end;
  TempS := TempS + WorkS;
  // Use DrawText to calculate the necessary size
  FillChar(R, SizeOf(R), 0);
  R.Right := MaxW;
  DrawText(CV.Handle, PChar(TempS), Length(TempS), R,
    DT_CALCRECT OR DT_WORDBREAK OR DT_NOPREFIX);
  L.WordWrap := True;
  L.Width := R.Right;  
  IF R.Bottom < MaxH THEN L.Height := R.Bottom
  ELSE L.Height := MaxH;
  L.Caption := TempS;
end;

function ReadALink(const S : String; pPath, pArgs, pWork, pIcon,
  pDesc : PChar; i : pInteger) : Boolean;
// Read the shortcut named in the S argument, and fill any
// non-nil arguments with data from the shortcut. Each non-nil
// PChar must be big enough to hold MAX_PATH characters
VAR             
  vUNK   : IUnknown;
  vISL   : IShellLink;
  vIPF   : IPersistFile;
  fNameW : ARRAY[0..MAX_PATH] OF WideChar;
  TW     : TWin32FindData;
begin
  Result := False;
  try
    StringToWideChar(S, fNameW, MAX_PATH);
    vUNK := CreateComObject(CLSID_ShellLink);
    vISL := vUNK AS IShellLink;
    vIPF := vUNK AS IPersistFile;
    IF vIPF.Load(@fNameW, STGM_READ) <> S_OK THEN Exit;
    Result := True;
    IF pPath <> nil THEN
      IF vISL.GetPath(pPath, MAX_PATH, TW, 0) <> S_OK THEN
        pPath[0] := #0;
    IF pArgs <> nil THEN
      IF vISL.GetArguments(pArgs, MAX_PATH) <> S_OK THEN
        pArgs[0] := #0;
    IF pWork <> nil THEN
      IF vISL.GetWorkingDirectory(pWork, MAX_PATH) <> S_OK THEN
        pWork[0] := #0;
    IF (pIcon <> nil) AND (i <> nil) THEN
      IF vISL.GetIconLocation(pIcon, MAX_PATH, i^) <> S_OK THEN
        pIcon[0] := #0;
    IF pDesc <> nil THEN
      IF vISL.GetDescription(PDesc, MAX_PATH) <> S_OK THEN
        pDesc[0] := #0;
  except
    ON Exception DO;
  end;
end;

function MakeALink(const S : String; pPath, pArgs, pWork, pIcon,
  pDesc : PChar; i : pInteger) : Boolean;
// Create the shortcut named in the S argument, and load its
// data from the non-nil arguments. Return True if successful.
VAR
  vUNK : IUnknown;
  vISL : IShellLink;
  vIPF : IPersistFile;
  fNameW : ARRAY[0..MAX_PATH] OF WideChar;
begin
  Result := False;
  try
    StringToWideChar(S, fNameW, MAX_PATH);
    vUNK := CreateComObject(CLSID_ShellLink);
    vISL := vUNK AS IShellLink;
    vIPF := vUNK AS IPersistFile;
    IF pPath <> nil THEN
      IF vISL.SetPath(pPath) <> S_OK THEN Exit;
    IF pArgs <> nil THEN
      IF vISL.SetArguments(pArgs) <> S_OK THEN Exit;
    IF (pIcon <> nil) AND (i <> nil) THEN
      IF vISL.SetIconLocation(pIcon, i^) <> S_OK THEN Exit;
    IF pWork <> nil THEN
      IF vISL.SetWorkingDirectory(pWork) <> S_OK THEN Exit;
    IF pDesc <> nil THEN
      IF vISL.SetDescription(pDesc) <> S_OK THEN Exit;
    IF  vIPF.Save(@fNameW, False) <> S_OK THEN Exit;
    Result := True;
  except
    ON Exception DO;
  end;
end;

function GetVolumeSerial(root : PChar) : DWORD;
VAR
  VolumeSerialNumber      : DWORD;
  MaximumComponentLength  : DWORD;
  FileSystemFlags         : DWORD;
begin
  VolumeSerialNumber     := 0;
  MaximumComponentLength := 0;
  FileSystemFlags        := 0;
  IF GetVolumeInformation(Root, nil, 0, @VolumeSerialNumber,
    MaximumComponentLength, FileSystemFlags, nil, 0) THEN
    Result := VolumeSerialNumber
  ELSE Result := DWORD(-1);
end;

function LocalDrives(const addl, delim : String;
  ignoreSubst : Boolean) : String;
// Returns a string listing the local drive letters.
// Each letter will have the string addl appended, and
// all but the last will also have the string delim.
// E.g. call LocalDrives(':\', ';') might yield the string
// 'C:\;D:\;E:\;F:\'
VAR
  DriveMask : DWORD;
  drvName   : ARRAY[0..3] OF Char;
  drvLet    : Char;
  N         : Integer;
  Serials   : TStringList;
  S         : String;

  function TestForQDD : Boolean;
  // QueryDosDevice can be used in Windows NT and Windows 98,
  // but not Windows 95. Under Windows 95, it will return 0,
  // and GetLastError will return a non-zero result.
  VAR Buffer : ARRAY[0..MAX_PATH] OF Char;
  begin
    SetLastError(0);
    Result := (QueryDosDevice('c:', buffer, MAX_PATH) > 0) AND
      (GetLastError = 0);
  end;

  function IsSubst(C : Char) : Boolean;
  // Do not call this function unless TestQDD has returned
  // true. Under WinNT, the name returned by QDD for a
  // SUBST drive begins with "\??\". Under Windows 98,
  // the returned name for a non-subst drive will be e.g.
  // "c:"; for a subst drive it will be the full path,
  // e.g. "c:\" or "c:\Windows".
  VAR
    BuffS : ARRAY[0..3] OF Char;
    BuffT : ARRAY[0..MAX_PATH] OF Char;
  begin
    BuffS[0] := C;
    BuffS[1] := ':';
    BuffS[2] := #0;
    Result := (NOT IgnoreSubst) AND
      (QueryDosDevice(BuffS, BuffT, MAX_PATH) > 0);
    IF Result THEN
      begin
        IF IsWinNT THEN
          Result := StrLIComp('\??\', BuffT, 4) = 0
        ELSE
          Result := StrIComp(BuffS, BuffT) <> 0;
      end;

  end;
begin
  DriveMask   := GetLogicalDrives;
  Result      := '';
  drvLet      := 'A';
  StrCopy(drvName, 'A:\');
  IF TestForQDD OR IgnoreSubst THEN
    begin
      FOR N := 0 TO 31 DO
        begin
          IF Odd(DriveMask) AND
            (GetDriveType(drvName) = DRIVE_FIXED) AND
            (NOT IsSubst(DrvLet)) THEN
            Result := Result + drvLet + addl + delim;
          DriveMask := DriveMask SHR 1;
          Inc(drvName[0]);
          Inc(drvLet);
        end;
      IF Result <> '' THEN
        SetLength(Result, Length(Result)-Length(delim));
    end
  ELSE
    begin
      Serials     := TStringList.Create;
      Serials.Sorted := True;
      try
        FOR N := 0 TO 31 DO
          begin
            IF Odd(DriveMask) AND
              (GetDriveType(drvName) = DRIVE_FIXED) THEN
              begin
                S := IntToHex(GetVolumeSerial(drvName), 8);
                IF Serials.IndexOf(S) = -1 THEN
                  begin
                    Result := Result + drvLet + addl + delim;
                    Serials.Add(S);
                  end;
              end;
            DriveMask := DriveMask SHR 1;
            Inc(drvName[0]);
            Inc(drvLet);
          end;
        IF Result <> '' THEN
          SetLength(Result, Length(Result)-Length(delim));
      finally
        Serials.Free;
      end;
    end;
end;

function StringForBytes(I : Int64) : String;
begin
  IF I < 9999 THEN
    Result := FormatFloat('0,', I) + ' bytes'
  ELSE IF I DIV 1024 < 9999 THEN
    Result := FormatFloat('0,.00', I / 1024) + ' KB'
  ELSE IF I DIV (1024*1024) < 9999 THEN
    Result := FormatFloat('0,.00', I / (1024*1024)) + ' MB'
  ELSE Result := FormatFloat('0,.00', I / (1024*1024*1024)) + ' GB';
end;

function TempFileName(const pref : String) : String;
VAR
  tempP,
  tempF : ARRAY[0..MAX_PATH] OF Char;
begin
  GetTempPath(MAX_PATH, tempP);
  GetTempFileName(tempP, pchar(pref), 0, tempF);
  Result := StrPas(TempF);
end;

function AniCurFromResource(ResName : PChar) : HCursor;
VAR
  Hr    : THandle;
  Hg    : THandle;
  Siz   : Integer;
  P     : PByteArray;
  F     : File;
  TempF : String;
begin
  Result := 0;
  Hr  := FindResource(0, ResName, RT_ANICURSOR);
  IF Hr = 0 THEN Exit;
  Hg  := LoadResource(0, Hr);
  IF Hg = 0 THEN Exit;
  Siz := SizeOfResource(0, Hr);
  P   := LockResource(Hg);
  TempF := TempFileName('ani');
  AssignFile(F, TempF);
  Rewrite(F, 1);
  BlockWrite(F, P^, Siz);
  CloseFile(F);
  Result := LoadCursorFromFile(PChar(tempF));
  DeleteFile(tempF);
end;

function ShellVersionIs4_70 : boolean;
type
  PDllVersionInfo = ^TDllVersionInfo;
  TDllVersionInfo = record
    cbSize          : DWORD;
    dwMajorVersion  : DWORD;
    dwMinorVersion  : DWORD;
    dwBuildNumber   : DWORD;
    dwPlatformID    : DWORD;
  end;
  DllGetVersionType = function (pdvi : PDllVersionInfo) :
    HRESULT; stdcall;
VAR
  LibH : hModule;
  DllGetVersion  : DllGetVersionType;
  DVI  : TDllVersionInfo;
begin
  Result := False;
  LibH := LoadLibrary('Shell32.dll');
  IF LibH <= 32 THEN Exit;
  try
    @DllGetVersion := GetProcAddress(LibH, 'DllGetVersion');
    IF @DllGetVersion = nil THEN Exit;
    FillChar(DVI, SizeOf(DVI), 0);
    DVI.cbSize := SizeOf(DVI);
    IF DllGetVersion(@DVI) = NOERROR THEN
      begin
        IF DVI.dwMajorVersion > 4 THEN Result := True
        ELSE IF (DVI.dwMajorVersion = 4) AND
          (DVI.dwMinorVersion >= 70) THEN Result := True;
      end;
  finally
    FreeLibrary(LibH);
  end;
end;

procedure SysFreeString(const S: WideString); stdcall;
  external 'oleaut32.dll' name 'SysFreeString';

Function PickIconDlgA(hOwner : LongInt; szFilename : PChar;
  cchFilename : LongInt; VAR IconIndex : LongInt) : LongBool;
  stdcall; external 'shell32.dll' index 62;

Function PickIconDlgW(hOwner : LongInt; szFilename : PWideChar;
  cchFilename : LongInt; VAR IconIndex : LongInt) : LongBool;
  stdcall; external 'shell32.dll' index 62;

Function PickIconDlg(hOwner : LongInt; szFilename : PChar;
  cchFilename : LongInt; VAR IconIndex : LongInt) : LongBool;
VAR szWFilename : PWideChar;
begin
  IF IsWinNT THEN
    begin
      szWFilename := StringToOleStr(StrPas(szFilename));
      Result := PickIconDlgW(hOwner, szWFilename, cchFilename,
        IconIndex);
      StrPCopy(szFilename, WideCharToString(szWFilename));
      SysfreeString(szWFilename);
    end
  ELSE Result := PickIconDlgA(hOwner, szFilename, cchFilename,
    IconIndex);
end;

function FileSizeFromName(const S : String) : Int64;
VAR
  FD : TWin32FindData;
  FH : THandle;
begin
  FH := FindFirstFile(PChar(S), FD);
  IF FH = INVALID_HANDLE_VALUE THEN Result := 0
  ELSE
    try
      Result := FD.nFileSizeHigh;
      Result := Result SHL 32;
      Result := Result + FD.nFileSizeLow;
    finally
      CloseHandle(FH);
    end;
end;

function CharEntity(C : Char) : String;
begin
  CASE C OF
    #034 : Result := 'quot';
    #038 : Result := 'amp';
    #060 : Result := 'lt';
    #062 : Result := 'gt';
    #160 : Result := 'nbsp';
    #161 : Result := 'iexcl';
    #162 : Result := 'cent';
    #163 : Result := 'pound';
    #164 : Result := 'curren';
    #165 : Result := 'yen';
    #166 : Result := 'brvbar';
    #167 : Result := 'sect';
    #168 : Result := 'uml';
    #169 : Result := 'copy';
    #170 : Result := 'ordf';
    #171 : Result := 'laquo';
    #172 : Result := 'not';
    #173 : Result := 'shy';
    #174 : Result := 'reg';
    #175 : Result := 'macr';
    #176 : Result := 'deg';
    #177 : Result := 'plusmn';
    #178 : Result := 'sup2';
    #179 : Result := 'sup3';
    #180 : Result := 'acute';
    #181 : Result := 'micro';
    #182 : Result := 'para';
    #183 : Result := 'middot';
    #184 : Result := 'cedil';
    #185 : Result := 'sup1';
    #186 : Result := 'ordm';
    #187 : Result := 'raquo';
    #188 : Result := 'frac14';
    #189 : Result := 'frac12';
    #190 : Result := 'frac34';
    #191 : Result := 'iquest';
    #192 : Result := 'Agrave';
    #193 : Result := 'Aacute';
    #194 : Result := 'Acirc';
    #195 : Result := 'Atilde';
    #196 : Result := 'Auml';
    #197 : Result := 'Aring';
    #198 : Result := 'AElig';
    #199 : Result := 'Ccedil';
    #200 : Result := 'Egrave';
    #201 : Result := 'Eacute';
    #202 : Result := 'Ecirc';
    #203 : Result := 'Euml';
    #204 : Result := 'Igrave';
    #205 : Result := 'Iacute';
    #206 : Result := 'Icirc';
    #207 : Result := 'Iuml';
    #208 : Result := 'ETH';
    #209 : Result := 'Ntilde';
    #210 : Result := 'Ograve';
    #211 : Result := 'Oacute';
    #212 : Result := 'Ocirc';
    #213 : Result := 'Otilde';
    #214 : Result := 'Ouml';
    #215 : Result := 'times';
    #216 : Result := 'Oslash';
    #217 : Result := 'Ugrave';
    #218 : Result := 'Uacute';
    #219 : Result := 'Ucirc';
    #220 : Result := 'Uuml';
    #221 : Result := 'Yacute';
    #222 : Result := 'THORN';
    #223 : Result := 'szlig';
    #224 : Result := 'agrave';
    #225 : Result := 'aacute';
    #226 : Result := 'acirc';
    #227 : Result := 'atilde';
    #228 : Result := 'auml';
    #229 : Result := 'aring';
    #230 : Result := 'aelig';
    #231 : Result := 'ccedil';
    #232 : Result := 'egrave';
    #233 : Result := 'eacute';
    #234 : Result := 'ecirc';
    #235 : Result := 'euml';
    #236 : Result := 'igrave';
    #237 : Result := 'iacute';
    #238 : Result := 'icirc';
    #239 : Result := 'iuml';
    #240 : Result := 'eth';
    #241 : Result := 'ntilde';
    #242 : Result := 'ograve';
    #243 : Result := 'oacute';
    #244 : Result := 'ocirc';
    #245 : Result := 'otilde';
    #246 : Result := 'ouml';
    #247 : Result := 'divide';
    #248 : Result := 'oslash';
    #249 : Result := 'ugrave';
    #250 : Result := 'uacute';
    #251 : Result := 'ucirc';
    #252 : Result := 'uuml';
    #253 : Result := 'yacute';
    #254 : Result := 'thorn';
    #255 : Result := 'yuml';
    ELSE
      begin
        Result := C;
        Exit;
      end;
  END;
  Result := Format('&%s;', [Result]);
end;

function RecycleFile(Handle : THandle; const S : String) : Boolean;
VAR
  FOS  : TSHFileOpStruct;
  Buff : ARRAY[0..MAX_PATH] OF Char;
begin
  FillChar(Buff, SizeOf(Buff), 0);
  StrPCopy(Buff, S);
  FillChar(FOS, SizeOf(FOS), 0);
  WITH FOS DO
    begin
      Wnd    := Handle;
      wFunc  := FO_DELETE;
      pFrom  := @Buff;
      fFlags := FOF_ALLOWUNDO;
    end;
  Result := ShFileOperation(FOS) = 0;
end;

function LongPathName(const S : String) : String;
VAR
  FH       : THandle;
  FD       : TWin32FindData;
begin
  IF Length(S) <= 3 THEN Result := Copy(S, 1, 2)
  ELSE
    begin
      FH := FindFirstFile(PChar(S), FD);
      IF FH = INVALID_HANDLE_VALUE THEN Result := S
      ELSE
        try
          Result := LongPathName(ExtractFileDir(S)) +
            '\' + FD.cFilename;
        finally
          Windows.FindClose(FH);
        end;
    end;
end;

function EE(const S : String) : String;
// Wrapper for ExpandEnvironmentStrings, so it works
// properly in both WinNTx and Win9x
VAR
  buff1, buff2 : PWideChar;
  Len : Integer;
begin
  IF Pos('%', S) = 0 THEN Result := S
  ELSE IF IsWinNT THEN
    begin
      Len := Length(S);
      GetMem(buff1, 2*Len+2);
      try
        StringToWideChar(S, Buff1, Len);
        Len := ExpandEnvironmentStringsW(buff1, nil, 0);
        GetMem(buff2, 2*Len+2);
        try
          ExpandEnvironmentStringsW(buff1, buff2, Len);
          Result := WideCharToString(buff2);
        finally
          freeMem(buff2);
        end;
      finally
        FreeMem(buff1);
      end;
    end
  ELSE
    begin
      Len := ExpandEnvironmentStringsA(PChar(S), nil, 0);
      SetLength(Result, Len);
      ExpandEnvironmentStringsA(PChar(S), PChar(Result), Len);
    end;
end;



VAR OVI : TOSVersionInfo;
initialization
  FillChar(OVI, SizeOf(OVI), 0);
  OVI.dwOSVersionInfoSize := SizeOf(OVI);
  IF GetVersionEx(OVI) THEN
    IsWinNT := OVI.dwPlatformId = VER_PLATFORM_WIN32_NT
  ELSE IsWinNT := False;
end.
